"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerCreateDatabaseRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _constants = require("../../../../common/constants");
var _serialization = require("./serialization");
var _normalize_database_name = require("./normalize_database_name");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const bodySchema = _configSchema.schema.object({
  databaseType: _configSchema.schema.oneOf([_configSchema.schema.literal('ipinfo'), _configSchema.schema.literal('maxmind')]),
  // maxmind is only needed for "geoip" type
  maxmind: _configSchema.schema.maybe(_configSchema.schema.string({
    maxLength: 1000
  })),
  // only allow database names in sync with ES
  databaseName: _configSchema.schema.oneOf([
  // geoip names https://github.com/elastic/elasticsearch/blob/f150e2c11df0fe3bef298c55bd867437e50f5f73/modules/ingest-geoip/src/main/java/org/elasticsearch/ingest/geoip/direct/DatabaseConfiguration.java#L58
  _configSchema.schema.literal('GeoIP2-Anonymous-IP'), _configSchema.schema.literal('GeoIP2-City'), _configSchema.schema.literal('GeoIP2-Connection-Type'), _configSchema.schema.literal('GeoIP2-Country'), _configSchema.schema.literal('GeoIP2-Domain'), _configSchema.schema.literal('GeoIP2-Enterprise'), _configSchema.schema.literal('GeoIP2-ISP'),
  // ipinfo names
  _configSchema.schema.literal('asn'), _configSchema.schema.literal('country'), _configSchema.schema.literal('standard_asn'), _configSchema.schema.literal('standard_location'), _configSchema.schema.literal('standard_privacy')])
});
const registerCreateDatabaseRoute = ({
  router,
  lib: {
    handleEsError
  }
}) => {
  router.post({
    path: `${_constants.API_BASE_PATH}/databases`,
    validate: {
      body: bodySchema
    }
  }, async (ctx, req, res) => {
    const {
      client: clusterClient
    } = (await ctx.core).elasticsearch;
    const {
      databaseType,
      databaseName,
      maxmind
    } = req.body;
    const serializedDatabase = (0, _serialization.serializeGeoipDatabase)({
      databaseType,
      databaseName,
      maxmind
    });
    const normalizedDatabaseName = (0, _normalize_database_name.normalizeDatabaseName)(databaseName);
    try {
      // TODO: Replace this request with the one below when the JS client fixed
      await clusterClient.asCurrentUser.transport.request({
        method: 'PUT',
        path: `/_ingest/ip_location/database/${normalizedDatabaseName}`,
        body: serializedDatabase
      });

      // This request fails because there is a bug in the JS client
      // await clusterClient.asCurrentUser.ingest.putGeoipDatabase({
      //   id: normalizedDatabaseName,
      //   body: serializedDatabase,
      // });

      return res.ok({
        body: {
          name: databaseName,
          id: normalizedDatabaseName
        }
      });
    } catch (error) {
      return handleEsError({
        error,
        response: res
      });
    }
  });
};
exports.registerCreateDatabaseRoute = registerCreateDatabaseRoute;