"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.metricsExplorerMetricToTSVBMetric = exports.createTSVBLink = exports.createFilterFromOptions = exports.TSVB_WORKAROUND_INDEX_PATTERN = void 0;
var _rison = require("@kbn/rison");
var _uuid = require("uuid");
var _saferLodashSet = require("@kbn/safer-lodash-set");
var _constants = require("../../../../../../common/constants");
var _color_palette = require("../../../../../../common/color_palette");
var _use_metrics_explorer_options = require("../../hooks/use_metrics_explorer_options");
var _metric_to_format = require("./metric_to_format");
var _types = require("../../../../../common/inventory/types");
var _create_metric_label = require("./create_metric_label");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/*
 We've recently changed the default index pattern in Metrics UI from `metricbeat-*` to
 `metrics-*,metricbeat-*`. There is a bug in TSVB when there is an empty index in the pattern
 the field dropdowns are not populated correctly. This index pattern is a temporary fix.
 See: https://github.com/elastic/kibana/issues/73987
*/
const TSVB_WORKAROUND_INDEX_PATTERN = exports.TSVB_WORKAROUND_INDEX_PATTERN = 'metric*';
const metricsExplorerMetricToTSVBMetric = metric => {
  if (metric.aggregation === 'rate') {
    const metricId = (0, _uuid.v4)();
    const positiveOnlyId = (0, _uuid.v4)();
    const derivativeId = (0, _uuid.v4)();
    return [{
      id: metricId,
      type: 'max',
      field: metric.field || void 0
    }, {
      id: derivativeId,
      type: 'derivative',
      field: metricId,
      unit: '1s'
    }, {
      id: positiveOnlyId,
      type: 'positive_only',
      field: derivativeId
    }];
  } else if (metric.aggregation === 'p95' || metric.aggregation === 'p99') {
    const percentileValue = metric.aggregation === 'p95' ? '95' : '99';
    return [{
      id: (0, _uuid.v4)(),
      type: 'percentile',
      field: metric.field,
      percentiles: [{
        id: (0, _uuid.v4)(),
        value: percentileValue,
        mode: 'line',
        percentile: '',
        shade: 0.2
      }]
    }];
  } else {
    return [{
      id: (0, _uuid.v4)(),
      type: metric.aggregation,
      field: metric.field || void 0
    }];
  }
};
exports.metricsExplorerMetricToTSVBMetric = metricsExplorerMetricToTSVBMetric;
const mapMetricToSeries = chartOptions => metric => {
  const format = (0, _metric_to_format.metricToFormat)(metric);
  return {
    label: (0, _create_metric_label.createMetricLabel)(metric),
    axis_position: 'right',
    chart_type: 'line',
    color: metric.color && (0, _color_palette.colorTransformer)(metric.color) || (0, _color_palette.colorTransformer)(_color_palette.Color.color0),
    fill: chartOptions.type === _use_metrics_explorer_options.MetricsExplorerChartType.area ? 0.5 : 0,
    formatter: format === _types.InfraFormatterType.bits ? _types.InfraFormatterType.bytes : format,
    value_template: 'rate' === metric.aggregation ? '{{value}}/s' : '{{value}}',
    id: (0, _uuid.v4)(),
    line_width: 2,
    metrics: metricsExplorerMetricToTSVBMetric(metric),
    point_size: 0,
    separate_axis: 0,
    split_mode: 'everything',
    stacked: chartOptions.stack ? 'stacked' : 'none'
  };
};
const createFilterFromOptions = (options, series) => {
  const filters = [];
  if (options.filterQuery) {
    filters.push(options.filterQuery);
  }
  if (options.groupBy) {
    const id = series.id.replace('"', '\\"');
    const groupByFilters = Array.isArray(options.groupBy) ? options.groupBy.map((field, index) => {
      if (!series.keys) {
        return null;
      }
      const value = series.keys[index];
      if (!value) {
        return null;
      }
      return `${field}: "${value.replace(/\\/g, '\\\\').replace(/"/g, '\\"')}"`;
    }).join(' and ') : `${options.groupBy} : "${id}"`;
    filters.push(groupByFilters);
  }
  return {
    language: 'kuery',
    query: filters.join(' and ')
  };
};
exports.createFilterFromOptions = createFilterFromOptions;
const createTSVBIndexPattern = alias => {
  if (alias.split(',').length > 1) {
    return TSVB_WORKAROUND_INDEX_PATTERN;
  }
  return alias;
};
const createTSVBLink = (indexPattern, options, series, timeRange, chartOptions) => {
  const tsvbIndexPattern = createTSVBIndexPattern(indexPattern);
  const appState = {
    filters: [],
    linked: false,
    query: {
      language: 'kuery',
      query: ''
    },
    uiState: {},
    vis: {
      aggs: [],
      params: {
        axis_formatter: 'number',
        axis_position: 'left',
        axis_scale: 'normal',
        id: (0, _uuid.v4)(),
        default_index_pattern: tsvbIndexPattern,
        index_pattern: tsvbIndexPattern,
        interval: 'auto',
        series: options.metrics.map(mapMetricToSeries(chartOptions)),
        show_grid: 1,
        show_legend: 1,
        time_field: _constants.TIMESTAMP_FIELD,
        type: 'timeseries',
        filter: createFilterFromOptions(options, series)
      },
      title: series.id,
      type: 'metrics'
    }
  };
  if (chartOptions.yAxisMode === _use_metrics_explorer_options.MetricsExplorerYAxisMode.fromZero) {
    (0, _saferLodashSet.set)(appState, 'vis.params.axis_min', 0);
  }
  const globalState = {
    refreshInterval: {
      pause: true,
      value: 0
    },
    time: {
      from: timeRange.from,
      to: timeRange.to
    }
  };
  return {
    app: 'visualize',
    hash: '/create',
    search: {
      type: 'metrics',
      _g: (0, _rison.encode)(globalState),
      _a: (0, _rison.encode)(appState)
    }
  };
};
exports.createTSVBLink = createTSVBLink;