"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useMetricsHostsAnomaliesResults = void 0;
var _react = require("react");
var _ioTsUtils = require("@kbn/io-ts-utils");
var _use_fetcher = require("../../../../hooks/use_fetcher");
var _infra_ml = require("../../../../../common/http_api/infra_ml");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const stateReducer = (state, action) => {
  const resetPagination = {
    page: 1,
    paginationCursor: undefined
  };
  switch (action.type) {
    case 'changePaginationOptions':
      return {
        ...state,
        ...resetPagination,
        ...action.payload
      };
    case 'changeSortOptions':
      return {
        ...state,
        ...resetPagination,
        ...action.payload
      };
    case 'changeHasNextPage':
      return {
        ...state,
        ...action.payload
      };
    case 'changeLastReceivedCursors':
      return {
        ...state,
        ...action.payload
      };
    case 'fetchNextPage':
      return state.lastReceivedCursors ? {
        ...state,
        page: state.page + 1,
        paginationCursor: {
          searchAfter: state.lastReceivedCursors.nextPageCursor
        }
      } : state;
    case 'fetchPreviousPage':
      return state.lastReceivedCursors ? {
        ...state,
        page: state.page - 1,
        paginationCursor: {
          searchBefore: state.lastReceivedCursors.previousPageCursor
        }
      } : state;
    case 'changeTimeRange':
      return {
        ...state,
        ...resetPagination,
        ...action.payload
      };
    case 'changeFilteredDatasets':
      return {
        ...state,
        ...resetPagination,
        ...action.payload
      };
    default:
      return state;
  }
};
const STATE_DEFAULTS = {
  // NOTE: This piece of state is purely for the client side, it could be extracted out of the hook.
  page: 1,
  // Cursor from the last request
  lastReceivedCursors: undefined,
  // Cursor to use for the next request. For the first request, and therefore not paging, this will be undefined.
  paginationCursor: undefined,
  hasNextPage: false
};
const initStateReducer = (endTime, startTime, defaultSortOptions, defaultPaginationOptions, filteredDatasets) => stateDefaults => {
  return {
    ...stateDefaults,
    paginationOptions: defaultPaginationOptions,
    sortOptions: defaultSortOptions,
    filteredDatasets,
    timeRange: {
      start: startTime,
      end: endTime
    }
  };
};
const useMetricsHostsAnomaliesResults = ({
  endTime,
  startTime,
  sourceId,
  anomalyThreshold,
  defaultSortOptions,
  defaultPaginationOptions,
  filteredDatasets,
  search,
  hostName,
  metric
}, {
  request$,
  active = true
}) => {
  const [reducerState, dispatch] = (0, _react.useReducer)(stateReducer, STATE_DEFAULTS, initStateReducer(endTime, startTime, defaultSortOptions, defaultPaginationOptions, filteredDatasets));
  const [metricsHostsAnomalies, setMetricsHostsAnomalies] = (0, _react.useState)([]);
  const {
    data: response,
    status,
    refetch
  } = (0, _use_fetcher.useFetcher)(async callApi => {
    const apiResponse = await callApi(_infra_ml.INFA_ML_GET_METRICS_HOSTS_ANOMALIES_PATH, {
      method: 'POST',
      body: JSON.stringify(_infra_ml.getMetricsHostsAnomaliesRequestPayloadRT.encode({
        data: {
          sourceId,
          anomalyThreshold,
          timeRange: {
            startTime: reducerState.timeRange.start,
            endTime: reducerState.timeRange.end
          },
          metric,
          query: search,
          sort: reducerState.sortOptions,
          pagination: {
            ...reducerState.paginationOptions,
            cursor: reducerState.paginationCursor
          },
          hostName
        }
      }))
    });
    return (0, _ioTsUtils.decodeOrThrow)(_infra_ml.getMetricsHostsAnomaliesSuccessReponsePayloadRT)(apiResponse);
  }, [anomalyThreshold, hostName, metric, reducerState.paginationCursor, reducerState.paginationOptions, reducerState.sortOptions, reducerState.timeRange.end, reducerState.timeRange.start, search, sourceId], {
    requestObservable$: request$,
    autoFetch: active
  });
  const {
    data
  } = response !== null && response !== void 0 ? response : {};
  (0, _react.useEffect)(() => {
    if ((0, _use_fetcher.isPending)(status) || !data) {
      return;
    }
    const {
      anomalies,
      paginationCursors: requestCursors,
      hasMoreEntries
    } = data;
    if (requestCursors) {
      dispatch({
        type: 'changeLastReceivedCursors',
        payload: {
          lastReceivedCursors: requestCursors
        }
      });
    }

    // Check if we have more "next" entries. "Page" covers the "previous" scenario,
    // since we need to know the page we're on anyway.
    if (!reducerState.paginationCursor || reducerState.paginationCursor && 'searchAfter' in reducerState.paginationCursor) {
      dispatch({
        type: 'changeHasNextPage',
        payload: {
          hasNextPage: hasMoreEntries
        }
      });
    } else if (reducerState.paginationCursor && 'searchBefore' in reducerState.paginationCursor) {
      // We've requested a previous page, therefore there is a next page.
      dispatch({
        type: 'changeHasNextPage',
        payload: {
          hasNextPage: true
        }
      });
    }
    setMetricsHostsAnomalies(anomalies);
  }, [reducerState.paginationCursor, data, status]);
  const changeSortOptions = (0, _react.useCallback)(nextSortOptions => {
    dispatch({
      type: 'changeSortOptions',
      payload: {
        sortOptions: nextSortOptions
      }
    });
  }, [dispatch]);
  const changePaginationOptions = (0, _react.useCallback)(nextPaginationOptions => {
    dispatch({
      type: 'changePaginationOptions',
      payload: {
        paginationOptions: nextPaginationOptions
      }
    });
  }, [dispatch]);

  // Time range has changed
  (0, _react.useEffect)(() => {
    dispatch({
      type: 'changeTimeRange',
      payload: {
        timeRange: {
          start: startTime,
          end: endTime
        }
      }
    });
  }, [startTime, endTime]);

  // Selected datasets have changed
  (0, _react.useEffect)(() => {
    dispatch({
      type: 'changeFilteredDatasets',
      payload: {
        filteredDatasets
      }
    });
  }, [filteredDatasets]);
  const handleFetchNextPage = (0, _react.useCallback)(() => {
    if (reducerState.lastReceivedCursors) {
      dispatch({
        type: 'fetchNextPage'
      });
    }
  }, [dispatch, reducerState]);
  const handleFetchPreviousPage = (0, _react.useCallback)(() => {
    if (reducerState.lastReceivedCursors) {
      dispatch({
        type: 'fetchPreviousPage'
      });
    }
  }, [dispatch, reducerState]);
  const isPendingMetricsHostsAnomalies = (0, _use_fetcher.isPending)(status);
  const hasFailedLoadingMetricsHostsAnomalies = (0, _use_fetcher.isFailure)(status);
  return {
    metricsHostsAnomalies,
    getMetricsHostsAnomalies: refetch,
    isPendingMetricsHostsAnomalies,
    hasFailedLoadingMetricsHostsAnomalies,
    changeSortOptions,
    sortOptions: reducerState.sortOptions,
    changePaginationOptions,
    paginationOptions: reducerState.paginationOptions,
    fetchPreviousPage: reducerState.page > 1 ? handleFetchPreviousPage : undefined,
    fetchNextPage: reducerState.hasNextPage ? handleFetchNextPage : undefined,
    page: reducerState.page,
    timeRange: reducerState.timeRange
  };
};
exports.useMetricsHostsAnomaliesResults = useMetricsHostsAnomaliesResults;