"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseInlineFunctionCalls = parseInlineFunctionCalls;
var _rxjs = require("rxjs");
var _chat_complete = require("../../../common/chat_complete");
var _errors = require("../../../common/errors");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function matchOnSignalStart(buffer) {
  if (buffer.includes(_constants.TOOL_USE_START)) {
    const split = buffer.split(_constants.TOOL_USE_START);
    return [split[0], _constants.TOOL_USE_START + split[1]];
  }
  for (let i = 0; i < buffer.length; i++) {
    const remaining = buffer.substring(i);
    if (_constants.TOOL_USE_START.startsWith(remaining)) {
      return [buffer.substring(0, i), remaining];
    }
  }
  return false;
}
function parseInlineFunctionCalls({
  logger
}) {
  return source => {
    let functionCallBuffer = '';

    // As soon as we see a TOOL_USE_START token, we write all chunks
    // to a buffer, that we flush as a function request if we
    // spot the stop sequence.

    return new _rxjs.Observable(subscriber => {
      function parseFunctionCall(buffer) {
        logger.debug('Parsing function call:\n' + buffer);
        const match = buffer.match(/<\|tool_use_start\|>\s*```json\n?(.*?)(\n?```\s*).*<\|tool_use_end\|>/s);
        const functionCallBody = match === null || match === void 0 ? void 0 : match[1];
        if (!functionCallBody) {
          throw (0, _errors.createInferenceInternalError)(`Invalid function call syntax`);
        }
        const parsedFunctionCall = JSON.parse(functionCallBody);
        logger.debug(() => 'Parsed function call:\n ' + JSON.stringify(parsedFunctionCall));
        if (!parsedFunctionCall.name) {
          throw (0, _errors.createInferenceInternalError)(`Missing name for tool use`);
        }
        subscriber.next({
          content: '',
          tool_calls: [{
            index: 0,
            toolCallId: parsedFunctionCall.name,
            function: {
              name: parsedFunctionCall.name,
              arguments: JSON.stringify(parsedFunctionCall.input || {})
            }
          }],
          type: _chat_complete.ChatCompletionEventType.ChatCompletionChunk
        });
      }
      source.subscribe({
        next: event => {
          if (event.type === _chat_complete.ChatCompletionEventType.ChatCompletionTokenCount) {
            subscriber.next(event);
            return;
          }
          const {
            type,
            content
          } = event;
          function next(contentToEmit) {
            subscriber.next({
              type,
              content: contentToEmit,
              tool_calls: []
            });
          }
          const match = matchOnSignalStart(functionCallBuffer + content);
          if (match) {
            const [beforeStartSignal, afterStartSignal] = match;
            functionCallBuffer = afterStartSignal;
            if (beforeStartSignal) {
              next(beforeStartSignal);
            }
            if (functionCallBuffer.includes(_constants.TOOL_USE_END)) {
              const [beforeEndSignal, afterEndSignal] = functionCallBuffer.split(_constants.TOOL_USE_END);
              try {
                parseFunctionCall(beforeEndSignal + _constants.TOOL_USE_END);
                functionCallBuffer = '';
                next(afterEndSignal);
              } catch (error) {
                subscriber.error(error);
              }
            }
          } else {
            functionCallBuffer = '';
            next(content);
          }
        },
        complete: () => {
          subscriber.complete();
        },
        error: error => {
          subscriber.error(error);
        }
      });
    });
  };
}