"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.openAIAdapter = void 0;
var _rxjs = require("rxjs");
var _stream = require("stream");
var _chat_complete = require("../../../../common/chat_complete");
var _errors = require("../../../../common/chat_complete/errors");
var _errors2 = require("../../../../common/errors");
var _event_source_stream_into_observable = require("../../../util/event_source_stream_into_observable");
var _simulated_function_calling = require("../../simulated_function_calling");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const openAIAdapter = exports.openAIAdapter = {
  chatComplete: ({
    executor,
    system,
    messages,
    toolChoice,
    tools,
    functionCalling,
    logger
  }) => {
    const stream = true;
    const simulatedFunctionCalling = functionCalling === 'simulated';
    let request;
    if (simulatedFunctionCalling) {
      const wrapped = (0, _simulated_function_calling.wrapWithSimulatedFunctionCalling)({
        system,
        messages,
        toolChoice,
        tools
      });
      request = {
        stream,
        messages: messagesToOpenAI({
          system: wrapped.system,
          messages: wrapped.messages
        }),
        temperature: 0
      };
    } else {
      request = {
        stream,
        messages: messagesToOpenAI({
          system,
          messages
        }),
        tool_choice: toolChoiceToOpenAI(toolChoice),
        tools: toolsToOpenAI(tools),
        temperature: 0
      };
    }
    return (0, _rxjs.from)(executor.invoke({
      subAction: 'stream',
      subActionParams: {
        body: JSON.stringify(request),
        stream
      }
    })).pipe((0, _rxjs.switchMap)(response => {
      if ((0, _stream.isReadable)(response.data)) {
        return (0, _event_source_stream_into_observable.eventSourceStreamIntoObservable)(response.data);
      }
      return (0, _rxjs.throwError)(() => (0, _errors2.createInferenceInternalError)('Unexpected error', response.data));
    }), (0, _rxjs.filter)(line => !!line && line !== '[DONE]'), (0, _rxjs.map)(line => JSON.parse(line)), (0, _rxjs.tap)(line => {
      if ('error' in line) {
        throw (0, _errors2.createInferenceInternalError)(line.error.message);
      }
      if ('choices' in line && line.choices.length && line.choices[0].finish_reason === 'length') {
        throw (0, _errors.createTokenLimitReachedError)();
      }
    }), (0, _rxjs.filter)(line => 'object' in line && line.object === 'chat.completion.chunk' && line.choices.length > 0), (0, _rxjs.map)(chunk => {
      var _delta$content, _delta$tool_calls$map, _delta$tool_calls;
      const delta = chunk.choices[0].delta;
      return {
        type: _chat_complete.ChatCompletionEventType.ChatCompletionChunk,
        content: (_delta$content = delta.content) !== null && _delta$content !== void 0 ? _delta$content : '',
        tool_calls: (_delta$tool_calls$map = (_delta$tool_calls = delta.tool_calls) === null || _delta$tool_calls === void 0 ? void 0 : _delta$tool_calls.map(toolCall => {
          var _toolCall$function$na, _toolCall$function, _toolCall$function$ar, _toolCall$function2, _toolCall$id;
          return {
            function: {
              name: (_toolCall$function$na = (_toolCall$function = toolCall.function) === null || _toolCall$function === void 0 ? void 0 : _toolCall$function.name) !== null && _toolCall$function$na !== void 0 ? _toolCall$function$na : '',
              arguments: (_toolCall$function$ar = (_toolCall$function2 = toolCall.function) === null || _toolCall$function2 === void 0 ? void 0 : _toolCall$function2.arguments) !== null && _toolCall$function$ar !== void 0 ? _toolCall$function$ar : ''
            },
            toolCallId: (_toolCall$id = toolCall.id) !== null && _toolCall$id !== void 0 ? _toolCall$id : '',
            index: toolCall.index
          };
        })) !== null && _delta$tool_calls$map !== void 0 ? _delta$tool_calls$map : []
      };
    }), simulatedFunctionCalling ? (0, _simulated_function_calling.parseInlineFunctionCalls)({
      logger
    }) : _rxjs.identity);
  }
};
function toolsToOpenAI(tools) {
  return tools ? Object.entries(tools).map(([toolName, {
    description,
    schema
  }]) => {
    return {
      type: 'function',
      function: {
        name: toolName,
        description,
        parameters: schema !== null && schema !== void 0 ? schema : {
          type: 'object',
          properties: {}
        }
      }
    };
  }) : undefined;
}
function toolChoiceToOpenAI(toolChoice) {
  return typeof toolChoice === 'string' ? toolChoice : toolChoice ? {
    function: {
      name: toolChoice.function
    },
    type: 'function'
  } : undefined;
}
function messagesToOpenAI({
  system,
  messages
}) {
  const systemMessage = system ? {
    role: 'system',
    content: system
  } : undefined;
  return [...(systemMessage ? [systemMessage] : []), ...messages.map(message => {
    var _message$toolCalls;
    const role = message.role;
    switch (role) {
      case _chat_complete.MessageRole.Assistant:
        const assistantMessage = {
          role: 'assistant',
          content: message.content,
          tool_calls: (_message$toolCalls = message.toolCalls) === null || _message$toolCalls === void 0 ? void 0 : _message$toolCalls.map(toolCall => {
            return {
              function: {
                name: toolCall.function.name,
                arguments: 'arguments' in toolCall.function ? JSON.stringify(toolCall.function.arguments) : '{}'
              },
              id: toolCall.toolCallId,
              type: 'function'
            };
          })
        };
        return assistantMessage;
      case _chat_complete.MessageRole.User:
        const userMessage = {
          role: 'user',
          content: message.content
        };
        return userMessage;
      case _chat_complete.MessageRole.Tool:
        const toolMessage = {
          role: 'tool',
          content: JSON.stringify(message.response),
          tool_call_id: message.toolCallId
        };
        return toolMessage;
    }
  })];
}