"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bedrockClaudeAdapter = void 0;
var _rxjs = require("rxjs");
var _serde_utils = require("./serde_utils");
var _chat_complete = require("../../../../common/chat_complete");
var _errors = require("../../../../common/errors");
var _tools = require("../../../../common/chat_complete/tools");
var _serde_eventstream_into_observable = require("./serde_eventstream_into_observable");
var _process_completion_chunks = require("./process_completion_chunks");
var _prompts = require("./prompts");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const bedrockClaudeAdapter = exports.bedrockClaudeAdapter = {
  chatComplete: ({
    executor,
    system,
    messages,
    toolChoice,
    tools
  }) => {
    const noToolUsage = toolChoice === _tools.ToolChoiceType.none;
    const subActionParams = {
      system: noToolUsage ? (0, _prompts.addNoToolUsageDirective)(system) : system,
      messages: messagesToBedrock(messages),
      tools: noToolUsage ? [] : toolsToBedrock(tools, messages),
      toolChoice: toolChoiceToBedrock(toolChoice),
      temperature: 0,
      stopSequences: ['\n\nHuman:']
    };
    return (0, _rxjs.from)(executor.invoke({
      subAction: 'invokeStream',
      subActionParams
    })).pipe((0, _rxjs.switchMap)(response => {
      const readable = response.data;
      return (0, _serde_eventstream_into_observable.serdeEventstreamIntoObservable)(readable);
    }), (0, _rxjs.tap)(eventData => {
      if ('modelStreamErrorException' in eventData) {
        throw (0, _errors.createInferenceInternalError)(eventData.modelStreamErrorException.originalMessage);
      }
    }), (0, _rxjs.filter)(value => {
      var _value$chunk, _value$chunk$headers, _value$chunk$headers$;
      return 'chunk' in value && ((_value$chunk = value.chunk) === null || _value$chunk === void 0 ? void 0 : (_value$chunk$headers = _value$chunk.headers) === null || _value$chunk$headers === void 0 ? void 0 : (_value$chunk$headers$ = _value$chunk$headers[':event-type']) === null || _value$chunk$headers$ === void 0 ? void 0 : _value$chunk$headers$.value) === 'chunk';
    }), (0, _rxjs.map)(message => {
      return (0, _serde_utils.parseSerdeChunkMessage)(message.chunk);
    }), (0, _process_completion_chunks.processCompletionChunks)());
  }
};
const toolChoiceToBedrock = toolChoice => {
  if (toolChoice === _tools.ToolChoiceType.required) {
    return {
      type: 'any'
    };
  } else if (toolChoice === _tools.ToolChoiceType.auto) {
    return {
      type: 'auto'
    };
  } else if (typeof toolChoice === 'object') {
    return {
      type: 'tool',
      name: toolChoice.function
    };
  }
  // ToolChoiceType.none is not supported by claude
  // we are adding a directive to the system instructions instead in that case.
  return undefined;
};
const toolsToBedrock = (tools, messages) => {
  function walkSchema(schemaPart) {
    if (schemaPart.type === 'object' && schemaPart.properties) {
      return {
        ...schemaPart,
        properties: Object.fromEntries(Object.entries(schemaPart.properties).map(([key, childSchemaPart]) => {
          return [key, walkSchema(childSchemaPart)];
        }))
      };
    }
    if (schemaPart.type === 'array') {
      return {
        ...schemaPart,
        // Claude is prone to ignoring the "array" part of an array type
        description: schemaPart.description + '. Must be provided as a JSON array',
        items: walkSchema(schemaPart.items)
      };
    }
    return schemaPart;
  }
  if (tools) {
    return Object.entries(tools).map(([toolName, toolDef]) => {
      var _toolDef$schema;
      return {
        name: toolName,
        description: toolDef.description,
        input_schema: walkSchema((_toolDef$schema = toolDef.schema) !== null && _toolDef$schema !== void 0 ? _toolDef$schema : {
          type: 'object',
          properties: {}
        })
      };
    });
  }
  const hasToolUse = messages.filter(message => {
    var _message$toolCalls;
    return message.role === _chat_complete.MessageRole.Tool || message.role === _chat_complete.MessageRole.Assistant && ((_message$toolCalls = message.toolCalls) === null || _message$toolCalls === void 0 ? void 0 : _message$toolCalls.length);
  });
  if (hasToolUse) {
    return [{
      name: 'do_not_call_this_tool',
      description: 'Do not call this tool, it is strictly forbidden',
      input_schema: {
        type: 'object',
        properties: {}
      }
    }];
  }
};
const messagesToBedrock = messages => {
  return messages.map(message => {
    switch (message.role) {
      case _chat_complete.MessageRole.User:
        return {
          role: 'user',
          rawContent: [{
            type: 'text',
            text: message.content
          }]
        };
      case _chat_complete.MessageRole.Assistant:
        return {
          role: 'assistant',
          rawContent: [...(message.content ? [{
            type: 'text',
            text: message.content
          }] : []), ...(message.toolCalls ? message.toolCalls.map(toolCall => {
            return {
              type: 'tool_use',
              id: toolCall.toolCallId,
              name: toolCall.function.name,
              input: 'arguments' in toolCall.function ? toolCall.function.arguments : {}
            };
          }) : [])]
        };
      case _chat_complete.MessageRole.Tool:
        return {
          role: 'user',
          rawContent: [{
            type: 'tool_result',
            tool_use_id: message.toolCallId,
            content: JSON.stringify(message.response)
          }]
        };
    }
  });
};