"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerCreateRoute = registerCreateRoute;
var _configSchema = require("@kbn/config-schema");
var _ = require("..");
var _enrich_policies = require("../../../lib/enrich_policies");
var _lib = require("../../../../common/lib");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const validationSchema = _configSchema.schema.object({
  policy: _configSchema.schema.object({
    name: _configSchema.schema.string({
      maxLength: 1000
    }),
    type: _configSchema.schema.oneOf([_configSchema.schema.literal('match'), _configSchema.schema.literal('range'), _configSchema.schema.literal('geo_match')]),
    matchField: _configSchema.schema.string(),
    enrichFields: _configSchema.schema.arrayOf(_configSchema.schema.string()),
    sourceIndices: _configSchema.schema.arrayOf(_configSchema.schema.string()),
    query: _configSchema.schema.maybe(_configSchema.schema.any())
  })
});
const querySchema = _configSchema.schema.object({
  executePolicyAfterCreation: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('true'), _configSchema.schema.literal('false')]))
});
const getMatchingIndicesSchema = _configSchema.schema.object({
  pattern: _configSchema.schema.string()
}, {
  unknowns: 'allow'
});
const getFieldsFromIndicesSchema = _configSchema.schema.object({
  indices: _configSchema.schema.arrayOf(_configSchema.schema.string())
});
function registerCreateRoute({
  router,
  lib: {
    handleEsError
  }
}) {
  router.post({
    path: (0, _.addInternalBasePath)('/enrich_policies'),
    validate: {
      body: validationSchema,
      query: querySchema
    }
  }, async (context, request, response) => {
    var _request$query;
    const client = (await context.core).elasticsearch.client;
    const executeAfter = Boolean((_request$query = request.query) === null || _request$query === void 0 ? void 0 : _request$query.executePolicyAfterCreation);
    const {
      policy
    } = request.body;
    const serializedPolicy = (0, _lib.serializeAsESPolicy)(policy);
    try {
      const res = await _enrich_policies.enrichPoliciesActions.create(client, policy.name, serializedPolicy);
      if (executeAfter) {
        try {
          await _enrich_policies.enrichPoliciesActions.execute(client, policy.name);
        } catch (error) {
          // If executing the policy fails, remove the previously created policy and
          // return the error.
          await _enrich_policies.enrichPoliciesActions.remove(client, policy.name);
          return handleEsError({
            error,
            response
          });
        }
      }
      return response.ok({
        body: res
      });
    } catch (error) {
      return handleEsError({
        error,
        response
      });
    }
  });
  router.post({
    path: (0, _.addInternalBasePath)('/enrich_policies/get_matching_indices'),
    validate: {
      body: getMatchingIndicesSchema
    }
  }, async (context, request, response) => {
    let {
      pattern
    } = request.body;
    const client = (await context.core).elasticsearch.client;

    // Add wildcards to the search query to match the behavior of the
    // index pattern search in the Kibana UI.
    if (!pattern.startsWith('*')) {
      pattern = `*${pattern}`;
    }
    if (!pattern.endsWith('*')) {
      pattern = `${pattern}*`;
    }
    try {
      const indices = await (0, _helpers.getIndices)(client, pattern);
      return response.ok({
        body: {
          indices
        }
      });
    } catch (error) {
      return handleEsError({
        error,
        response
      });
    }
  });
  router.post({
    path: (0, _.addInternalBasePath)('/enrich_policies/get_fields_from_indices'),
    validate: {
      body: getFieldsFromIndicesSchema
    }
  }, async (context, request, response) => {
    const {
      indices
    } = request.body;
    const client = (await context.core).elasticsearch.client;
    try {
      const fieldsPerIndex = await Promise.all(indices.map(index => client.asCurrentUser.fieldCaps({
        index,
        fields: ['*'],
        allow_no_indices: true,
        ignore_unavailable: true,
        filters: '-metadata'
      }, {
        ignore: [404],
        meta: true
      })));
      const serializedFieldsPerIndex = indices.map((indexName, mapIndex) => {
        const fields = fieldsPerIndex[mapIndex];
        const json = fields.statusCode === 404 ? {
          fields: []
        } : fields.body;
        return {
          index: indexName,
          fields: (0, _helpers.normalizeFieldsList)(json.fields)
        };
      });
      return response.ok({
        body: {
          indices: serializedFieldsPerIndex,
          commonFields: (0, _helpers.getCommonFields)(serializedFieldsPerIndex)
        }
      });
    } catch (error) {
      return handleEsError({
        error,
        response
      });
    }
  });
}