"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.activateTranslation = activateTranslation;
exports.getIsInitialized = void 0;
exports.getLocale = getLocale;
exports.getTranslation = getTranslation;
exports.init = init;
exports.load = load;
exports.translate = translate;
var _intl = require("@formatjs/intl");
var _error_handler = require("./error_handler");
var _formats = require("./formats");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const EN_LOCALE = 'en';
const defaultLocale = EN_LOCALE;

/**
 * Currently we are depending on this singleton pattern to
 * update the locale. This is mainly to make it easier on developers
 * to use i18n by importing it anywhere in their code and using it directly
 * without having to pass it around.
 * This pattern has several limitations and can cause unexpected bugs. The main limitation
 * is that we cannot server multiple locales on the server side based on the user requested
 * locale.
 */
let intl;
let isInitialized = false;
/**
 * ideally here we would be using a `throw new Error()` if i18n.translate is called before init();
 * to make sure i18n is initialized before any message is attempting to be translated.
 *
 * Especially since these messages will go unnoticed since the translations might be provided in the translation files
 * but Kibana will use the default message since the locales are not loaded yet.
 *
 * we need to get there at some point but this means removing all static i18n imports from the server side.
 */
intl = (0, _intl.createIntl)({
  locale: defaultLocale,
  defaultFormats: _formats.defaultEnFormats,
  defaultLocale,
  onError: () => void 0
});
const getIsInitialized = () => {
  return isInitialized;
};
/**
 * Normalizes locale to make it consistent with IntlMessageFormat locales
 * @param locale
 */
exports.getIsInitialized = getIsInitialized;
function normalizeLocale(locale) {
  return locale.toLowerCase();
}

/**
 * Provides a way to register translations with the engine
 */
function activateTranslation(newTranslation) {
  if (!newTranslation.locale || typeof newTranslation.locale !== 'string') {
    throw new Error('[I18n] A `locale` must be a non-empty string to add messages.');
  }
  const config = {
    locale: normalizeLocale(newTranslation.locale),
    messages: newTranslation.messages,
    defaultFormats: _formats.defaultEnFormats,
    defaultLocale,
    onError: _error_handler.handleIntlError
  };

  // formatJS differentiates between `formats: undefined` and unset `formats`.
  if (newTranslation.formats) {
    config.formats = newTranslation.formats;
  }
  const cache = (0, _intl.createIntlCache)();
  intl = (0, _intl.createIntl)(config, cache);
}

/**
 * Returns messages for the current language
 */
function getTranslation() {
  return {
    messages: intl.messages,
    locale: intl.locale,
    defaultLocale: intl.defaultLocale,
    defaultFormats: intl.defaultFormats,
    formats: intl.formats
  };
}

/**
 * Returns the current locale
 * Shortcut to getTranslation().locale
 */
function getLocale() {
  return intl.locale;
}
/**
 * Translate message by id
 * @param id - translation id to be translated
 * @param [options]
 * @param [options.values] - values to pass into translation
 * @param [options.defaultMessage] - will be used unless translation was successful
 * @param [options.description] - message description, used by translators and other devs to understand the message context.
 * @param [options.ignoreTag] - Whether to treat HTML/XML tags as string literal instead of parsing them as tag token. When this is false we only allow simple tags without any attributes
 */
function translate(id, {
  values = {},
  description,
  defaultMessage,
  ignoreTag
}) {
  if (!id || typeof id !== 'string') {
    throw new Error('[I18n] An `id` must be a non-empty string to translate a message.');
  }
  try {
    if (!defaultMessage) {
      throw new Error('Missing `defaultMessage`.');
    }
    return intl.formatMessage({
      id,
      defaultMessage,
      description
    }, values, {
      ignoreTag,
      shouldParseSkeletons: true
    });
  } catch (e) {
    throw new Error(`[I18n] Error formatting the default message for: "${id}".\n${e}`);
  }
}

/**
 * Initializes the engine
 * @param newTranslation
 */
function init(newTranslation) {
  if (typeof (newTranslation === null || newTranslation === void 0 ? void 0 : newTranslation.locale) !== 'string') {
    return;
  }
  activateTranslation(newTranslation);
  isInitialized = true;
}

/**
 * Loads JSON with translations from the specified URL and initializes i18n engine with them.
 * @param translationsUrl URL pointing to the JSON bundle with translations.
 */
async function load(translationsUrl) {
  // Once this package is integrated into core Kibana we should switch to an abstraction
  // around `fetch` provided by the platform, e.g. `kfetch`.
  const response = await fetch(translationsUrl, {
    credentials: 'same-origin'
  });
  if (response.status >= 400) {
    throw new Error(`Translations request failed with status code: ${response.status}`);
  }
  const newTranslation = await response.json();
  if (!newTranslation || !newTranslation.locale || typeof newTranslation.locale !== 'string') {
    return;
  }
  init(newTranslation);
  isInitialized = true;
}