"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EventReporter = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _analytics = require("@kbn/analytics");
var _types = require("../types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class EventReporter {
  constructor({
    analytics,
    usageCollection
  }) {
    (0, _defineProperty2.default)(this, "reportEvent", void 0);
    (0, _defineProperty2.default)(this, "trackUiMetric", void 0);
    (0, _defineProperty2.default)(this, "focusStart", Infinity);
    this.reportEvent = analytics.reportEvent;
    if (usageCollection) {
      this.trackUiMetric = (metricType, eventName, context) => {
        let counter;
        switch (eventName) {
          case _types.ClickMetric.USER_NAVIGATED_TO_APPLICATION:
          case _types.ClickMetric.USER_NAVIGATED_TO_SAVED_OBJECT:
            counter = [metricType, [eventName, `${eventName}_${context}`]];
            break;
          default:
            // track simple UI Counter metrics
            counter = [metricType, eventName];
        }
        usageCollection.reportUiCounter('global_search_bar', ...counter);
      };
    } else {
      this.trackUiMetric = () => {};
    }
  }

  /**
   * Called when the text input component has the cursor focused
   */
  searchFocus() {
    this.trackUiMetric(_analytics.METRIC_TYPE.COUNT, _types.CountMetric.SEARCH_FOCUS);
    this.focusStart = Date.now();
  }

  /**
   * Called when the text input component has lost focus
   */
  searchBlur() {
    const focusTime = Date.now() - this.focusStart;
    if (focusTime > 0) {
      this.reportEvent(_types.EventMetric.SEARCH_BLUR, {
        [_types.FieldType.FOCUS_TIME]: focusTime
      });
    }

    // reset internal states
    this.focusStart = Infinity;
  }

  /**
   * Called on each keystroke as the user changes the search terms
   */
  searchRequest() {
    this.trackUiMetric(_analytics.METRIC_TYPE.COUNT, _types.CountMetric.SEARCH_REQUEST);
  }

  /**
   * Called when the users uses the shortcut to make the search bar visible and focus the cursor
   */
  shortcutUsed() {
    this.trackUiMetric(_analytics.METRIC_TYPE.COUNT, _types.CountMetric.SHORTCUT_USED);
  }

  /**
   * Called when the users selects an application in their search results
   */
  navigateToApplication(context) {
    var _context$application, _context$searchValue;
    const application = (_context$application = context === null || context === void 0 ? void 0 : context.application) !== null && _context$application !== void 0 ? _context$application : 'unknown';
    this.trackUiMetric(_analytics.METRIC_TYPE.CLICK, _types.ClickMetric.USER_NAVIGATED_TO_APPLICATION, application);
    const terms = (_context$searchValue = context === null || context === void 0 ? void 0 : context.searchValue) !== null && _context$searchValue !== void 0 ? _context$searchValue : '';
    this.reportEvent(_types.EventMetric.CLICK_APPLICATION, {
      [_types.FieldType.TERMS]: terms,
      [_types.FieldType.APPLICATION]: application,
      [_types.FieldType.SELECTED_RANK]: context.selectedRank,
      [_types.FieldType.SELECTED_LABEL]: context.selectedLabel
    });
  }

  /**
   * Called when the users selects Saved Object in their search results
   */
  navigateToSavedObject(context) {
    var _context$type, _context$searchValue2;
    const type = (_context$type = context === null || context === void 0 ? void 0 : context.type) !== null && _context$type !== void 0 ? _context$type : 'unknown';
    this.trackUiMetric(_analytics.METRIC_TYPE.CLICK, _types.ClickMetric.USER_NAVIGATED_TO_SAVED_OBJECT, type);
    const terms = (_context$searchValue2 = context === null || context === void 0 ? void 0 : context.searchValue) !== null && _context$searchValue2 !== void 0 ? _context$searchValue2 : '';
    this.reportEvent(_types.EventMetric.CLICK_SAVED_OBJECT, {
      [_types.FieldType.TERMS]: terms,
      [_types.FieldType.SAVED_OBJECT_TYPE]: type,
      [_types.FieldType.SELECTED_RANK]: context.selectedRank,
      [_types.FieldType.SELECTED_LABEL]: context.selectedLabel
    });
  }

  /**
   * Called from error handlers
   */
  error(context) {
    var _context$message$toSt, _context$searchValue3;
    this.trackUiMetric(_analytics.METRIC_TYPE.COUNT, _types.CountMetric.UNHANDLED_ERROR);
    const message = (_context$message$toSt = context === null || context === void 0 ? void 0 : context.message.toString()) !== null && _context$message$toSt !== void 0 ? _context$message$toSt : 'unknown';
    const terms = (_context$searchValue3 = context === null || context === void 0 ? void 0 : context.searchValue) !== null && _context$searchValue3 !== void 0 ? _context$searchValue3 : '';
    this.reportEvent(_types.EventMetric.ERROR, {
      [_types.FieldType.TERMS]: terms,
      [_types.FieldType.ERROR_MESSAGE]: message
    });
  }
}
exports.EventReporter = EventReporter;