"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CsvESQLGenerator = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _elasticsearch = require("@elastic/elasticsearch");
var _common = require("@kbn/data-plugin/common");
var _esQuery = require("@kbn/es-query");
var _i18n = require("@kbn/i18n");
var _reportingCommon = require("@kbn/reporting-common");
var _lodash = require("lodash");
var _constants = require("../constants");
var _get_export_settings = require("./lib/get_export_settings");
var _i18n_texts = require("./lib/i18n_texts");
var _max_size_string_builder = require("./lib/max_size_string_builder");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class CsvESQLGenerator {
  constructor(job, config, taskInstanceFields, clients, cancellationToken, logger, stream) {
    (0, _defineProperty2.default)(this, "csvContainsFormulas", false);
    (0, _defineProperty2.default)(this, "maxSizeReached", false);
    (0, _defineProperty2.default)(this, "csvRowCount", 0);
    this.job = job;
    this.config = config;
    this.taskInstanceFields = taskInstanceFields;
    this.clients = clients;
    this.cancellationToken = cancellationToken;
    this.logger = logger;
    this.stream = stream;
  }
  async generateData() {
    var _reportingError;
    const settings = await (0, _get_export_settings.getExportSettings)(this.clients.uiSettings, this.taskInstanceFields, this.config, this.job.browserTimezone, this.logger);
    let reportingError;
    const warnings = [];
    const {
      maxSizeBytes,
      bom,
      escapeFormulaValues
    } = settings;
    const builder = new _max_size_string_builder.MaxSizeStringBuilder(this.stream, (0, _reportingCommon.byteSizeValueToNumber)(maxSizeBytes), bom);
    const filter = this.job.filters && (0, _esQuery.buildEsQuery)(undefined, [], this.job.filters, (0, _common.getEsQueryConfig)(this.clients.uiSettings));
    const searchParams = {
      params: {
        query: this.job.query.esql,
        filter,
        // locale can be used for number/date formatting
        locale: _i18n.i18n.getLocale()
        // TODO: time_zone support was temporarily removed from ES|QL,
        // we will need to add it back in once it is supported again.
        // https://github.com/elastic/elasticsearch/pull/102767
        // timezone
      }
    };
    try {
      var _rawResponse$columns$, _rawResponse$columns;
      const abortController = new AbortController();
      this.cancellationToken.on(() => abortController.abort());
      const {
        rawResponse,
        warning
      } = await (0, _rxjs.lastValueFrom)(this.clients.data.search(searchParams, {
        strategy: _common.ESQL_SEARCH_STRATEGY,
        abortSignal: abortController.signal,
        transport: {
          requestTimeout: settings.scroll.duration(this.taskInstanceFields)
        }
      }));
      if (warning) {
        warnings.push(warning);
      }
      const responseColumns = (_rawResponse$columns$ = (_rawResponse$columns = rawResponse.columns) === null || _rawResponse$columns === void 0 ? void 0 : _rawResponse$columns.map(({
        name
      }) => name)) !== null && _rawResponse$columns$ !== void 0 ? _rawResponse$columns$ : [];
      const visibleColumns = this.job.columns && this.job.columns.length > 0 ? this.job.columns.filter(column => responseColumns.includes(column)) : responseColumns;
      const rows = rawResponse.values.map(row => (0, _lodash.zipObject)(responseColumns, row));
      const header = Array.from(visibleColumns).map(this.escapeValues(settings)).join(settings.separator) + '\n';
      builder.tryAppend(header);
      await this.generateRows(visibleColumns, rows, builder, settings);
    } catch (err) {
      this.logger.error(err);
      if (err instanceof _elasticsearch.errors.ResponseError) {
        var _err$statusCode;
        if ([401, 403].includes((_err$statusCode = err.statusCode) !== null && _err$statusCode !== void 0 ? _err$statusCode : 0)) {
          reportingError = new _reportingCommon.AuthenticationExpiredError();
          warnings.push(_i18n_texts.i18nTexts.authenticationError.partialResultsMessage);
        } else {
          var _err$statusCode2;
          warnings.push(_i18n_texts.i18nTexts.esErrorMessage((_err$statusCode2 = err.statusCode) !== null && _err$statusCode2 !== void 0 ? _err$statusCode2 : 0, String(err.body)));
        }
      } else {
        var _err$message;
        warnings.push(_i18n_texts.i18nTexts.unknownError((_err$message = err === null || err === void 0 ? void 0 : err.message) !== null && _err$message !== void 0 ? _err$message : err));
      }
    }
    return {
      content_type: _constants.CONTENT_TYPE_CSV,
      csv_contains_formulas: this.csvContainsFormulas && !escapeFormulaValues,
      max_size_reached: this.maxSizeReached,
      metrics: {
        csv: {
          rows: this.csvRowCount
        }
      },
      warnings,
      error_code: (_reportingError = reportingError) === null || _reportingError === void 0 ? void 0 : _reportingError.code
    };
  }

  /*
   * Format a Datatable into rows of CSV content
   */
  async generateRows(columns, rows, builder, settings) {
    this.logger.debug(`Building ${rows.length} CSV data rows`);
    for (const dataTableRow of rows) {
      if (this.cancellationToken.isCancelled()) {
        break;
      }

      /*
       * Intrinsically, generating the rows is a synchronous process. Awaiting
       * on a setImmediate call here partititions what could be a very long and
       * CPU-intenstive synchronous process into an asychronous process. This
       * give NodeJS to process other asychronous events that wait on the Event
       * Loop.
       *
       * See: https://nodejs.org/en/docs/guides/dont-block-the-event-loop/
       *
       * It's likely this creates a lot of context switching, and adds to the
       * time it would take to generate the CSV. There are alternatives to the
       * chosen performance solution:
       *
       * 1. Partition the synchronous process with fewer partitions, by using
       * the loop counter to call setImmediate only every N amount of rows.
       * Testing is required to see what the best N value for most data will
       * be.
       *
       * 2. Use a C++ add-on to generate the CSV using the Node Worker Pool
       * instead of using the Event Loop
       */
      await new Promise(setImmediate);
      const rowDefinition = [];
      const escape = this.escapeValues(settings);
      for (const column of columns) {
        let formattedValue = escape(`${dataTableRow[column]}`);
        if (formattedValue === 'null') formattedValue = '';
        if (formattedValue === 'undefined') formattedValue = '';
        rowDefinition.push(formattedValue);
      }
      if (!builder.tryAppend(rowDefinition.join(settings.separator) + '\n')) {
        this.logger.warn(`Max Size Reached after ${this.csvRowCount} rows.`);
        this.maxSizeReached = true;
        if (this.cancellationToken) {
          this.cancellationToken.cancel();
        }
        break;
      }
      this.csvRowCount++;
    }
  }
  escapeValues(settings) {
    return value => {
      if (settings.checkForFormulas && (0, _common.cellHasFormulas)(value)) {
        this.csvContainsFormulas = true; // set warning if cell value has a formula
      }
      return settings.escapeValue(value);
    };
  }
}
exports.CsvESQLGenerator = CsvESQLGenerator;