"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.VERSION = exports.UnenrollInactiveAgentsTask = exports.TYPE = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/core/server");
var _uuid = require("uuid");
var _task = require("@kbn/task-manager-plugin/server/task");
var _elasticsearch = require("@elastic/elasticsearch");
var _constants = require("../constants");
var _agents = require("../services/agents");
var _unenroll_action_runner = require("../services/agents/unenroll_action_runner");
var _services = require("../services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TYPE = exports.TYPE = 'fleet:unenroll-inactive-agents-task';
const VERSION = exports.VERSION = '1.0.0';
const TITLE = 'Fleet Unenroll Inactive Agent Task';
const SCOPE = ['fleet'];
const INTERVAL = '10m';
const TIMEOUT = '1m';
const UNENROLLMENT_BATCHSIZE = 1000;
const POLICIES_BATCHSIZE = 500;
class UnenrollInactiveAgentsTask {
  constructor(setupContract) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "wasStarted", false);
    (0, _defineProperty2.default)(this, "abortController", new AbortController());
    (0, _defineProperty2.default)(this, "start", async ({
      taskManager
    }) => {
      if (!taskManager) {
        this.logger.error('[UnenrollInactiveAgentsTask] Missing required service during start');
        return;
      }
      this.wasStarted = true;
      this.logger.info(`[UnenrollInactiveAgentsTask] Started with interval of [${INTERVAL}]`);
      try {
        await taskManager.ensureScheduled({
          id: this.taskId,
          taskType: TYPE,
          scope: SCOPE,
          schedule: {
            interval: INTERVAL
          },
          state: {},
          params: {
            version: VERSION
          }
        });
      } catch (e) {
        this.logger.error(`Error scheduling task UnenrollInactiveAgentsTask, error: ${e.message}`, e);
      }
    });
    (0, _defineProperty2.default)(this, "runTask", async (taskInstance, core) => {
      if (!this.wasStarted) {
        this.logger.debug('[UnenrollInactiveAgentsTask] runTask Aborted. Task not started yet');
        return;
      }
      // Check that this task is current
      if (taskInstance.id !== this.taskId) {
        this.logger.debug(`[UnenrollInactiveAgentsTask] Outdated task version: Got [${taskInstance.id}] from task instance. Current version is [${this.taskId}]`);
        return (0, _task.getDeleteTaskRunResult)();
      }
      this.logger.info(`[runTask()] started`);
      const [coreStart] = await core.getStartServices();
      const esClient = coreStart.elasticsearch.client.asInternalUser;
      const soClient = new _server.SavedObjectsClient(coreStart.savedObjects.createInternalRepository());
      try {
        await this.unenrollInactiveAgents(esClient, soClient);
        this.endRun('success');
      } catch (err) {
        if (err instanceof _elasticsearch.errors.RequestAbortedError) {
          this.logger.warn(`[UnenrollInactiveAgentsTask] request aborted due to timeout: ${err}`);
          this.endRun();
          return;
        }
        this.logger.error(`[UnenrollInactiveAgentsTask] error: ${err}`);
        this.endRun('error');
      }
    });
    const {
      core: _core,
      taskManager: _taskManager,
      logFactory
    } = setupContract;
    this.logger = logFactory.get(this.taskId);
    _taskManager.registerTaskDefinitions({
      [TYPE]: {
        title: TITLE,
        timeout: TIMEOUT,
        createTaskRunner: ({
          taskInstance
        }) => {
          return {
            run: async () => {
              return this.runTask(taskInstance, _core);
            },
            cancel: async () => {
              this.abortController.abort('Task timed out');
            }
          };
        }
      }
    });
  }
  get taskId() {
    return `${TYPE}:${VERSION}`;
  }
  endRun(msg = '') {
    this.logger.info(`[UnenrollInactiveAgentsTask] runTask ended${msg ? ': ' + msg : ''}`);
  }
  async unenrollInactiveAgents(esClient, soClient) {
    this.logger.debug(`[UnenrollInactiveAgentsTask] Fetching agent policies with unenroll_timeout > 0`);
    // find all agent policies that are not managed and having unenroll_timeout > 0
    // limit the search to POLICIES_BATCHSIZE at a time and loop until there are no agent policies left
    const policiesKuery = `${_constants.AGENT_POLICY_SAVED_OBJECT_TYPE}.is_managed: false AND ${_constants.AGENT_POLICY_SAVED_OBJECT_TYPE}.unenroll_timeout > 0`;
    let agentCounter = 0;
    const agentPolicyFetcher = await _services.agentPolicyService.fetchAllAgentPolicies(soClient, {
      kuery: policiesKuery,
      perPage: POLICIES_BATCHSIZE
    });
    for await (const agentPolicyPageResults of agentPolicyFetcher) {
      this.logger.debug(`[UnenrollInactiveAgentsTask] Found "${agentPolicyPageResults.length}" agent policies with unenroll_timeout > 0`);
      if (!agentPolicyPageResults.length) {
        this.endRun('Found no policies to process');
        return;
      }

      // find inactive agents enrolled on above policies
      // limit batch size to UNENROLLMENT_BATCHSIZE to avoid scale issues
      const kuery = `(${_constants.AGENTS_PREFIX}.policy_id:${agentPolicyPageResults.map(policy => `"${policy.id}"`).join(' or ')}) and ${_constants.AGENTS_PREFIX}.status: inactive`;
      const res = await (0, _agents.getAgentsByKuery)(esClient, soClient, {
        kuery,
        showInactive: true,
        page: 1,
        perPage: UNENROLLMENT_BATCHSIZE
      });
      if (!res.agents.length) {
        this.endRun('No inactive agents to unenroll');
        return;
      }
      agentCounter += res.agents.length;
      if (agentCounter >= UNENROLLMENT_BATCHSIZE) {
        this.endRun('Reached the maximum amount of agents to unenroll, exiting.');
        return;
      }
      this.logger.debug(`[UnenrollInactiveAgentsTask] Found "${res.agents.length}" inactive agents to unenroll. Attempting unenrollment`);
      const unenrolledBatch = await (0, _unenroll_action_runner.unenrollBatch)(soClient, esClient, res.agents, {
        revoke: true,
        force: true,
        actionId: `UnenrollInactiveAgentsTask-${(0, _uuid.v4)()}`
      });
      _services.auditLoggingService.writeCustomAuditLog({
        message: `Recurrent unenrollment of ${agentCounter} inactive agents due to unenroll_timeout option set on agent policy. Fleet action [id=${unenrolledBatch.actionId}]`
      });
      this.logger.debug(`[UnenrollInactiveAgentsTask] Executed unenrollment of ${agentCounter} inactive agents with actionId: ${unenrolledBatch.actionId}`);
    }
  }
}
exports.UnenrollInactiveAgentsTask = UnenrollInactiveAgentsTask;