"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.checkFleetServerVersionsForSecretsStorage = checkFleetServerVersionsForSecretsStorage;
exports.getFleetServerPolicies = void 0;
exports.hasFleetServers = hasFleetServers;
exports.hasFleetServersForPolicies = void 0;
var _common = require("@kbn/spaces-plugin/common");
var _gte = _interopRequireDefault(require("semver/functions/gte"));
var _coerce = _interopRequireDefault(require("semver/functions/coerce"));
var _lodash = require("lodash");
var _constants = require("../../../common/constants");
var _constants2 = require("../../constants");
var _agents = require("../agents");
var _package_policy = require("../package_policy");
var _agent_policy = require("../agent_policy");
var _ = require("..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Retrieve all agent policies which has a Fleet Server package policy
 */
const getFleetServerPolicies = async soClient => {
  const fleetServerPackagePolicies = await _package_policy.packagePolicyService.list(soClient, {
    kuery: `${_constants.LEGACY_PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.name:${_constants.FLEET_SERVER_PACKAGE}`,
    spaceId: '*'
  });

  // Extract associated fleet server agent policy IDs
  const fleetServerAgentPolicyIds = fleetServerPackagePolicies.items.flatMap(p => {
    var _p$policy_ids;
    return (_p$policy_ids = p.policy_ids) === null || _p$policy_ids === void 0 ? void 0 : _p$policy_ids.map(id => {
      var _id$spaceId, _p$spaceIds$, _p$spaceIds;
      return (_id$spaceId = {
        id,
        spaceId: (_p$spaceIds$ = (_p$spaceIds = p.spaceIds) === null || _p$spaceIds === void 0 ? void 0 : _p$spaceIds[0]) !== null && _p$spaceIds$ !== void 0 ? _p$spaceIds$ : _common.DEFAULT_SPACE_ID
      }) !== null && _id$spaceId !== void 0 ? _id$spaceId : [];
    });
  });

  // Retrieve associated agent policies
  const fleetServerAgentPolicies = fleetServerAgentPolicyIds.length ? await _agent_policy.agentPolicyService.getByIDs(soClient, (0, _lodash.uniqBy)(fleetServerAgentPolicyIds, p => p.id)) : [];
  return fleetServerAgentPolicies;
};

/**
 * Check if there is at least one agent enrolled into the given agent policies.
 * Assumes that `agentPolicyIds` contains list of Fleet Server agent policies.
 * `activeOnly` flag can be used to filter only active agents.
 */
exports.getFleetServerPolicies = getFleetServerPolicies;
const hasFleetServersForPolicies = async (esClient, soClient, agentPolicies, activeOnly = false) => {
  if (agentPolicies.length > 0) {
    const agentStatusesRes = await (0, _agents.getAgentStatusForAgentPolicy)(esClient, soClient, undefined, agentPolicies.map(({
      id,
      space_ids: spaceIds
    }) => {
      const space = spaceIds !== null && spaceIds !== void 0 && spaceIds[0] && (spaceIds === null || spaceIds === void 0 ? void 0 : spaceIds[0]) !== _common.DEFAULT_SPACE_ID ? `namespaces:"${spaceIds === null || spaceIds === void 0 ? void 0 : spaceIds[0]}"` : `not namespaces:* or namespaces:"${_common.DEFAULT_SPACE_ID}"`;
      return `(policy_id:"${id}" and (${space}))`;
    }).join(' or '));
    return activeOnly ? agentStatusesRes.online > 0 || agentStatusesRes.updating > 0 : agentStatusesRes.all > 0;
  }
  return false;
};

/**
 * Check if at least one fleet server agent exists, regardless of its online status
 */
exports.hasFleetServersForPolicies = hasFleetServersForPolicies;
async function hasFleetServers(esClient, soClient) {
  return await hasFleetServersForPolicies(esClient, soClient, await getFleetServerPolicies(soClient));
}

/**
 * This function checks if all Fleet Server agents are running at least a given version, but with
 * some important caveats related to enabling the secrets storage feature:
 *
 * 1. Any unenrolled agents are ignored if they are running an outdated version
 * 2. Managed agents in an inactive state are ignored if they are running an outdated version.
 */
async function checkFleetServerVersionsForSecretsStorage(esClient, soClient, version) {
  const logger = _.appContextService.getLogger();
  let hasMore = true;
  const policyIds = new Set();
  let page = 1;
  const perPage = 200;
  while (hasMore) {
    const res = await _package_policy.packagePolicyService.list(soClient, {
      page: page++,
      perPage,
      kuery: 'ingest-package-policies.package.name:fleet_server',
      fields: ['policy_ids']
    });
    for (const item of res.items) {
      item.policy_ids.forEach(id => policyIds.add(id));
    }
    if (res.items.length < perPage) {
      hasMore = false;
    }
  }
  if (policyIds.size === 0) {
    return false;
  }
  const kuery = `policy_id:(${Array.from(policyIds).map(id => `"${id}"`).join(' or ')})`;
  const managedAgentPolicies = await _agent_policy.agentPolicyService.getAllManagedAgentPolicies(soClient);
  const fleetServerAgents = await (0, _agents.getAgentsByKuery)(esClient, soClient, {
    showInactive: true,
    perPage: _constants2.SO_SEARCH_LIMIT,
    kuery
  });
  if (fleetServerAgents.agents.length === 0) {
    return false;
  }
  let result = true;
  for (const fleetServerAgent of fleetServerAgents.agents) {
    var _fleetServerAgent$loc, _fleetServerAgent$loc2, _fleetServerAgent$loc3;
    const agentVersion = (_fleetServerAgent$loc = fleetServerAgent.local_metadata) === null || _fleetServerAgent$loc === void 0 ? void 0 : (_fleetServerAgent$loc2 = _fleetServerAgent$loc.elastic) === null || _fleetServerAgent$loc2 === void 0 ? void 0 : (_fleetServerAgent$loc3 = _fleetServerAgent$loc2.agent) === null || _fleetServerAgent$loc3 === void 0 ? void 0 : _fleetServerAgent$loc3.version;
    if (!agentVersion) {
      continue;
    }
    const isNewerVersion = (0, _gte.default)((0, _coerce.default)(agentVersion), version);
    if (!isNewerVersion) {
      const agentStatus = await (0, _agents.getAgentStatusById)(esClient, soClient, fleetServerAgent.id);

      // Any unenrolled Fleet Server agents can be ignored
      if (agentStatus === 'unenrolled' || fleetServerAgent.status === 'unenrolling' || fleetServerAgent.unenrolled_at) {
        logger.debug(`Found unenrolled Fleet Server agent ${fleetServerAgent.id} on version ${agentVersion} when checking for secrets storage compatibility - ignoring`);
        continue;
      }
      const isManagedAgentPolicy = managedAgentPolicies.some(managedPolicy => managedPolicy.id === fleetServerAgent.policy_id);

      // If this is an agent enrolled in a managed policy, and it is no longer active then we ignore it if it's
      // running on an outdated version. This prevents users with offline Elastic Agent on Cloud policies from
      // being stuck when it comes to enabling secrets, as agents can't be unenrolled from managed policies via Fleet UI.
      if (isManagedAgentPolicy && ['offline', 'inactive'].includes(agentStatus) || !fleetServerAgent.active) {
        logger.debug(`Found outdated managed Fleet Server agent ${fleetServerAgent.id} on version ${agentVersion} when checking for secrets storage compatibility - ignoring due to ${agentStatus} status`);
        continue;
      }
      logger.debug(`Found outdated Fleet Server agent ${fleetServerAgent.id} on version ${agentVersion} - secrets won't be enabled until all Fleet Server agents are running at least ${version}`);
      result = false;
      break;
    }
  }
  return result;
}