"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.agentlessAgentService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _https = _interopRequireDefault(require("https"));
var _serverHttpTools = require("@kbn/server-http-tools");
var _axios = _interopRequireDefault(require("axios"));
var _elasticApmNode = _interopRequireDefault(require("elastic-apm-node"));
var _constants = require("../../constants");
var _errors = require("../../errors");
var _app_context = require("../app_context");
var _api_keys = require("../api_keys");
var _fleet_server_host = require("../fleet_server_host");
var _agentless = require("../utils/agentless");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class AgentlessAgentService {
  constructor() {
    (0, _defineProperty2.default)(this, "convertCauseErrorsToString", error => {
      if (error.cause instanceof AggregateError) {
        return error.cause.errors.map(e => e.message);
      }
      return error.cause;
    });
  }
  async createAgentlessAgent(esClient, soClient, agentlessAgentPolicy) {
    var _apm$currentTransacti, _appContextService$ge, _agentlessConfig$api, _agentlessConfig$api$, _agentlessConfig$api2, _agentlessConfig$api3, _agentlessConfig$api4, _agentlessConfig$api5;
    const traceId = (_apm$currentTransacti = _elasticApmNode.default.currentTransaction) === null || _apm$currentTransacti === void 0 ? void 0 : _apm$currentTransacti.traceparent;
    const withRequestIdMessage = message => `${message} [Request Id: ${traceId}]`;
    const errorMetadata = {
      trace: {
        id: traceId
      }
    };
    const logger = _app_context.appContextService.getLogger();
    logger.debug(`[Agentless API] Creating agentless agent ${agentlessAgentPolicy.id}`);
    if (!_agentless.isAgentlessApiEnabled) {
      logger.error('[Agentless API] Creating agentless agent not supported in non-cloud or non-serverless environments');
      throw new _errors.AgentlessAgentCreateError('Agentless agent not supported');
    }
    if (!agentlessAgentPolicy.supports_agentless) {
      logger.error('[Agentless API] Agentless agent policy does not have agentless enabled');
      throw new _errors.AgentlessAgentCreateError('Agentless agent policy does not have agentless enabled');
    }
    const agentlessConfig = (_appContextService$ge = _app_context.appContextService.getConfig()) === null || _appContextService$ge === void 0 ? void 0 : _appContextService$ge.agentless;
    if (!agentlessConfig) {
      logger.error('[Agentless API] Missing agentless configuration', errorMetadata);
      throw new _errors.AgentlessAgentCreateError('missing agentless configuration');
    }
    const policyId = agentlessAgentPolicy.id;
    const {
      fleetUrl,
      fleetToken
    } = await this.getFleetUrlAndTokenForAgentlessAgent(esClient, policyId, soClient);
    logger.debug(`[Agentless API] Creating agentless agent with fleetUrl ${fleetUrl} and fleet_token: [REDACTED]`);
    logger.debug(`[Agentless API] Creating agentless agent with TLS cert: ${agentlessConfig !== null && agentlessConfig !== void 0 && (_agentlessConfig$api = agentlessConfig.api) !== null && _agentlessConfig$api !== void 0 && (_agentlessConfig$api$ = _agentlessConfig$api.tls) !== null && _agentlessConfig$api$ !== void 0 && _agentlessConfig$api$.certificate ? '[REDACTED]' : 'undefined'} and TLS key: ${agentlessConfig !== null && agentlessConfig !== void 0 && (_agentlessConfig$api2 = agentlessConfig.api) !== null && _agentlessConfig$api2 !== void 0 && (_agentlessConfig$api3 = _agentlessConfig$api2.tls) !== null && _agentlessConfig$api3 !== void 0 && _agentlessConfig$api3.key ? '[REDACTED]' : 'undefined'}
      and TLS ca: ${agentlessConfig !== null && agentlessConfig !== void 0 && (_agentlessConfig$api4 = agentlessConfig.api) !== null && _agentlessConfig$api4 !== void 0 && (_agentlessConfig$api5 = _agentlessConfig$api4.tls) !== null && _agentlessConfig$api5 !== void 0 && _agentlessConfig$api5.ca ? '[REDACTED]' : 'undefined'}`);
    const tlsConfig = this.createTlsConfig(agentlessConfig);
    const requestConfig = {
      url: (0, _agentless.prependAgentlessApiBasePathToEndpoint)(agentlessConfig, '/deployments'),
      data: {
        policy_id: policyId,
        fleet_url: fleetUrl,
        fleet_token: fleetToken
      },
      method: 'POST',
      headers: {
        'Content-type': 'application/json',
        'X-Request-ID': traceId
      },
      httpsAgent: new _https.default.Agent({
        rejectUnauthorized: tlsConfig.rejectUnauthorized,
        cert: tlsConfig.certificate,
        key: tlsConfig.key,
        ca: tlsConfig.certificateAuthorities
      })
    };
    const cloudSetup = _app_context.appContextService.getCloud();
    if (!(cloudSetup !== null && cloudSetup !== void 0 && cloudSetup.isServerlessEnabled)) {
      requestConfig.data.stack_version = _app_context.appContextService.getKibanaVersion();
    }
    const requestConfigDebugStatus = this.createRequestConfigDebug(requestConfig);
    logger.debug(`[Agentless API] Creating agentless agent with request config ${requestConfigDebugStatus}`);
    const errorMetadataWithRequestConfig = {
      ...errorMetadata,
      http: {
        request: {
          id: traceId,
          body: requestConfig.data
        }
      }
    };
    const response = await (0, _axios.default)(requestConfig).catch(error => {
      if (!_axios.default.isAxiosError(error)) {
        logger.error(`[Agentless API] Creating agentless failed with an error ${error} ${requestConfigDebugStatus}`, errorMetadataWithRequestConfig);
        throw new _errors.AgentlessAgentCreateError(withRequestIdMessage(error.message));
      }
      const errorLogCodeCause = `${error.code}  ${this.convertCauseErrorsToString(error)}`;
      if (error.response) {
        // The request was made and the server responded with a status code and error data
        logger.error(`[Agentless API] Creating agentless failed because the Agentless API responding with a status code that falls out of the range of 2xx: ${JSON.stringify(error.response.status)}} ${JSON.stringify(error.response.data)}} ${requestConfigDebugStatus}`, {
          ...errorMetadataWithRequestConfig,
          http: {
            ...errorMetadataWithRequestConfig.http,
            response: {
              status_code: error.response.status,
              body: error.response.data
            }
          }
        });
        throw new _errors.AgentlessAgentCreateError(withRequestIdMessage(`the Agentless API could not create the agentless agent`));
      } else if (error.request) {
        // The request was made but no response was received
        logger.error(`[Agentless API] Creating agentless agent failed while sending the request to the Agentless API: ${errorLogCodeCause} ${requestConfigDebugStatus}`, errorMetadataWithRequestConfig);
        throw new _errors.AgentlessAgentCreateError(withRequestIdMessage(`no response received from the Agentless API`));
      } else {
        // Something happened in setting up the request that triggered an Error
        logger.error(`[Agentless API] Creating agentless agent failed to be created ${errorLogCodeCause} ${requestConfigDebugStatus}`, errorMetadataWithRequestConfig);
        throw new _errors.AgentlessAgentCreateError(withRequestIdMessage('the Agentless API could not create the agentless agent'));
      }
    });
    logger.debug(`[Agentless API] Created an agentless agent ${response}`);
    return response;
  }
  async deleteAgentlessAgent(agentlessPolicyId) {
    var _appContextService$ge2;
    const logger = _app_context.appContextService.getLogger();
    const agentlessConfig = (_appContextService$ge2 = _app_context.appContextService.getConfig()) === null || _appContextService$ge2 === void 0 ? void 0 : _appContextService$ge2.agentless;
    const tlsConfig = this.createTlsConfig(agentlessConfig);
    const requestConfig = {
      url: (0, _agentless.prependAgentlessApiBasePathToEndpoint)(agentlessConfig, `/deployments/${agentlessPolicyId}`),
      method: 'DELETE',
      headers: {
        'Content-type': 'application/json'
      },
      httpsAgent: new _https.default.Agent({
        rejectUnauthorized: tlsConfig.rejectUnauthorized,
        cert: tlsConfig.certificate,
        key: tlsConfig.key,
        ca: tlsConfig.certificateAuthorities
      })
    };
    const requestConfigDebugStatus = this.createRequestConfigDebug(requestConfig);
    logger.debug(`[Agentless API] Start deleting agentless agent for agent policy ${requestConfigDebugStatus}`);
    if (!_agentless.isAgentlessApiEnabled) {
      logger.error('[Agentless API] Agentless API is not supported. Deleting agentless agent is not supported in non-cloud or non-serverless environments');
    }
    if (!agentlessConfig) {
      logger.error('[Agentless API] kibana.yml is currently missing Agentless API configuration');
    }
    logger.debug(`[Agentless API] Deleting agentless agent with TLS config with certificate`);
    logger.debug(`[Agentless API] Deleting agentless deployment with request config ${requestConfigDebugStatus}`);
    const response = await (0, _axios.default)(requestConfig).catch(error => {
      const errorLogCodeCause = `${error.code} ${this.convertCauseErrorsToString(error)}`;
      if (!_axios.default.isAxiosError(error)) {
        logger.error(`[Agentless API] Deleting agentless deployment failed with an error ${JSON.stringify(error)} ${requestConfigDebugStatus}`);
      }
      if (error.response) {
        logger.error(`[Agentless API] Deleting Agentless deployment Failed Response Error: ${JSON.stringify(error.response.status)}} ${JSON.stringify(error.response.data)}} ${requestConfigDebugStatus} `);
      } else if (error.request) {
        logger.error(`[Agentless API] Deleting agentless deployment failed to receive a response from the Agentless API ${errorLogCodeCause} ${requestConfigDebugStatus}`);
      } else {
        logger.error(`[Agentless API] Deleting agentless deployment failed to delete the request ${errorLogCodeCause} ${requestConfigDebugStatus}`);
      }
    });
    return response;
  }
  createTlsConfig(agentlessConfig) {
    var _agentlessConfig$api6, _agentlessConfig$api7, _agentlessConfig$api8, _agentlessConfig$api9, _agentlessConfig$api10, _agentlessConfig$api11;
    return new _serverHttpTools.SslConfig(_serverHttpTools.sslSchema.validate({
      enabled: true,
      certificate: agentlessConfig === null || agentlessConfig === void 0 ? void 0 : (_agentlessConfig$api6 = agentlessConfig.api) === null || _agentlessConfig$api6 === void 0 ? void 0 : (_agentlessConfig$api7 = _agentlessConfig$api6.tls) === null || _agentlessConfig$api7 === void 0 ? void 0 : _agentlessConfig$api7.certificate,
      key: agentlessConfig === null || agentlessConfig === void 0 ? void 0 : (_agentlessConfig$api8 = agentlessConfig.api) === null || _agentlessConfig$api8 === void 0 ? void 0 : (_agentlessConfig$api9 = _agentlessConfig$api8.tls) === null || _agentlessConfig$api9 === void 0 ? void 0 : _agentlessConfig$api9.key,
      certificateAuthorities: agentlessConfig === null || agentlessConfig === void 0 ? void 0 : (_agentlessConfig$api10 = agentlessConfig.api) === null || _agentlessConfig$api10 === void 0 ? void 0 : (_agentlessConfig$api11 = _agentlessConfig$api10.tls) === null || _agentlessConfig$api11 === void 0 ? void 0 : _agentlessConfig$api11.ca
    }));
  }
  createRequestConfigDebug(requestConfig) {
    return JSON.stringify({
      ...requestConfig,
      data: {
        ...requestConfig.data,
        fleet_token: '[REDACTED]'
      },
      httpsAgent: {
        ...requestConfig.httpsAgent,
        options: {
          ...requestConfig.httpsAgent.options,
          cert: requestConfig.httpsAgent.options.cert ? 'REDACTED' : undefined,
          key: requestConfig.httpsAgent.options.key ? 'REDACTED' : undefined,
          ca: requestConfig.httpsAgent.options.ca ? 'REDACTED' : undefined
        }
      }
    });
  }
  async getFleetUrlAndTokenForAgentlessAgent(esClient, policyId, soClient) {
    const {
      items: enrollmentApiKeys
    } = await (0, _api_keys.listEnrollmentApiKeys)(esClient, {
      perPage: _constants.SO_SEARCH_LIMIT,
      showInactive: true,
      kuery: `policy_id:"${policyId}"`
    });
    const {
      items: fleetHosts
    } = await (0, _fleet_server_host.listFleetServerHosts)(soClient);
    // Tech Debt: change this when we add the internal fleet server config to use the internal fleet server host
    // https://github.com/elastic/security-team/issues/9695
    const defaultFleetHost = fleetHosts.length === 1 ? fleetHosts[0] : fleetHosts.find(host => host.is_default);
    if (!defaultFleetHost) {
      throw new _errors.AgentlessAgentCreateError('missing Fleet server host');
    }
    if (!enrollmentApiKeys.length) {
      throw new _errors.AgentlessAgentCreateError('missing Fleet enrollment token');
    }
    const fleetToken = enrollmentApiKeys[0].api_key;
    const fleetUrl = defaultFleetHost === null || defaultFleetHost === void 0 ? void 0 : defaultFleetHost.host_urls[0];
    return {
      fleetUrl,
      fleetToken
    };
  }
}
const agentlessAgentService = exports.agentlessAgentService = new AgentlessAgentService();