"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getFleetStatusHandler = exports.fleetSetupHandler = void 0;
var _services = require("../../services");
var _setup = require("../../services/setup");
var _fleet_server = require("../../services/fleet_server");
var _errors = require("../../errors");
var _package_verification = require("../../services/epm/packages/package_verification");
var _secrets = require("../../services/secrets");
var _helpers = require("../../services/spaces/helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getFleetStatusHandler = async (context, request, response) => {
  const coreContext = await context.core;
  const esClient = coreContext.elasticsearch.client.asInternalUser;
  const soClient = _services.appContextService.getInternalUserSOClientWithoutSpaceExtension();
  try {
    var _appContextService$ge, _appContextService$ge2, _appContextService$ge3, _appContextService$ge4;
    const isApiKeysEnabled = await _services.appContextService.getSecurity().authc.apiKeys.areAPIKeysEnabled();
    const [hasFleetServersRes, useSecretsStorage, isSpaceAwarenessEnabledRes] = await Promise.all([(0, _fleet_server.hasFleetServers)(esClient, soClient), (0, _secrets.isSecretStorageEnabled)(esClient, soClient), (0, _helpers.isSpaceAwarenessEnabled)()]);
    const isFleetServerMissing = !hasFleetServersRes;
    const isFleetServerStandalone = (_appContextService$ge = (_appContextService$ge2 = _services.appContextService.getConfig()) === null || _appContextService$ge2 === void 0 ? void 0 : (_appContextService$ge3 = _appContextService$ge2.internal) === null || _appContextService$ge3 === void 0 ? void 0 : _appContextService$ge3.fleetServerStandalone) !== null && _appContextService$ge !== void 0 ? _appContextService$ge : false;
    const missingRequirements = [];
    const missingOptionalFeatures = [];
    if (!isApiKeysEnabled) {
      missingRequirements.push('api_keys');
    }
    if (!isFleetServerStandalone && isFleetServerMissing) {
      missingRequirements.push('fleet_server');
    }
    if (!((_appContextService$ge4 = _services.appContextService.getEncryptedSavedObjectsSetup()) !== null && _appContextService$ge4 !== void 0 && _appContextService$ge4.canEncrypt)) {
      missingOptionalFeatures.push('encrypted_saved_object_encryption_key_required');
    }
    const body = {
      isReady: missingRequirements.length === 0,
      missing_requirements: missingRequirements,
      missing_optional_features: missingOptionalFeatures,
      is_secrets_storage_enabled: useSecretsStorage,
      is_space_awareness_enabled: isSpaceAwarenessEnabledRes
    };
    const packageVerificationKeyId = await (0, _package_verification.getGpgKeyIdOrUndefined)();
    if (packageVerificationKeyId) {
      body.package_verification_key_id = packageVerificationKeyId;
    }
    return response.ok({
      body
    });
  } catch (error) {
    return (0, _errors.defaultFleetErrorHandler)({
      error,
      response
    });
  }
};
exports.getFleetStatusHandler = getFleetStatusHandler;
const fleetSetupHandler = async (context, request, response) => {
  try {
    const soClient = (await context.fleet).internalSoClient;
    const esClient = (await context.core).elasticsearch.client.asInternalUser;
    const setupStatus = await (0, _setup.setupFleet)(soClient, esClient);
    const body = {
      ...setupStatus,
      nonFatalErrors: (0, _setup.formatNonFatalErrors)(setupStatus.nonFatalErrors)
    };
    return response.ok({
      body
    });
  } catch (error) {
    return (0, _errors.defaultFleetErrorHandler)({
      error,
      response
    });
  }
};
exports.fleetSetupHandler = fleetSetupHandler;