"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getFleetServerOrAgentPolicies = exports.getEnrollmentSettingsHandler = exports.getDownloadSource = void 0;
var _constants = require("../../../common/constants");
var _errors = require("../../errors");
var _services = require("../../services");
var _fleet_server_host = require("../../services/fleet_server_host");
var _fleet_proxies = require("../../services/fleet_proxies");
var _fleet_server = require("../../services/fleet_server");
var _agent_policies = require("../../services/agent_policies");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getEnrollmentSettingsHandler = async (context, request, response) => {
  var _request$query;
  const agentPolicyId = (_request$query = request.query) === null || _request$query === void 0 ? void 0 : _request$query.agentPolicyId;
  const settingsResponse = {
    fleet_server: {
      policies: [],
      has_active: false
    }
  };
  const coreContext = await context.core;
  const esClient = coreContext.elasticsearch.client.asInternalUser;
  const soClient = coreContext.savedObjects.client;
  try {
    // Get all possible fleet server or scoped normal agent policies
    const {
      fleetServerPolicies,
      scopedAgentPolicy: scopedAgentPolicyResponse
    } = await getFleetServerOrAgentPolicies(soClient, agentPolicyId);
    const scopedAgentPolicy = scopedAgentPolicyResponse || {
      id: undefined,
      name: undefined,
      fleet_server_host_id: undefined,
      download_source_id: undefined,
      data_output_id: undefined
    };

    // Check if there is any active fleet server enrolled into the fleet server policies policies
    if (fleetServerPolicies) {
      settingsResponse.fleet_server.policies = fleetServerPolicies;
      settingsResponse.fleet_server.has_active = await (0, _fleet_server.hasFleetServersForPolicies)(esClient, _services.appContextService.getInternalUserSOClientWithoutSpaceExtension(), fleetServerPolicies, true);
    }

    // Get download source
    // ignore errors if the download source is not found
    try {
      var _scopedAgentPolicy$do;
      settingsResponse.download_source = await getDownloadSource(soClient, (_scopedAgentPolicy$do = scopedAgentPolicy.download_source_id) !== null && _scopedAgentPolicy$do !== void 0 ? _scopedAgentPolicy$do : undefined);
    } catch (e) {
      settingsResponse.download_source = undefined;
    }

    // Get download source proxy
    // ignore errors if the download source proxy is not found
    try {
      var _settingsResponse$dow;
      if ((_settingsResponse$dow = settingsResponse.download_source) !== null && _settingsResponse$dow !== void 0 && _settingsResponse$dow.proxy_id) {
        settingsResponse.download_source_proxy = await (0, _fleet_proxies.getFleetProxy)(soClient, settingsResponse.download_source.proxy_id);
      }
    } catch (e) {
      settingsResponse.download_source_proxy = undefined;
    }

    // Get associated fleet server host, or default one if it doesn't exist
    // `getFleetServerHostsForAgentPolicy` errors if there is no default, so catch it
    try {
      settingsResponse.fleet_server.host = await (0, _fleet_server_host.getFleetServerHostsForAgentPolicy)(soClient, scopedAgentPolicy);
    } catch (e) {
      settingsResponse.fleet_server.host = undefined;
    }

    // If a fleet server host was found, get associated fleet server host proxy if any
    // ignore errors if the proxy is not found
    try {
      var _settingsResponse$fle;
      if ((_settingsResponse$fle = settingsResponse.fleet_server.host) !== null && _settingsResponse$fle !== void 0 && _settingsResponse$fle.proxy_id) {
        settingsResponse.fleet_server.host_proxy = await (0, _fleet_proxies.getFleetProxy)(soClient, settingsResponse.fleet_server.host.proxy_id);
      }
    } catch (e) {
      settingsResponse.fleet_server.host_proxy = undefined;
    }

    // Get associated output and proxy (if any) to use for Fleet Server enrollment
    try {
      if (settingsResponse.fleet_server.policies.length > 0) {
        var _dataOutput$hosts;
        const dataOutput = await (0, _agent_policies.getDataOutputForAgentPolicy)(soClient, scopedAgentPolicy);
        if (dataOutput.type === 'elasticsearch' && (_dataOutput$hosts = dataOutput.hosts) !== null && _dataOutput$hosts !== void 0 && _dataOutput$hosts[0]) {
          settingsResponse.fleet_server.es_output = dataOutput;
          if (dataOutput.proxy_id) {
            settingsResponse.fleet_server.es_output_proxy = await (0, _fleet_proxies.getFleetProxy)(soClient, dataOutput.proxy_id);
          }
        }
      }
    } catch (e) {
      settingsResponse.fleet_server.es_output = undefined;
      settingsResponse.fleet_server.es_output_proxy = undefined;
    }
    return response.ok({
      body: settingsResponse
    });
  } catch (error) {
    return (0, _errors.defaultFleetErrorHandler)({
      error,
      response
    });
  }
};
exports.getEnrollmentSettingsHandler = getEnrollmentSettingsHandler;
const getFleetServerOrAgentPolicies = async (soClient, agentPolicyId) => {
  const mapPolicy = policy => ({
    id: policy.id,
    name: policy.name,
    is_managed: policy.is_managed,
    is_default_fleet_server: policy.is_default_fleet_server,
    has_fleet_server: policy.has_fleet_server,
    fleet_server_host_id: policy.fleet_server_host_id,
    download_source_id: policy.download_source_id,
    space_ids: policy.space_ids,
    data_output_id: policy.data_output_id
  });

  // If an agent policy is specified, return only that policy
  if (agentPolicyId) {
    const agentPolicy = await _services.agentPolicyService.get(soClient, agentPolicyId, true);
    if (agentPolicy) {
      var _agentPolicy$package_;
      if ((_agentPolicy$package_ = agentPolicy.package_policies) !== null && _agentPolicy$package_ !== void 0 && _agentPolicy$package_.find(p => {
        var _p$package;
        return ((_p$package = p.package) === null || _p$package === void 0 ? void 0 : _p$package.name) === _constants.FLEET_SERVER_PACKAGE;
      })) {
        return {
          fleetServerPolicies: [mapPolicy(agentPolicy)],
          scopedAgentPolicy: mapPolicy(agentPolicy)
        };
      } else {
        return {
          scopedAgentPolicy: mapPolicy(agentPolicy)
        };
      }
    }
    return {};
  }

  // If an agent policy is not specified, return all fleet server policies
  const fleetServerPolicies = (await (0, _fleet_server.getFleetServerPolicies)(_services.appContextService.getInternalUserSOClientWithoutSpaceExtension())).map(mapPolicy);
  return {
    fleetServerPolicies
  };
};
exports.getFleetServerOrAgentPolicies = getFleetServerOrAgentPolicies;
const getDownloadSource = async (soClient, downloadSourceId) => {
  const sources = await _services.downloadSourceService.list(soClient);
  const foundSource = downloadSourceId ? sources.items.find(s => s.id === downloadSourceId) : undefined;
  return foundSource || sources.items.find(s => s.is_default);
};
exports.getDownloadSource = getDownloadSource;