"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.alignInputsAndStreams = alignInputsAndStreams;
exports.isSimplifiedCreatePackagePolicyRequest = isSimplifiedCreatePackagePolicyRequest;
exports.removeFieldsFromInputSchema = removeFieldsFromInputSchema;
exports.renameAgentlessAgentPolicy = renameAgentlessAgentPolicy;
var _agentless = require("../../../services/utils/agentless");
var _agentless_policy_helper = require("../../../../common/services/agentless_policy_helper");
var _services = require("../../../services");
var _errors = require("../../../errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function isSimplifiedCreatePackagePolicyRequest(body) {
  // If `inputs` is not defined or if it's a non-array, the request body is using the new simplified API
  if (body.inputs && Array.isArray(body.inputs)) {
    return false;
  }
  return true;
}
function removeFieldsFromInputSchema(packagePolicyInputs) {
  // removed fields not recognized by schema
  return packagePolicyInputs.map(input => {
    const newInput = {
      ...input,
      streams: input.streams.map(stream => {
        const newStream = {
          ...stream
        };
        delete newStream.compiled_stream;
        return newStream;
      })
    };
    delete newInput.compiled_input;
    return newInput;
  });
}

/**
 * If an agentless agent policy is associated with the package policy,
 * it will rename the agentless agent policy of a package policy to keep it in sync with the package policy name.
 */
async function renameAgentlessAgentPolicy(soClient, esClient, packagePolicy, name) {
  if (!(0, _agentless.isAgentlessApiEnabled)()) {
    return;
  }
  // If agentless is enabled for cloud, we need to rename the agent policy
  // tech debt: update this condition when Serverless uses the Agentless API
  // https://github.com/elastic/security-team/issues/9781
  const packagePolicyAgentPolicyId = packagePolicy === null || packagePolicy === void 0 ? void 0 : packagePolicy.policy_id;
  if (!packagePolicyAgentPolicyId) {
    return;
  }
  const agentPolicy = await _services.agentPolicyService.get(soClient, packagePolicyAgentPolicyId);
  if (!agentPolicy) {
    return;
  }
  if (!agentPolicy.supports_agentless) {
    return;
  }
  const agentlessAgentPolicyName = (0, _agentless_policy_helper.getAgentlessAgentPolicyNameFromPackagePolicyName)(name);

  // If the agent policy is already correct, we don't need to update it
  if (agentPolicy.name === agentlessAgentPolicyName) {
    return;
  }
  try {
    await _services.agentPolicyService.update(soClient, esClient, agentPolicy.id, {
      name: agentlessAgentPolicyName
    }, {
      force: true
    });
  } catch (error) {
    throw new _errors.PackagePolicyRequestError(`Failed to update agent policy name for agentless policy: ${error.message}`);
  }
}
function areAllInputStreamDisabled(streams) {
  return streams.reduce((acc, stream, i) => {
    return !stream.enabled && acc;
  }, true);
}

/**
 *
 * Check if one input is enabled but all of its streams are disabled
 * If true, switch input.enabled to false
 */
function alignInputsAndStreams(packagePolicyInputs) {
  return packagePolicyInputs.map(input => {
    if (input.enabled === true && (input === null || input === void 0 ? void 0 : input.streams.length) > 0 && areAllInputStreamDisabled(input.streams)) {
      const newInput = {
        ...input,
        enabled: false
      };
      return newInput;
    }
    return input;
  });
}