"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useAgentless = void 0;
exports.useSetupTechnology = useSetupTechnology;
var _react = require("react");
var _hooks = require("../../../../../hooks");
var _services = require("../../../../../services");
var _generate_new_agent_policy = require("../../../../../../../../common/services/generate_new_agent_policy");
var _types = require("../../../../../types");
var _components = require("../../components");
var _constants = require("../../../../../../../../common/constants");
var _agentless_policy_helper = require("../../../../../../../../common/services/agentless_policy_helper");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useAgentless = () => {
  var _config$agentless;
  const config = (0, _hooks.useConfig)();
  const {
    agentless: agentlessExperimentalFeatureEnabled
  } = _services.ExperimentalFeaturesService.get();
  const {
    cloud
  } = (0, _hooks.useStartServices)();
  const isServerless = !!(cloud !== null && cloud !== void 0 && cloud.isServerlessEnabled);
  const isCloud = !!(cloud !== null && cloud !== void 0 && cloud.isCloudEnabled);
  const isAgentlessApiEnabled = (isCloud || isServerless) && ((_config$agentless = config.agentless) === null || _config$agentless === void 0 ? void 0 : _config$agentless.enabled);
  const isDefaultAgentlessPolicyEnabled = !isAgentlessApiEnabled && isServerless && agentlessExperimentalFeatureEnabled;
  const isAgentlessEnabled = isAgentlessApiEnabled || isDefaultAgentlessPolicyEnabled;
  const isAgentlessAgentPolicy = agentPolicy => {
    if (!agentPolicy) return false;
    return isAgentlessEnabled && ((agentPolicy === null || agentPolicy === void 0 ? void 0 : agentPolicy.id) === _constants.AGENTLESS_POLICY_ID || !!(agentPolicy !== null && agentPolicy !== void 0 && agentPolicy.supports_agentless));
  };

  // When an integration has at least a policy template enabled for agentless
  const isAgentlessIntegration = packageInfo => {
    if (isAgentlessEnabled && (0, _agentless_policy_helper.isAgentlessIntegration)(packageInfo)) {
      return true;
    }
    return false;
  };

  // TODO: remove this check when CSPM implements the above flag and rely only on `isAgentlessIntegration`
  const isAgentlessPackagePolicy = packagePolicy => {
    return isAgentlessEnabled && packagePolicy.policy_ids.includes(_constants.AGENTLESS_POLICY_ID);
  };
  return {
    isAgentlessApiEnabled,
    isDefaultAgentlessPolicyEnabled,
    isAgentlessEnabled,
    isAgentlessAgentPolicy,
    isAgentlessIntegration,
    isAgentlessPackagePolicy
  };
};
exports.useAgentless = useAgentless;
function useSetupTechnology({
  setNewAgentPolicy,
  newAgentPolicy,
  updateAgentPolicies,
  setSelectedPolicyTab,
  packageInfo,
  packagePolicy,
  isEditPage,
  agentPolicies
}) {
  const {
    isAgentlessEnabled,
    isAgentlessApiEnabled,
    isDefaultAgentlessPolicyEnabled
  } = useAgentless();

  // this is a placeholder for the new agent-BASED policy that will be used when the user switches from agentless to agent-based and back
  const newAgentBasedPolicy = (0, _react.useRef)(newAgentPolicy);
  const [selectedSetupTechnology, setSelectedSetupTechnology] = (0, _react.useState)(_types.SetupTechnology.AGENT_BASED);
  const [newAgentlessPolicy, setNewAgentlessPolicy] = (0, _react.useState)(() => {
    const agentless = (0, _generate_new_agent_policy.generateNewAgentPolicyWithDefaults)({
      inactivity_timeout: 3600,
      supports_agentless: true,
      monitoring_enabled: ['logs', 'metrics']
    });
    return agentless;
  });
  (0, _react.useEffect)(() => {
    if (isEditPage && agentPolicies && agentPolicies.some(policy => policy.supports_agentless)) {
      setSelectedSetupTechnology(_types.SetupTechnology.AGENTLESS);
      return;
    }
    if (isAgentlessApiEnabled && selectedSetupTechnology === _types.SetupTechnology.AGENTLESS) {
      const nextNewAgentlessPolicy = {
        ...newAgentlessPolicy,
        name: (0, _agentless_policy_helper.getAgentlessAgentPolicyNameFromPackagePolicyName)(packagePolicy.name)
      };
      if (!newAgentlessPolicy.name || nextNewAgentlessPolicy.name !== newAgentlessPolicy.name) {
        setNewAgentlessPolicy(nextNewAgentlessPolicy);
        setNewAgentPolicy(nextNewAgentlessPolicy);
        updateAgentPolicies([nextNewAgentlessPolicy]);
      }
    }
  }, [isAgentlessApiEnabled, isEditPage, newAgentlessPolicy, packagePolicy.name, selectedSetupTechnology, updateAgentPolicies, setNewAgentPolicy, agentPolicies, setSelectedSetupTechnology]);

  // tech debt: remove this useEffect when Serverless uses the Agentless API
  // https://github.com/elastic/security-team/issues/9781
  (0, _react.useEffect)(() => {
    const fetchAgentlessPolicy = async () => {
      const {
        data,
        error
      } = await (0, _hooks.sendGetOneAgentPolicy)(_constants.AGENTLESS_POLICY_ID);
      const isAgentlessAvailable = !error && data && data.item;
      if (isAgentlessAvailable) {
        setNewAgentlessPolicy(data.item);
      }
    };
    if (isDefaultAgentlessPolicyEnabled) {
      fetchAgentlessPolicy();
    }
  }, [isDefaultAgentlessPolicyEnabled]);
  const handleSetupTechnologyChange = (0, _react.useCallback)(setupTechnology => {
    if (!isAgentlessEnabled || setupTechnology === selectedSetupTechnology) {
      return;
    }
    if (setupTechnology === _types.SetupTechnology.AGENTLESS) {
      if (isAgentlessApiEnabled) {
        setNewAgentPolicy(newAgentlessPolicy);
        setSelectedPolicyTab(_components.SelectedPolicyTab.NEW);
        updateAgentPolicies([newAgentlessPolicy]);
      }
      // tech debt: remove this when Serverless uses the Agentless API
      // https://github.com/elastic/security-team/issues/9781
      if (isDefaultAgentlessPolicyEnabled) {
        setNewAgentPolicy(newAgentlessPolicy);
        updateAgentPolicies([newAgentlessPolicy]);
        setSelectedPolicyTab(_components.SelectedPolicyTab.EXISTING);
      }
    } else if (setupTechnology === _types.SetupTechnology.AGENT_BASED) {
      setNewAgentPolicy({
        ...newAgentBasedPolicy.current,
        supports_agentless: false
      });
      setSelectedPolicyTab(_components.SelectedPolicyTab.NEW);
      updateAgentPolicies([newAgentBasedPolicy.current]);
    }
    setSelectedSetupTechnology(setupTechnology);
  }, [isAgentlessEnabled, selectedSetupTechnology, isAgentlessApiEnabled, isDefaultAgentlessPolicyEnabled, setNewAgentPolicy, newAgentlessPolicy, setSelectedPolicyTab, updateAgentPolicies]);
  return {
    handleSetupTechnologyChange,
    selectedSetupTechnology
  };
}