"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.areFieldAndVariableTypesCompatible = areFieldAndVariableTypesCompatible;
exports.checkFunctionArgMatchesDefinition = checkFunctionArgMatchesDefinition;
exports.correctQuerySyntax = correctQuerySyntax;
exports.countBracketsUnclosed = countBracketsUnclosed;
exports.createMapFromList = createMapFromList;
exports.extractSingularType = extractSingularType;
exports.findFinalWord = findFinalWord;
exports.findPreviousWord = findPreviousWord;
exports.getAllArrayTypes = getAllArrayTypes;
exports.getAllArrayValues = getAllArrayValues;
exports.getAllCommands = getAllCommands;
exports.getAllFunctions = getAllFunctions;
exports.getColumnByName = getColumnByName;
exports.getColumnExists = getColumnExists;
exports.getColumnForASTNode = getColumnForASTNode;
exports.getCommandDefinition = getCommandDefinition;
exports.getCommandOption = getCommandOption;
exports.getExpressionType = getExpressionType;
exports.getFunctionDefinition = getFunctionDefinition;
exports.getLastCharFromTrimmed = getLastCharFromTrimmed;
exports.getParamAtPosition = getParamAtPosition;
exports.getQuotedColumnName = void 0;
exports.getSignaturesWithMatchingArity = getSignaturesWithMatchingArity;
exports.hasWildcard = hasWildcard;
exports.inKnownTimeInterval = inKnownTimeInterval;
exports.isAggFunction = void 0;
exports.isArrayType = isArrayType;
exports.isAssignment = isAssignment;
exports.isAssignmentComplete = isAssignmentComplete;
exports.isColumnItem = isColumnItem;
exports.isComma = isComma;
exports.isFunctionItem = isFunctionItem;
exports.isIncompleteItem = isIncompleteItem;
exports.isInlineCastItem = isInlineCastItem;
exports.isLiteralItem = isLiteralItem;
exports.isMathFunction = isMathFunction;
exports.isOptionItem = isOptionItem;
exports.isParam = void 0;
exports.isRestartingExpression = isRestartingExpression;
exports.isSettingItem = isSettingItem;
exports.isSingleItem = isSingleItem;
exports.isSourceCommand = isSourceCommand;
exports.isSourceItem = isSourceItem;
exports.isSupportedFunction = isSupportedFunction;
exports.isTimeIntervalItem = isTimeIntervalItem;
exports.isValidLiteralOption = isValidLiteralOption;
exports.isVariable = isVariable;
exports.noCaseCompare = void 0;
exports.nonNullable = nonNullable;
exports.pipePrecedesCurrentWord = pipePrecedesCurrentWord;
exports.printFunctionSignature = printFunctionSignature;
exports.shouldBeQuotedSource = shouldBeQuotedSource;
exports.shouldBeQuotedText = shouldBeQuotedText;
exports.sourceExists = sourceExists;
var _aggregation_functions = require("../definitions/generated/aggregation_functions");
var _builtin = require("../definitions/builtin");
var _commands = require("../definitions/commands");
var _scalar_functions = require("../definitions/generated/scalar_functions");
var _grouping = require("../definitions/grouping");
var _test_functions = require("./test_functions");
var _helpers = require("../definitions/helpers");
var _literals = require("../definitions/literals");
var _options = require("../definitions/options");
var _context = require("./context");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function nonNullable(v) {
  return v != null;
}
function isSingleItem(arg) {
  return arg && !Array.isArray(arg);
}
function isSettingItem(arg) {
  return isSingleItem(arg) && arg.type === 'mode';
}
function isFunctionItem(arg) {
  return isSingleItem(arg) && arg.type === 'function';
}
function isOptionItem(arg) {
  return isSingleItem(arg) && arg.type === 'option';
}
function isSourceItem(arg) {
  return isSingleItem(arg) && arg.type === 'source';
}
function isColumnItem(arg) {
  return isSingleItem(arg) && arg.type === 'column';
}
function isLiteralItem(arg) {
  return isSingleItem(arg) && arg.type === 'literal';
}
function isInlineCastItem(arg) {
  return isSingleItem(arg) && arg.type === 'inlineCast';
}
function isTimeIntervalItem(arg) {
  return isSingleItem(arg) && arg.type === 'timeInterval';
}
function isAssignment(arg) {
  return isFunctionItem(arg) && arg.name === '=';
}
function isAssignmentComplete(node) {
  var _removeMarkerArgFromA, _removeMarkerArgFromA2;
  const assignExpression = (_removeMarkerArgFromA = (0, _context.removeMarkerArgFromArgsList)(node)) === null || _removeMarkerArgFromA === void 0 ? void 0 : (_removeMarkerArgFromA2 = _removeMarkerArgFromA.args) === null || _removeMarkerArgFromA2 === void 0 ? void 0 : _removeMarkerArgFromA2[1];
  return Boolean(assignExpression && Array.isArray(assignExpression) && assignExpression.length);
}
function isIncompleteItem(arg) {
  return !arg || !Array.isArray(arg) && arg.incomplete;
}
function isMathFunction(query, offset) {
  const queryTrimmed = query.trimEnd();
  // try to get the full operation token (e.g. "+", "in", "like", etc...) but it requires the token
  // to be spaced out from a field/function (e.g. "field + ") so it is subject to issues
  const [opString] = queryTrimmed.split(' ').reverse();
  // compare last char for all math functions
  // limit only to 2 chars operators
  const fns = _builtin.builtinFunctions.filter(({
    name
  }) => name.length < 3).map(({
    name
  }) => name);
  const tokenMatch = fns.some(op => opString === op);
  // there's a match, that's good
  if (tokenMatch) {
    return true;
  }
  // either there's no match or it is the case where field/function and op are not spaced out
  // e.g "field+" or "fn()+"
  // so try to extract the last char and compare it with the single char math functions
  const singleCharFns = fns.filter(name => name.length === 1);
  return singleCharFns.some(c => c === opString[opString.length - 1]);
}
function isComma(char) {
  return char === ',';
}
function isSourceCommand({
  label
}) {
  return ['FROM', 'ROW', 'SHOW', 'METRICS'].includes(label);
}
let fnLookups;
let commandLookups;
function buildFunctionLookup() {
  // we always refresh if we have test functions
  if (!fnLookups || (0, _test_functions.getTestFunctions)().length) {
    fnLookups = _builtin.builtinFunctions.concat(_scalar_functions.scalarFunctionDefinitions, _aggregation_functions.aggregationFunctionDefinitions, _grouping.groupingFunctionDefinitions, (0, _test_functions.getTestFunctions)()).reduce((memo, def) => {
      memo.set(def.name, def);
      if (def.alias) {
        for (const alias of def.alias) {
          memo.set(alias, def);
        }
      }
      return memo;
    }, new Map());
  }
  return fnLookups;
}
function isSupportedFunction(name, parentCommand, option) {
  var _fn$supportedOptions;
  if (!parentCommand) {
    return {
      supported: false,
      reason: 'missingCommand'
    };
  }
  const fn = buildFunctionLookup().get(name);
  const isSupported = Boolean(option == null ? fn === null || fn === void 0 ? void 0 : fn.supportedCommands.includes(parentCommand) : fn === null || fn === void 0 ? void 0 : (_fn$supportedOptions = fn.supportedOptions) === null || _fn$supportedOptions === void 0 ? void 0 : _fn$supportedOptions.includes(option));
  return {
    supported: isSupported,
    reason: isSupported ? undefined : fn ? 'unsupportedFunction' : 'unknownFunction'
  };
}
function getAllFunctions(options) {
  const fns = buildFunctionLookup();
  if (!(options !== null && options !== void 0 && options.type)) {
    return Array.from(fns.values());
  }
  const types = new Set(Array.isArray(options.type) ? options.type : [options.type]);
  return Array.from(fns.values()).filter(fn => types.has(fn.type));
}
function getFunctionDefinition(name) {
  return buildFunctionLookup().get(name.toLowerCase());
}
const unwrapStringLiteralQuotes = value => value.slice(1, -1);
function buildCommandLookup() {
  if (!commandLookups) {
    commandLookups = _commands.commandDefinitions.reduce((memo, def) => {
      memo.set(def.name, def);
      if (def.alias) {
        memo.set(def.alias, def);
      }
      return memo;
    }, new Map());
  }
  return commandLookups;
}
function getCommandDefinition(name) {
  return buildCommandLookup().get(name.toLowerCase());
}
function getAllCommands() {
  return Array.from(buildCommandLookup().values());
}
function getCommandOption(optionName) {
  return [_options.byOption, _options.metadataOption, _options.asOption, _options.onOption, _options.withOption, _options.appendSeparatorOption].find(({
    name
  }) => name === optionName);
}
function doesLiteralMatchParameterType(argType, item) {
  if (item.literalType === argType) {
    return true;
  }
  if (item.literalType === 'null') {
    // all parameters accept null, but this is not yet reflected
    // in our function definitions so we let it through here
    return true;
  }

  // some parameters accept string literals because of ES auto-casting
  if (item.literalType === 'keyword' && (argType === 'date' || argType === 'date_period' || argType === 'version' || argType === 'ip' || argType === 'boolean')) {
    return true;
  }
  return false;
}

/**
 * This function returns the variable or field matching a column
 */
function getColumnForASTNode(column, {
  fields,
  variables
}) {
  return getColumnByName(column.parts.join('.'), {
    fields,
    variables
  });
}

/**
 * This function returns the variable or field matching a column
 */
function getColumnByName(columnName, {
  fields,
  variables
}) {
  var _variables$get;
  // TODO this doesn't cover all escaping scenarios... the best thing to do would be
  // to use the AST column node parts array, but in some cases the AST node isn't available.
  if (columnName.startsWith(_constants.SINGLE_BACKTICK) && columnName.endsWith(_constants.SINGLE_BACKTICK)) {
    columnName = columnName.slice(1, -1).replace(_constants.DOUBLE_TICKS_REGEX, _constants.SINGLE_BACKTICK);
  }
  return fields.get(columnName) || ((_variables$get = variables.get(columnName)) === null || _variables$get === void 0 ? void 0 : _variables$get[0]);
}
const ARRAY_REGEXP = /\[\]$/;
function isArrayType(type) {
  return ARRAY_REGEXP.test(type);
}
const arrayToSingularMap = new Map([['double[]', 'double'], ['unsigned_long[]', 'unsigned_long'], ['long[]', 'long'], ['integer[]', 'integer'], ['counter_integer[]', 'counter_integer'], ['counter_long[]', 'counter_long'], ['counter_double[]', 'counter_double'], ['keyword[]', 'keyword'], ['text[]', 'text'], ['date[]', 'date'], ['date_period[]', 'date_period'], ['boolean[]', 'boolean'], ['any[]', 'any']]);

/**
 * Given an array type for example `string[]` it will return `string`
 */
function extractSingularType(type) {
  return isArrayType(type) ? arrayToSingularMap.get(type) : type;
}
function createMapFromList(arr) {
  const arrMap = new Map();
  for (const item of arr) {
    arrMap.set(item.name, item);
  }
  return arrMap;
}
function areFieldAndVariableTypesCompatible(fieldType, variableType) {
  if (fieldType == null) {
    return false;
  }
  return fieldType === variableType;
}
function printFunctionSignature(arg) {
  const fnDef = getFunctionDefinition(arg.name);
  if (fnDef) {
    const signature = (0, _helpers.getFunctionSignatures)({
      ...fnDef,
      signatures: [{
        ...(fnDef === null || fnDef === void 0 ? void 0 : fnDef.signatures[0]),
        params: arg.args.map(innerArg => Array.isArray(innerArg) ? {
          name: `InnerArgument[]`,
          type: 'any'
        } :
        // this cast isn't actually correct, but we're abusing the
        // getFunctionSignatures API anyways
        {
          name: innerArg.text,
          type: innerArg.type
        }),
        // this cast isn't actually correct, but we're abusing the
        // getFunctionSignatures API anyways
        returnType: ''
      }]
    }, {
      withTypes: false,
      capitalize: true
    });
    return signature[0].declaration;
  }
  return '';
}
function getAllArrayValues(arg) {
  const values = [];
  if (Array.isArray(arg)) {
    for (const subArg of arg) {
      if (Array.isArray(subArg)) {
        break;
      }
      if (subArg.type === 'literal') {
        values.push(String(subArg.value));
      }
      if (isColumnItem(subArg) || isTimeIntervalItem(subArg)) {
        values.push(subArg.name);
      }
      if (subArg.type === 'function') {
        const signature = printFunctionSignature(subArg);
        if (signature) {
          values.push(signature);
        }
      }
    }
  }
  return values;
}
function getAllArrayTypes(arg, parentCommand, references) {
  const types = [];
  if (Array.isArray(arg)) {
    for (const subArg of arg) {
      if (Array.isArray(subArg)) {
        break;
      }
      if (subArg.type === 'literal') {
        types.push(subArg.literalType);
      }
      if (subArg.type === 'column') {
        const hit = getColumnForASTNode(subArg, references);
        types.push((hit === null || hit === void 0 ? void 0 : hit.type) || 'unsupported');
      }
      if (subArg.type === 'timeInterval') {
        types.push('time_literal');
      }
      if (subArg.type === 'function') {
        if (isSupportedFunction(subArg.name, parentCommand).supported) {
          const fnDef = buildFunctionLookup().get(subArg.name);
          types.push(fnDef.signatures[0].returnType);
        }
      }
    }
  }
  return types;
}
function inKnownTimeInterval(item) {
  return _literals.timeUnits.some(unit => unit === item.unit.toLowerCase());
}

/**
 * Checks if this argument is one of the possible options
 * if they are defined on the arg definition.
 *
 * TODO - Consider merging with isEqualType to create a unified arg validation function
 */
function isValidLiteralOption(arg, argDef) {
  return arg.literalType === 'keyword' && argDef.acceptedValues && !argDef.acceptedValues.map(option => option.toLowerCase()).includes(unwrapStringLiteralQuotes(arg.value).toLowerCase());
}

/**
 * Checks if an AST function argument is of the correct type
 * given the definition.
 */
function checkFunctionArgMatchesDefinition(arg, parameterDefinition, references, parentCommand) {
  const argType = parameterDefinition.type;
  if (argType === 'any' || isParam(arg)) {
    return true;
  }
  if (arg.type === 'literal') {
    const matched = doesLiteralMatchParameterType(argType, arg);
    return matched;
  }
  if (arg.type === 'function') {
    if (isSupportedFunction(arg.name, parentCommand).supported) {
      const fnDef = buildFunctionLookup().get(arg.name);
      return fnDef.signatures.some(signature => signature.returnType === 'unknown' || argType === signature.returnType);
    }
  }
  if (arg.type === 'timeInterval') {
    return argType === 'time_literal' && inKnownTimeInterval(arg);
  }
  if (arg.type === 'column') {
    const hit = getColumnForASTNode(arg, references);
    const validHit = hit;
    if (!validHit) {
      return false;
    }
    const wrappedTypes = Array.isArray(validHit.type) ? validHit.type : [validHit.type];
    return wrappedTypes.some(ct => ct === argType || ct === 'null');
  }
  if (arg.type === 'inlineCast') {
    const lowerArgType = argType === null || argType === void 0 ? void 0 : argType.toLowerCase();
    const castedType = getExpressionType(arg);
    return castedType === lowerArgType;
  }
}
function fuzzySearch(fuzzyName, resources) {
  const wildCardPosition = getWildcardPosition(fuzzyName);
  if (wildCardPosition !== 'none') {
    const matcher = getMatcher(fuzzyName, wildCardPosition);
    for (const resourceName of resources) {
      if (matcher(resourceName)) {
        return true;
      }
    }
  }
}
function getMatcher(name, position) {
  if (position === 'start') {
    const prefix = name.substring(1);
    return resource => resource.endsWith(prefix);
  }
  if (position === 'end') {
    const prefix = name.substring(0, name.length - 1);
    return resource => resource.startsWith(prefix);
  }
  if (position === 'multiple-within') {
    // make sure to remove the * at the beginning of the name if present
    const safeName = name.startsWith('*') ? name.slice(1) : name;
    // replace 2 ore more consecutive wildcards with a single one
    const setOfChars = safeName.replace(/\*{2+}/g, '*').split('*');
    return resource => {
      let index = -1;
      return setOfChars.every(char => {
        index = resource.indexOf(char, index + 1);
        return index !== -1;
      });
    };
  }
  const [prefix, postFix] = name.split('*');
  return resource => resource.startsWith(prefix) && resource.endsWith(postFix);
}
function getWildcardPosition(name) {
  if (!hasWildcard(name)) {
    return 'none';
  }
  const wildCardCount = name.match(/\*/g).length;
  if (wildCardCount > 1) {
    return 'multiple-within';
  }
  if (name.startsWith('*')) {
    return 'start';
  }
  if (name.endsWith('*')) {
    return 'end';
  }
  return 'middle';
}
function hasWildcard(name) {
  return /\*/.test(name);
}
function isVariable(column) {
  return Boolean(column && 'location' in column);
}

/**
 * This returns the name with any quotes that were present.
 *
 * E.g. "`bytes`" will be "`bytes`"
 *
 * @param column
 * @returns
 */
const getQuotedColumnName = column => column.quoted ? column.text : column.name;

/**
 * TODO - consider calling lookupColumn under the hood of this function. Seems like they should really do the same thing.
 */
exports.getQuotedColumnName = getQuotedColumnName;
function getColumnExists(column, {
  fields,
  variables
}) {
  const columnName = column.parts.join('.');
  if (fields.has(columnName) || variables.has(columnName)) {
    return true;
  }

  // TODO — I don't see this fuzzy searching in lookupColumn... should it be there?
  if (Boolean(fuzzySearch(columnName, fields.keys()) || fuzzySearch(columnName, variables.keys()))) {
    return true;
  }
  return false;
}
function sourceExists(index, sources) {
  if (sources.has(index) || index.startsWith('-')) {
    return true;
  }
  return Boolean(fuzzySearch(index, sources.keys()));
}

/**
 * Works backward from the cursor position to determine if
 * the final character of the previous word matches the given character.
 */
function characterPrecedesCurrentWord(text, char) {
  let inCurrentWord = true;
  for (let i = text.length - 1; i >= 0; i--) {
    if (inCurrentWord && /\s/.test(text[i])) {
      inCurrentWord = false;
    }
    if (!inCurrentWord && !/\s/.test(text[i])) {
      return text[i] === char;
    }
  }
}
function pipePrecedesCurrentWord(text) {
  return characterPrecedesCurrentWord(text, '|');
}
function getLastCharFromTrimmed(text) {
  return text[text.trimEnd().length - 1];
}

/**
 * Are we after a comma? i.e. STATS fieldA, <here>
 */
function isRestartingExpression(text) {
  return getLastCharFromTrimmed(text) === ',' || characterPrecedesCurrentWord(text, ',');
}
function findPreviousWord(text) {
  const words = text.split(/\s+/);
  return words[words.length - 2];
}

/**
 * Returns the word at the end of the text if there is one.
 * @param text
 * @returns
 */
function findFinalWord(text) {
  const words = text.split(/\s+/);
  return words[words.length - 1];
}
function shouldBeQuotedSource(text) {
  // Based on lexer `fragment UNQUOTED_SOURCE_PART`
  return /[:"=|,[\]\/ \t\r\n]/.test(text);
}
function shouldBeQuotedText(text, {
  dashSupported
} = {}) {
  return dashSupported ? /[^a-zA-Z\d_\.@-]/.test(text) : /[^a-zA-Z\d_\.@]/.test(text);
}
const isAggFunction = arg => {
  var _getFunctionDefinitio;
  return ((_getFunctionDefinitio = getFunctionDefinition(arg.name)) === null || _getFunctionDefinitio === void 0 ? void 0 : _getFunctionDefinitio.type) === 'agg';
};
exports.isAggFunction = isAggFunction;
const isParam = x => !!x && typeof x === 'object' && x.type === 'literal' && x.literalType === 'param';

/**
 * Compares two strings in a case-insensitive manner
 */
exports.isParam = isParam;
const noCaseCompare = (a, b) => a.toLowerCase() === b.toLowerCase();

/**
 * This function count the number of unclosed brackets in order to
 * locally fix the queryString to generate a valid AST
 * A known limitation of this is that is not aware of commas "," or pipes "|"
 * so it is not yet helpful on a multiple commands errors (a workaround it to pass each command here...)
 * @param bracketType
 * @param text
 * @returns
 */
exports.noCaseCompare = noCaseCompare;
function countBracketsUnclosed(bracketType, text) {
  const stack = [];
  const closingBrackets = {
    '(': ')',
    '[': ']',
    '"': '"',
    '"""': '"""'
  };
  for (let i = 0; i < text.length; i++) {
    const substr = text.substring(i, i + bracketType.length);
    if (substr === closingBrackets[bracketType] && stack.length) {
      stack.pop();
    } else if (substr === bracketType) {
      stack.push(bracketType);
    }
  }
  return stack.length;
}

/**
 * This function attempts to correct the syntax of a partial query to make it valid.
 *
 * This is important because a syntactically-invalid query will not generate a good AST.
 *
 * @param _query
 * @param context
 * @returns
 */
function correctQuerySyntax(_query, context) {
  let query = _query;
  // check if all brackets are closed, otherwise close them
  const unclosedRoundBrackets = countBracketsUnclosed('(', query);
  const unclosedSquaredBrackets = countBracketsUnclosed('[', query);
  const unclosedQuotes = countBracketsUnclosed('"', query);
  const unclosedTripleQuotes = countBracketsUnclosed('"""', query);
  // if it's a comma by the user or a forced trigger by a function argument suggestion
  // add a marker to make the expression still valid
  const charThatNeedMarkers = [',', ':'];
  if (context.triggerCharacter && charThatNeedMarkers.includes(context.triggerCharacter) ||
  // monaco.editor.CompletionTriggerKind['Invoke'] === 0
  context.triggerKind === 0 && unclosedRoundBrackets === 0 || context.triggerCharacter === ' ' && isMathFunction(query, query.length) || isComma(query.trimEnd()[query.trimEnd().length - 1])) {
    query += _constants.EDITOR_MARKER;
  }

  // if there are unclosed brackets, close them
  if (unclosedRoundBrackets || unclosedSquaredBrackets || unclosedQuotes) {
    for (const [char, count] of [['"""', unclosedTripleQuotes], ['"', unclosedQuotes], [')', unclosedRoundBrackets], [']', unclosedSquaredBrackets]]) {
      if (count) {
        // inject the closing brackets
        query += Array(count).fill(char).join('');
      }
    }
  }
  return query;
}

/**
 * Gets the signatures of a function that match the number of arguments
 * provided in the AST.
 */
function getSignaturesWithMatchingArity(fnDef, astFunction) {
  return fnDef.signatures.filter(def => {
    if (def.minParams) {
      return astFunction.args.length >= def.minParams;
    }
    return astFunction.args.length >= def.params.filter(({
      optional
    }) => !optional).length && astFunction.args.length <= def.params.length;
  });
}

/**
 * Given a function signature, returns the parameter at the given position.
 *
 * Takes into account variadic functions (minParams), returning the last
 * parameter if the position is greater than the number of parameters.
 *
 * @param signature
 * @param position
 * @returns
 */
function getParamAtPosition({
  params,
  minParams
}, position) {
  return params.length > position ? params[position] : minParams ? params[params.length - 1] : null;
}

/**
 * Determines the type of the expression
 */
function getExpressionType(root, fields, variables) {
  if (!isSingleItem(root)) {
    if (root.length === 0) {
      return 'unknown';
    }
    return getExpressionType(root[0], fields, variables);
  }
  if (isLiteralItem(root) && root.literalType !== 'param') {
    return root.literalType;
  }
  if (isTimeIntervalItem(root)) {
    return 'time_literal';
  }

  // from https://github.com/elastic/elasticsearch/blob/122e7288200ee03e9087c98dff6cebbc94e774aa/docs/reference/esql/functions/kibana/inline_cast.json
  if (isInlineCastItem(root)) {
    switch (root.castType) {
      case 'int':
        return 'integer';
      case 'bool':
        return 'boolean';
      case 'string':
        return 'keyword';
      case 'text':
        return 'keyword';
      case 'datetime':
        return 'date';
      default:
        return root.castType;
    }
  }
  if (isColumnItem(root) && fields && variables) {
    const column = getColumnForASTNode(root, {
      fields,
      variables
    });
    if (!column) {
      return 'unknown';
    }
    return column.type;
  }
  if (root.type === 'list') {
    return getExpressionType(root.values[0], fields, variables);
  }
  if (isFunctionItem(root)) {
    const fnDefinition = getFunctionDefinition(root.name);
    if (!fnDefinition) {
      return 'unknown';
    }

    /**
     * Special case for COUNT(*) because
     * the "*" column doesn't match any
     * of COUNT's function definitions
     */
    if (fnDefinition.name === 'count' && root.args[0] && isColumnItem(root.args[0]) && root.args[0].name === '*') {
      return 'long';
    }
    if (fnDefinition.name === 'case' && root.args.length) {
      /**
       * The CASE function doesn't fit our system of function definitions
       * and needs special handling. This is imperfect, but it's a start because
       * at least we know that the final argument to case will never be a conditional
       * expression, always a result expression.
       *
       * One problem with this is that if a false case is not provided, the return type
       * will be null, which we aren't detecting. But this is ok because we consider
       * variables and fields to be nullable anyways and account for that during validation.
       */
      return getExpressionType(root.args[root.args.length - 1], fields, variables);
    }
    const signaturesWithCorrectArity = getSignaturesWithMatchingArity(fnDefinition, root);
    if (!signaturesWithCorrectArity.length) {
      return 'unknown';
    }
    const argTypes = root.args.map(arg => getExpressionType(arg, fields, variables));

    // When functions are passed null for any argument, they generally return null
    // This is a special case that is not reflected in our function definitions
    if (argTypes.some(argType => argType === 'null')) return 'null';
    const matchingSignature = signaturesWithCorrectArity.find(signature => {
      return argTypes.every((argType, i) => {
        const param = getParamAtPosition(signature, i);
        return param && (param.type === argType || argType === 'keyword' && ['date', 'date_period'].includes(param.type));
      });
    });
    if (!matchingSignature) {
      return 'unknown';
    }
    return matchingSignature.returnType === 'any' ? 'unknown' : matchingSignature.returnType;
  }
  return 'unknown';
}