"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getCompatibleTypesToSuggestNext = getCompatibleTypesToSuggestNext;
exports.getFunctionsToIgnoreForStats = getFunctionsToIgnoreForStats;
exports.getOverlapRange = getOverlapRange;
exports.getQueryForFields = getQueryForFields;
exports.getSourcesFromCommands = getSourcesFromCommands;
exports.getSupportedTypesForBinaryOperators = getSupportedTypesForBinaryOperators;
exports.getValidFunctionSignaturesForPreviousArgs = getValidFunctionSignaturesForPreviousArgs;
exports.getValidSignaturesAndTypesToSuggestNext = getValidSignaturesAndTypesToSuggestNext;
exports.isAggFunctionUsedAlready = isAggFunctionUsedAlready;
exports.isLiteralDateItem = isLiteralDateItem;
exports.removeQuoteForSuggestedSources = removeQuoteForSuggestedSources;
exports.strictlyGetParamAtPosition = strictlyGetParamAtPosition;
var _lodash = require("lodash");
var _helpers = require("../shared/helpers");
var _esql_types = require("../shared/esql_types");
var _factories = require("./factories");
var _constants = require("../shared/constants");
var _autocomplete = require("./autocomplete");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function extractFunctionArgs(args) {
  return args.flatMap(arg => (0, _helpers.isAssignment)(arg) ? arg.args[1] : arg).filter(_helpers.isFunctionItem);
}
function checkContent(fn) {
  const fnDef = (0, _helpers.getFunctionDefinition)(fn.name);
  return !!fnDef && fnDef.type === 'agg' || extractFunctionArgs(fn.args).some(checkContent);
}
function isAggFunctionUsedAlready(command, argIndex) {
  if (argIndex < 0) {
    return false;
  }
  const arg = command.args[argIndex];
  return (0, _helpers.isFunctionItem)(arg) ? checkContent(arg) : false;
}
function getFnContent(fn) {
  return [fn.name].concat(extractFunctionArgs(fn.args).flatMap(getFnContent));
}
function getFunctionsToIgnoreForStats(command, argIndex) {
  if (argIndex < 0) {
    return [];
  }
  const arg = command.args[argIndex];
  return (0, _helpers.isFunctionItem)(arg) ? getFnContent(arg) : [];
}

/**
 * Given a function signature, returns the parameter at the given position, even if it's undefined or null
 *
 * @param {params}
 * @param position
 * @returns
 */
function strictlyGetParamAtPosition({
  params
}, position) {
  return params[position] ? params[position] : null;
}
function getQueryForFields(queryString, commands) {
  // If there is only one source command and it does not require fields, do not
  // fetch fields, hence return an empty string.
  return commands.length === 1 && ['row', 'show'].includes(commands[0].name) ? '' : queryString;
}
function getSourcesFromCommands(commands, sourceType) {
  var _fromCommand$args;
  const fromCommand = commands.find(({
    name
  }) => name === 'from');
  const args = (_fromCommand$args = fromCommand === null || fromCommand === void 0 ? void 0 : fromCommand.args) !== null && _fromCommand$args !== void 0 ? _fromCommand$args : [];
  return args.filter(arg => arg.sourceType === sourceType);
}
function removeQuoteForSuggestedSources(suggestions) {
  return suggestions.map(d => ({
    ...d,
    // "text" -> text
    text: d.text.startsWith('"') && d.text.endsWith('"') ? d.text.slice(1, -1) : d.text
  }));
}
function getSupportedTypesForBinaryOperators(fnDef, previousType) {
  // Retrieve list of all 'right' supported types that match the left hand side of the function
  return fnDef && Array.isArray(fnDef === null || fnDef === void 0 ? void 0 : fnDef.signatures) ? fnDef.signatures.filter(({
    params
  }) => params.find(p => p.name === 'left' && p.type === previousType)).map(({
    params
  }) => params[1].type) : [previousType];
}
function getValidFunctionSignaturesForPreviousArgs(fnDefinition, enrichedArgs, argIndex) {
  // Filter down to signatures that match every params up to the current argIndex
  // e.g. BUCKET(longField, /) => all signatures with first param as long column type
  // or BUCKET(longField, 2, /) => all signatures with (longField, integer, ...)
  const relevantFuncSignatures = fnDefinition.signatures.filter(s => {
    var _s$params;
    return ((_s$params = s.params) === null || _s$params === void 0 ? void 0 : _s$params.length) >= argIndex && s.params.slice(0, argIndex).every(({
      type: dataType
    }, idx) => {
      return dataType === enrichedArgs[idx].dataType || (0, _esql_types.compareTypesWithLiterals)(dataType, enrichedArgs[idx].dataType);
    });
  });
  return relevantFuncSignatures;
}

/**
 * Given a function signature, returns the compatible types to suggest for the next argument
 *
 * @param fnDefinition: the function definition
 * @param enrichedArgs: AST args with enriched esType info to match with function signatures
 * @param argIndex: the index of the argument to suggest for
 * @returns
 */
function getCompatibleTypesToSuggestNext(fnDefinition, enrichedArgs, argIndex) {
  // First, narrow down to valid function signatures based on previous arguments
  const relevantFuncSignatures = getValidFunctionSignaturesForPreviousArgs(fnDefinition, enrichedArgs, argIndex);

  // Then, get the compatible types to suggest for the next argument
  const compatibleTypesToSuggestForArg = (0, _lodash.uniqBy)(relevantFuncSignatures.map(f => f.params[argIndex]).filter(d => d), o => `${o.type}-${o.constantOnly}`);
  return compatibleTypesToSuggestForArg;
}

/**
 * Checks the suggestion text for overlap with the current query.
 *
 * This is useful to determine the range of the existing query that should be
 * replaced if the suggestion is accepted.
 *
 * For example
 * QUERY: FROM source | WHERE field IS NO
 * SUGGESTION: IS NOT NULL
 *
 * The overlap is "IS NO" and the range to replace is "IS NO" in the query.
 *
 * @param query
 * @param suggestionText
 * @returns
 */
function getOverlapRange(query, suggestionText) {
  let overlapLength = 0;

  // Convert both strings to lowercase for case-insensitive comparison
  const lowerQuery = query.toLowerCase();
  const lowerSuggestionText = suggestionText.toLowerCase();
  for (let i = 0; i <= lowerSuggestionText.length; i++) {
    const substr = lowerSuggestionText.substring(0, i);
    if (lowerQuery.endsWith(substr)) {
      overlapLength = i;
    }
  }
  return {
    start: Math.min(query.length - overlapLength + 1, query.length),
    end: query.length
  };
}
function isValidDateString(dateString) {
  if (typeof dateString !== 'string') return false;
  const timestamp = Date.parse(dateString.replace(/\"/g, ''));
  return !isNaN(timestamp);
}

/**
 * Returns true is node is a valid literal that represents a date
 * either a system time parameter or a date string generated by date picker
 * @param dateString
 * @returns
 */
function isLiteralDateItem(nodeArg) {
  return (0, _helpers.isLiteralItem)(nodeArg) && (
  // If text is ?start or ?end, it's a system time parameter
  _factories.TIME_SYSTEM_PARAMS.includes(nodeArg.text) ||
  // Or if it's a string generated by date picker
  isValidDateString(nodeArg.value));
}
function getValidSignaturesAndTypesToSuggestNext(node, references, fnDefinition, fullText, offset) {
  const enrichedArgs = node.args.map(nodeArg => {
    let dataType = (0, _autocomplete.extractTypeFromASTArg)(nodeArg, references);

    // For named system time parameters ?start and ?end, make sure it's compatiable
    if (isLiteralDateItem(nodeArg)) {
      dataType = 'date';
    }
    return {
      ...nodeArg,
      dataType
    };
  });

  // pick the type of the next arg
  const shouldGetNextArgument = node.text.includes(_constants.EDITOR_MARKER);
  let argIndex = Math.max(node.args.length, 0);
  if (!shouldGetNextArgument && argIndex) {
    argIndex -= 1;
  }
  const validSignatures = getValidFunctionSignaturesForPreviousArgs(fnDefinition, enrichedArgs, argIndex);
  // Retrieve unique of types that are compatiable for the current arg
  const typesToSuggestNext = getCompatibleTypesToSuggestNext(fnDefinition, enrichedArgs, argIndex);
  const hasMoreMandatoryArgs = !validSignatures
  // Types available to suggest next after this argument is completed
  .map(signature => strictlyGetParamAtPosition(signature, argIndex + 1))
  // when a param is null, it means param is optional
  // If there's at least one param that is optional, then
  // no need to suggest comma
  .some(p => p === null || (p === null || p === void 0 ? void 0 : p.optional) === true);

  // Whether to prepend comma to suggestion string
  // E.g. if true, "fieldName" -> "fieldName, "
  const alreadyHasComma = fullText ? fullText[offset] === ',' : false;
  const shouldAddComma = hasMoreMandatoryArgs && fnDefinition.type !== 'builtin' && !alreadyHasComma;
  const currentArg = enrichedArgs[argIndex];
  return {
    shouldAddComma,
    typesToSuggestNext,
    validSignatures,
    hasMoreMandatoryArgs,
    enrichedArgs,
    argIndex,
    currentArg
  };
}