"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.extractTypeFromASTArg = extractTypeFromASTArg;
exports.getFieldsByTypeRetriever = getFieldsByTypeRetriever;
exports.suggest = suggest;
exports.suggestForSortCmd = void 0;
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
var _esql_types = require("../shared/esql_types");
var _helpers = require("../shared/helpers");
var _variables = require("../shared/variables");
var _complete_items = require("./complete_items");
var _factories = require("./factories");
var _constants = require("../shared/constants");
var _context = require("../shared/context");
var _resources_helpers = require("../shared/resources_helpers");
var _helper = require("./helper");
var _helper2 = require("./commands/sort/helper");
var _types = require("../definitions/types");
var _options = require("../definitions/options");
var _builtin = require("../definitions/builtin");
var _suggestions2 = require("./recommended_queries/suggestions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function hasSameArgBothSides(assignFn) {
  if (assignFn.name === '=' && (0, _helpers.isColumnItem)(assignFn.args[0]) && assignFn.args[1]) {
    const assignValue = assignFn.args[1];
    if (Array.isArray(assignValue) && (0, _helpers.isColumnItem)(assignValue[0])) {
      return assignFn.args[0].name === assignValue[0].name;
    }
  }
}
function appendEnrichFields(fieldsMap, policyMetadata) {
  if (!policyMetadata) {
    return fieldsMap;
  }
  // @TODO: improve this
  const newMap = new Map(fieldsMap);
  for (const field of policyMetadata.enrichFields) {
    newMap.set(field, {
      name: field,
      type: 'double'
    });
  }
  return newMap;
}
function getFinalSuggestions({
  comma
} = {
  comma: true
}) {
  const finalSuggestions = [_complete_items.pipeCompleteItem];
  if (comma) {
    finalSuggestions.push(_complete_items.commaCompleteItem);
  }
  return finalSuggestions;
}
async function suggest(fullText, offset, context, astProvider, resourceRetriever) {
  const innerText = fullText.substring(0, offset);
  const correctedQuery = (0, _helpers.correctQuerySyntax)(innerText, context);
  const {
    ast
  } = await astProvider(correctedQuery);
  const astContext = (0, _context.getAstContext)(innerText, ast, offset);
  // build the correct query to fetch the list of fields
  const queryForFields = (0, _helper.getQueryForFields)((0, _resources_helpers.buildQueryUntilPreviousCommand)(ast, correctedQuery), ast);
  const {
    getFieldsByType,
    getFieldsMap
  } = getFieldsByTypeRetriever(queryForFields.replace(_constants.EDITOR_MARKER, ''), resourceRetriever);
  const getSources = (0, _resources_helpers.getSourcesHelper)(resourceRetriever);
  const {
    getPolicies,
    getPolicyMetadata
  } = getPolicyRetriever(resourceRetriever);
  if (astContext.type === 'newCommand') {
    // propose main commands here
    // filter source commands if already defined
    const suggestions = _complete_items.commandAutocompleteDefinitions;
    if (!ast.length) {
      // Display the recommended queries if there are no commands (empty state)
      const recommendedQueriesSuggestions = [];
      if (getSources) {
        let fromCommand = '';
        const sources = await getSources();
        const visibleSources = sources.filter(source => !source.hidden);
        if (visibleSources.find(source => source.name.startsWith('logs'))) {
          fromCommand = 'FROM logs*';
        } else fromCommand = `FROM ${visibleSources[0].name}`;
        const {
          getFieldsByType: getFieldsByTypeEmptyState
        } = getFieldsByTypeRetriever(fromCommand, resourceRetriever);
        recommendedQueriesSuggestions.push(...(await (0, _suggestions2.getRecommendedQueriesSuggestions)(getFieldsByTypeEmptyState, fromCommand)));
      }
      const sourceCommandsSuggestions = suggestions.filter(_helpers.isSourceCommand);
      return [...sourceCommandsSuggestions, ...recommendedQueriesSuggestions];
    }
    return suggestions.filter(def => !(0, _helpers.isSourceCommand)(def));
  }
  if (astContext.type === 'expression') {
    // suggest next possible argument, or option
    // otherwise a variable
    return getExpressionSuggestionsByType(innerText, ast, astContext, getSources, getFieldsByType, getFieldsMap, getPolicies, getPolicyMetadata);
  }
  if (astContext.type === 'setting') {
    return getSettingArgsSuggestions(innerText, ast, astContext, getFieldsByType, getFieldsMap, getPolicyMetadata);
  }
  if (astContext.type === 'option') {
    // need this wrap/unwrap thing to make TS happy
    const {
      option,
      ...rest
    } = astContext;
    if (option && (0, _helpers.isOptionItem)(option)) {
      return getOptionArgsSuggestions(innerText, ast, {
        option,
        ...rest
      }, getFieldsByType, getFieldsMap, getPolicyMetadata, resourceRetriever === null || resourceRetriever === void 0 ? void 0 : resourceRetriever.getPreferences);
    }
  }
  if (astContext.type === 'function') {
    return getFunctionArgsSuggestions(innerText, ast, astContext, getFieldsByType, getFieldsMap, getPolicyMetadata, fullText, offset);
  }
  if (astContext.type === 'list') {
    return getListArgsSuggestions(innerText, ast, astContext, getFieldsByType, getFieldsMap, getPolicyMetadata);
  }
  return [];
}
function getFieldsByTypeRetriever(queryString, resourceRetriever) {
  const helpers = (0, _resources_helpers.getFieldsByTypeHelper)(queryString, resourceRetriever);
  return {
    getFieldsByType: async (expectedType = 'any', ignored = [], options) => {
      const fields = await helpers.getFieldsByType(expectedType, ignored);
      return (0, _factories.buildFieldsDefinitionsWithMetadata)(fields, options);
    },
    getFieldsMap: helpers.getFieldsMap
  };
}
function getPolicyRetriever(resourceRetriever) {
  const helpers = (0, _resources_helpers.getPolicyHelper)(resourceRetriever);
  return {
    getPolicies: async () => {
      const policies = await helpers.getPolicies();
      return (0, _factories.buildPoliciesDefinitions)(policies);
    },
    getPolicyMetadata: helpers.getPolicyMetadata
  };
}
function getSourceSuggestions(sources) {
  // hide indexes that start with .
  return (0, _factories.buildSourcesDefinitions)(sources.filter(({
    hidden
  }) => !hidden).map(({
    name,
    dataStreams,
    title,
    type
  }) => {
    return {
      name,
      isIntegration: Boolean(dataStreams && dataStreams.length),
      title,
      type
    };
  }));
}
function findNewVariable(variables) {
  let autoGeneratedVariableCounter = 0;
  let name = `var${autoGeneratedVariableCounter++}`;
  while (variables.has(name)) {
    name = `var${autoGeneratedVariableCounter++}`;
  }
  return name;
}
function workoutBuiltinOptions(nodeArg, references) {
  var _ref, _nodeArg$text;
  const commandsToInclude = ((_ref = (0, _helpers.isSingleItem)(nodeArg) && ((_nodeArg$text = nodeArg.text) === null || _nodeArg$text === void 0 ? void 0 : _nodeArg$text.toLowerCase().trim().endsWith('null'))) !== null && _ref !== void 0 ? _ref : false) ? ['and', 'or'] : undefined;

  // skip assign operator if it's a function or an existing field to avoid promoting shadowing
  return {
    skipAssign: Boolean(!(0, _helpers.isColumnItem)(nodeArg) || (0, _helpers.getColumnForASTNode)(nodeArg, references)),
    commandsToInclude
  };
}
function areCurrentArgsValid(command, node, references) {
  // unfortunately here we need to bake some command-specific logic
  if (command.name === 'stats') {
    if (node) {
      // consider the following expressions not complete yet
      // ... | stats a
      // ... | stats a =
      if ((0, _helpers.isColumnItem)(node) || (0, _helpers.isAssignment)(node) && !(0, _helpers.isAssignmentComplete)(node)) {
        return false;
      }
    }
  }
  if (command.name === 'eval') {
    if (node) {
      if ((0, _helpers.isFunctionItem)(node)) {
        if ((0, _helpers.isAssignment)(node)) {
          return (0, _helpers.isAssignmentComplete)(node);
        } else {
          return isFunctionArgComplete(node, references).complete;
        }
      }
    }
  }
  if (command.name === 'where') {
    if (node) {
      if ((0, _helpers.isColumnItem)(node) || (0, _helpers.isFunctionItem)(node) && !isFunctionArgComplete(node, references).complete) {
        return false;
      } else {
        return extractTypeFromASTArg(node, references) === (0, _helpers.getCommandDefinition)(command.name).signature.params[0].type;
      }
    }
  }
  if (command.name === 'rename') {
    if (node) {
      if ((0, _helpers.isColumnItem)(node)) {
        return true;
      }
    }
  }
  return true;
}
function extractTypeFromASTArg(arg, references) {
  if (Array.isArray(arg)) {
    return extractTypeFromASTArg(arg[0], references);
  }
  if ((0, _helpers.isColumnItem)(arg) || (0, _helpers.isLiteralItem)(arg)) {
    if ((0, _helpers.isLiteralItem)(arg)) {
      return arg.literalType;
    }
    if ((0, _helpers.isColumnItem)(arg)) {
      const hit = (0, _helpers.getColumnForASTNode)(arg, references);
      if (hit) {
        return hit.type;
      }
    }
  }
  if ((0, _helpers.isTimeIntervalItem)(arg)) {
    return arg.type;
  }
  if ((0, _helpers.isFunctionItem)(arg)) {
    const fnDef = (0, _helpers.getFunctionDefinition)(arg.name);
    if (fnDef) {
      // @TODO: improve this to better filter down the correct return type based on existing arguments
      // just mind that this can be highly recursive...
      return fnDef.signatures[0].returnType;
    }
  }
}

// @TODO: refactor this to be shared with validation
function isFunctionArgComplete(arg, references) {
  const fnDefinition = (0, _helpers.getFunctionDefinition)(arg.name);
  if (!fnDefinition) {
    return {
      complete: false
    };
  }
  const cleanedArgs = (0, _context.removeMarkerArgFromArgsList)(arg).args;
  const argLengthCheck = fnDefinition.signatures.some(def => {
    if (def.minParams && cleanedArgs.length >= def.minParams) {
      return true;
    }
    if (cleanedArgs.length === def.params.length) {
      return true;
    }
    return cleanedArgs.length >= def.params.filter(({
      optional
    }) => !optional).length;
  });
  if (!argLengthCheck) {
    return {
      complete: false,
      reason: 'fewArgs'
    };
  }
  if (fnDefinition.name === 'in' && Array.isArray(arg.args[1]) && !arg.args[1].length) {
    return {
      complete: false,
      reason: 'fewArgs'
    };
  }
  const hasCorrectTypes = fnDefinition.signatures.some(def => {
    return arg.args.every((a, index) => {
      return fnDefinition.name.endsWith('null') && def.params[index].type === 'any' || def.params[index].type === extractTypeFromASTArg(a, references);
    });
  });
  if (!hasCorrectTypes) {
    return {
      complete: false,
      reason: 'wrongTypes'
    };
  }
  return {
    complete: true
  };
}
function extractArgMeta(commandOrOption, node) {
  let argIndex = commandOrOption.args.length;
  const prevIndex = Math.max(argIndex - 1, 0);
  const lastArg = (0, _context.removeMarkerArgFromArgsList)(commandOrOption).args[prevIndex];
  if ((0, _helpers.isIncompleteItem)(lastArg)) {
    argIndex = prevIndex;
  }

  // if a node is not specified use the lastArg
  // mind to give priority to node as lastArg might be a function root
  // => "a > b and c == d" gets translated into and( gt(a, b) , eq(c, d) ) => hence "and" is lastArg
  const nodeArg = node || lastArg;
  return {
    argIndex,
    prevIndex,
    lastArg,
    nodeArg
  };
}
async function getExpressionSuggestionsByType(innerText, commands, {
  command,
  option,
  node
}, getSources, getFieldsByType, getFieldsMap, getPolicies, getPolicyMetadata) {
  var _argDef;
  const commandDef = (0, _helpers.getCommandDefinition)(command.name);
  const {
    argIndex,
    prevIndex,
    lastArg,
    nodeArg
  } = extractArgMeta(command, node);

  // TODO - this is a workaround because it was too difficult to handle this case in a generic way :(
  if (commandDef.name === 'from' && node && (0, _helpers.isSourceItem)(node) && /\s/.test(node.name)) {
    // FROM " <suggest>"
    return [];
  }

  // A new expression is considered either
  // * just after a command name => i.e. ... | STATS <here>
  // * or after a comma => i.e. STATS fieldA, <here>
  const isNewExpression = (0, _helpers.isRestartingExpression)(innerText) || argIndex === 0 && (!(0, _helpers.isFunctionItem)(nodeArg) || !(nodeArg !== null && nodeArg !== void 0 && nodeArg.args.length));

  // the not function is a special operator that can be used in different ways,
  // and not all these are mapped within the AST data structure: in particular
  // <COMMAND> <field> NOT <here>
  // is an incomplete statement and it results in a missing AST node, so we need to detect
  // from the query string itself
  const endsWithNot = / not$/i.test(innerText.trimEnd()) && !command.args.some(arg => (0, _helpers.isFunctionItem)(arg) && arg.name === 'not');

  // early exit in case of a missing function
  if ((0, _helpers.isFunctionItem)(lastArg) && !(0, _helpers.getFunctionDefinition)(lastArg.name)) {
    return [];
  }

  // Are options already declared? This is useful to suggest only new ones
  const optionsAlreadyDeclared = command.args.filter(arg => (0, _helpers.isOptionItem)(arg)).map(({
    name
  }) => ({
    name,
    index: commandDef.options.findIndex(({
      name: defName
    }) => defName === name)
  }));
  const optionsAvailable = commandDef.options.filter(({
    name
  }, index) => {
    const optArg = optionsAlreadyDeclared.find(({
      name: optionName
    }) => optionName === name);
    return !optArg && !optionsAlreadyDeclared.length || optArg && index > optArg.index;
  });
  const hasRecommendedQueries = Boolean(commandDef === null || commandDef === void 0 ? void 0 : commandDef.hasRecommendedQueries);
  // get the next definition for the given command
  let argDef = commandDef.signature.params[argIndex];
  // tune it for the variadic case
  if (!argDef) {
    // this is the case of a comma argument
    if (commandDef.signature.multipleParams) {
      if (isNewExpression || (0, _helpers.isAssignment)(lastArg) && !(0, _helpers.isAssignmentComplete)(lastArg)) {
        // i.e. ... | <COMMAND> a, <here>
        // i.e. ... | <COMMAND> a = ..., b = <here>
        argDef = commandDef.signature.params[0];
      }
    }

    // this is the case where there's an argument, but it's of the wrong type
    // i.e. ... | WHERE numberField <here> (WHERE wants a boolean expression!)
    // i.e. ... | STATS numberfield <here> (STATS wants a function expression!)
    if (!isNewExpression && nodeArg && !Array.isArray(nodeArg)) {
      const prevArg = commandDef.signature.params[prevIndex];
      // in some cases we do not want to go back as the command only accepts a literal
      // i.e. LIMIT 5 <suggest> -> that's it, so no argDef should be assigned

      // make an exception for STATS (STATS is the only command who accept a function type as arg)
      if (prevArg && (prevArg.type === 'function' || !Array.isArray(nodeArg) && prevArg.type !== nodeArg.type)) {
        if (!(0, _helpers.isLiteralItem)(nodeArg) || !prevArg.constantOnly) {
          argDef = prevArg;
        }
      }
    }
  }

  // collect all fields + variables to suggest
  const fieldsMap = await (argDef ? getFieldsMap() : new Map());
  const anyVariables = (0, _variables.collectVariables)(commands, fieldsMap, innerText);
  const previousWord = (0, _helpers.findPreviousWord)(innerText);
  // enrich with assignment has some special rules who are handled somewhere else
  const canHaveAssignments = ['eval', 'stats', 'row'].includes(command.name) && !_builtin.comparisonFunctions.map(fn => fn.name).includes(previousWord);
  const references = {
    fields: fieldsMap,
    variables: anyVariables
  };
  if (command.name === 'sort') {
    return await suggestForSortCmd(innerText, getFieldsByType, col => Boolean((0, _helpers.getColumnByName)(col, references)));
  }
  const suggestions = [];

  // When user types and accepts autocomplete suggestion, and cursor is placed at the end of a valid field
  // we should not show irrelevant functions that might have words matching
  const columnWithActiveCursor = commands.find(c => c.name === command.name && command.name === 'eval' && c.args.some(arg => (0, _helpers.isColumnItem)(arg) && arg.name.includes(_constants.EDITOR_MARKER)));
  const shouldShowFunctions = !columnWithActiveCursor;

  // in this flow there's a clear plan here from argument definitions so try to follow it
  if (argDef) {
    if (argDef.type === 'column' || argDef.type === 'any' || argDef.type === 'function') {
      if (isNewExpression && canHaveAssignments) {
        if (endsWithNot) {
          // i.e.
          // ... | ROW field NOT <suggest>
          // ... | EVAL field NOT <suggest>
          // there's not way to know the type of the field here, so suggest anything
          suggestions.push(...(0, _complete_items.getNextTokenForNot)(command.name, option === null || option === void 0 ? void 0 : option.name, 'any'));
        } else {
          // i.e.
          // ... | ROW <suggest>
          // ... | STATS <suggest>
          // ... | STATS ..., <suggest>
          // ... | EVAL <suggest>
          // ... | EVAL ..., <suggest>
          suggestions.push((0, _factories.buildNewVarDefinition)(findNewVariable(anyVariables)));
        }
      }
    }
    // Suggest fields or variables
    if (argDef.type === 'column' || argDef.type === 'any') {
      if ((!nodeArg || isNewExpression) && !endsWithNot) {
        const fieldSuggestions = await getFieldsOrFunctionsSuggestions(argDef.innerTypes || ['any'], command.name, option === null || option === void 0 ? void 0 : option.name, getFieldsByType, {
          // TODO instead of relying on canHaveAssignments and other command name checks
          // we should have a more generic way to determine if a command can have functions.
          // I think it comes down to the definition of 'column' since 'any' should always
          // include functions.
          functions: canHaveAssignments || command.name === 'sort',
          fields: !argDef.constantOnly,
          variables: anyVariables,
          literals: argDef.constantOnly
        }, {
          ignoreColumns: isNewExpression ? command.args.filter(_helpers.isColumnItem).map(({
            name
          }) => name) : []
        });
        const fieldFragmentSuggestions = await handleFragment(innerText, fragment => Boolean((0, _helpers.getColumnByName)(fragment, references)), (_fragment, rangeToReplace) => {
          // COMMAND fie<suggest>
          return fieldSuggestions.map(suggestion => ({
            ...suggestion,
            text: suggestion.text + (['grok', 'dissect'].includes(command.name) ? ' ' : ''),
            command: _factories.TRIGGER_SUGGESTION_COMMAND,
            rangeToReplace
          }));
        }, (fragment, rangeToReplace) => {
          // COMMAND field<suggest>
          if (['grok', 'dissect'].includes(command.name)) {
            return fieldSuggestions.map(suggestion => ({
              ...suggestion,
              text: suggestion.text + ' ',
              command: _factories.TRIGGER_SUGGESTION_COMMAND,
              rangeToReplace
            }));
          }
          const finalSuggestions = [{
            ..._complete_items.pipeCompleteItem,
            text: ' | '
          }];
          if (fieldSuggestions.length > 1)
            // when we fix the editor marker, this should probably be checked against 0 instead of 1
            // this is because the last field in the AST is currently getting removed (because it contains
            // the editor marker) so it is not included in the ignored list which is used to filter out
            // existing fields above.
            finalSuggestions.push({
              ..._complete_items.commaCompleteItem,
              text: ', '
            });
          return finalSuggestions.map(s => ({
            ...s,
            filterText: fragment,
            text: fragment + s.text,
            command: _factories.TRIGGER_SUGGESTION_COMMAND,
            rangeToReplace
          }));
        });
        suggestions.push(...fieldFragmentSuggestions);
      }
    }
    if (argDef.type === 'function' || argDef.type === 'any') {
      if ((0, _helpers.isColumnItem)(nodeArg)) {
        // ... | STATS a <suggest>
        // ... | EVAL a <suggest>
        const nodeArgType = extractTypeFromASTArg(nodeArg, references);
        if ((0, _types.isParameterType)(nodeArgType)) {
          suggestions.push(...(0, _complete_items.getBuiltinCompatibleFunctionDefinition)(command.name, undefined, nodeArgType, undefined, workoutBuiltinOptions(nodeArg, references)));
        } else {
          suggestions.push((0, _complete_items.getAssignmentDefinitionCompletitionItem)());
        }
      }
      if (isNewExpression && !endsWithNot || (0, _helpers.isAssignment)(nodeArg) && !(0, _helpers.isAssignmentComplete)(nodeArg)) {
        // ... | STATS a = <suggest>
        // ... | EVAL a = <suggest>
        // ... | STATS a = ..., <suggest>
        // ... | EVAL a = ..., <suggest>
        // ... | STATS a = ..., b = <suggest>
        // ... | EVAL a = ..., b = <suggest>
        suggestions.push(...(await getFieldsOrFunctionsSuggestions(['any'], command.name, option === null || option === void 0 ? void 0 : option.name, getFieldsByType, {
          functions: shouldShowFunctions,
          fields: false,
          variables: nodeArg ? undefined : anyVariables,
          literals: argDef.constantOnly
        })));
        if (['show', 'meta'].includes(command.name)) {
          suggestions.push(...(0, _complete_items.getBuiltinCompatibleFunctionDefinition)(command.name, undefined, 'any'));
        }
      }
    }
    if (argDef.type === 'any') {
      // ... | EVAL var = field <suggest>
      // ... | EVAL var = fn(field) <suggest>
      // make sure we're still in the same assignment context and there's no comma (newExpression ensures that)
      if (!isNewExpression) {
        if ((0, _helpers.isAssignment)(nodeArg) && (0, _helpers.isAssignmentComplete)(nodeArg)) {
          const [rightArg] = nodeArg.args[1];
          const nodeArgType = extractTypeFromASTArg(rightArg, references);
          suggestions.push(...(0, _complete_items.getBuiltinCompatibleFunctionDefinition)(command.name, undefined, (0, _types.isParameterType)(nodeArgType) ? nodeArgType : 'any', undefined, workoutBuiltinOptions(rightArg, references)));
          if ((0, _esql_types.isNumericType)(nodeArgType) && (0, _helpers.isLiteralItem)(rightArg)) {
            // ... EVAL var = 1 <suggest>
            suggestions.push(...(0, _factories.getCompatibleLiterals)(command.name, ['time_literal_unit']));
          }
          if ((0, _helpers.isFunctionItem)(rightArg)) {
            if (rightArg.args.some(_helpers.isTimeIntervalItem)) {
              const lastFnArg = rightArg.args[rightArg.args.length - 1];
              const lastFnArgType = extractTypeFromASTArg(lastFnArg, references);
              if ((0, _esql_types.isNumericType)(lastFnArgType) && (0, _helpers.isLiteralItem)(lastFnArg))
                // ... EVAL var = 1 year + 2 <suggest>
                suggestions.push(...(0, _factories.getCompatibleLiterals)(command.name, ['time_literal_unit']));
            }
          }
        } else {
          if ((0, _helpers.isFunctionItem)(nodeArg)) {
            if (nodeArg.name === 'not') {
              suggestions.push(...(await getFieldsOrFunctionsSuggestions(['boolean'], command.name, option === null || option === void 0 ? void 0 : option.name, getFieldsByType, {
                functions: true,
                fields: true,
                variables: anyVariables
              })));
            } else {
              const nodeArgType = extractTypeFromASTArg(nodeArg, references);
              suggestions.push(...(await getBuiltinFunctionNextArgument(innerText, command, option, argDef, nodeArg, nodeArgType || 'any', references, getFieldsByType)));
              if (nodeArg.args.some(_helpers.isTimeIntervalItem)) {
                const lastFnArg = nodeArg.args[nodeArg.args.length - 1];
                const lastFnArgType = extractTypeFromASTArg(lastFnArg, references);
                if ((0, _esql_types.isNumericType)(lastFnArgType) && (0, _helpers.isLiteralItem)(lastFnArg))
                  // ... EVAL var = 1 year + 2 <suggest>
                  suggestions.push(...(0, _factories.getCompatibleLiterals)(command.name, ['time_literal_unit']));
              }
            }
          }
        }
      }
    }

    // if the definition includes a list of constants, suggest them
    if (argDef.values) {
      // ... | <COMMAND> ... <suggest enums>
      suggestions.push(...(0, _factories.buildConstantsDefinitions)(argDef.values, undefined, undefined, {
        advanceCursorAndOpenSuggestions: true
      }));
    }
    // If the type is specified try to dig deeper in the definition to suggest the best candidate
    if (['string', 'text', 'keyword', 'boolean', ..._esql_types.ESQL_NUMBER_TYPES].includes(argDef.type) && !argDef.values) {
      // it can be just literal values (i.e. "string")
      if (argDef.constantOnly) {
        // ... | <COMMAND> ... <suggest>
        suggestions.push(...(0, _factories.getCompatibleLiterals)(command.name, [argDef.type], [argDef.name]));
      } else {
        // or it can be anything else as long as it is of the right type and the end (i.e. column or function)
        if (!nodeArg) {
          if (endsWithNot) {
            // i.e.
            // ... | WHERE field NOT <suggest>
            // there's not way to know the type of the field here, so suggest anything
            suggestions.push(...(0, _complete_items.getNextTokenForNot)(command.name, option === null || option === void 0 ? void 0 : option.name, 'any'));
          } else {
            // ... | <COMMAND> <suggest>
            // In this case start suggesting something not strictly based on type
            suggestions.push(...(await getFieldsByType('any', [], {
              advanceCursor: true,
              openSuggestions: true
            })), ...(await getFieldsOrFunctionsSuggestions(['any'], command.name, option === null || option === void 0 ? void 0 : option.name, getFieldsByType, {
              functions: true,
              fields: false,
              variables: anyVariables
            })));
          }
        } else {
          // if something is already present, leverage its type to suggest something in context
          const nodeArgType = extractTypeFromASTArg(nodeArg, references);
          // These cases can happen here, so need to identify each and provide the right suggestion
          // i.e. ... | <COMMAND> field <suggest>
          // i.e. ... | <COMMAND> field + <suggest>
          // i.e. ... | <COMMAND> field >= <suggest>
          // i.e. ... | <COMMAND> field > 0 <suggest>
          // i.e. ... | <COMMAND> field + otherN <suggest>
          // "FROM a | WHERE doubleField IS NOT N"
          if (nodeArgType) {
            if ((0, _helpers.isFunctionItem)(nodeArg)) {
              if (nodeArg.name === 'not') {
                suggestions.push(...(await getFieldsOrFunctionsSuggestions(['boolean'], command.name, option === null || option === void 0 ? void 0 : option.name, getFieldsByType, {
                  functions: true,
                  fields: true,
                  variables: anyVariables
                })));
              } else {
                suggestions.push(...(await getBuiltinFunctionNextArgument(innerText, command, option, argDef, nodeArg, nodeArgType, references, getFieldsByType)));
              }
            } else if ((0, _types.isParameterType)(nodeArgType)) {
              // i.e. ... | <COMMAND> field <suggest>
              suggestions.push(...(0, _complete_items.getBuiltinCompatibleFunctionDefinition)(command.name, undefined, nodeArgType, undefined, workoutBuiltinOptions(nodeArg, references)));
            }
          }
        }
      }
    }
    if (argDef.type === 'source') {
      var _argDef$innerTypes;
      if ((_argDef$innerTypes = argDef.innerTypes) !== null && _argDef$innerTypes !== void 0 && _argDef$innerTypes.includes('policy')) {
        // ... | ENRICH <suggest>
        const policies = await getPolicies();
        const lastWord = (0, _helpers.findFinalWord)(innerText);
        if (lastWord !== '') {
          policies.forEach(suggestion => {
            suggestions.push({
              ...suggestion,
              rangeToReplace: {
                start: innerText.length - lastWord.length + 1,
                end: innerText.length + 1
              }
            });
          });
        }
        suggestions.push(...(policies.length ? policies : [(0, _factories.buildNoPoliciesAvailableDefinition)()]));
      } else {
        const indexes = (0, _helper.getSourcesFromCommands)(commands, 'index');
        const lastIndex = indexes[indexes.length - 1];
        const canRemoveQuote = isNewExpression && innerText.includes('"');
        // Function to add suggestions based on canRemoveQuote
        const addSuggestionsBasedOnQuote = async definitions => {
          suggestions.push(...(canRemoveQuote ? (0, _helper.removeQuoteForSuggestedSources)(definitions) : definitions));
        };
        if (lastIndex && lastIndex.text && lastIndex.text !== _constants.EDITOR_MARKER) {
          const sources = await getSources();
          const recommendedQueriesSuggestions = hasRecommendedQueries ? await (0, _suggestions2.getRecommendedQueriesSuggestions)(getFieldsByType) : [];
          const suggestionsToAdd = await handleFragment(innerText, fragment => (0, _helpers.sourceExists)(fragment, new Set(sources.map(({
            name: sourceName
          }) => sourceName))), (_fragment, rangeToReplace) => {
            return getSourceSuggestions(sources).map(suggestion => ({
              ...suggestion,
              rangeToReplace
            }));
          }, (fragment, rangeToReplace) => {
            const exactMatch = sources.find(({
              name: _name
            }) => _name === fragment);
            if (exactMatch !== null && exactMatch !== void 0 && exactMatch.dataStreams) {
              // this is an integration name, suggest the datastreams
              const definitions = (0, _factories.buildSourcesDefinitions)(exactMatch.dataStreams.map(({
                name
              }) => ({
                name,
                isIntegration: false
              })));
              return canRemoveQuote ? (0, _helper.removeQuoteForSuggestedSources)(definitions) : definitions;
            } else {
              const _suggestions = [{
                ..._complete_items.pipeCompleteItem,
                filterText: fragment,
                text: fragment + ' | ',
                command: _factories.TRIGGER_SUGGESTION_COMMAND,
                rangeToReplace
              }, {
                ..._complete_items.commaCompleteItem,
                filterText: fragment,
                text: fragment + ', ',
                command: _factories.TRIGGER_SUGGESTION_COMMAND,
                rangeToReplace
              }, {
                ...(0, _factories.buildOptionDefinition)(_options.metadataOption),
                filterText: fragment,
                text: fragment + ' METADATA ',
                asSnippet: false,
                // turn this off because $ could be contained within the source name
                rangeToReplace
              }, ...recommendedQueriesSuggestions.map(suggestion => ({
                ...suggestion,
                rangeToReplace,
                filterText: fragment,
                text: fragment + suggestion.text
              }))];
              return _suggestions;
            }
          });
          addSuggestionsBasedOnQuote(suggestionsToAdd);
        } else {
          // FROM <suggest> or no index/text
          await addSuggestionsBasedOnQuote(getSourceSuggestions(await getSources()));
        }
      }
    }
  }
  const nonOptionArgs = command.args.filter(arg => !(0, _helpers.isOptionItem)(arg) && !(0, _helpers.isSettingItem)(arg) && !Array.isArray(arg) && !arg.incomplete);
  // Perform some checks on mandatory arguments
  const mandatoryArgsAlreadyPresent = commandDef.signature.multipleParams && nonOptionArgs.length > 1 || nonOptionArgs.length >= commandDef.signature.params.filter(({
    optional
  }) => !optional).length || ((_argDef = argDef) === null || _argDef === void 0 ? void 0 : _argDef.type) === 'function';

  // check if declared args are fully valid for the given command
  const currentArgsAreValidForCommand = areCurrentArgsValid(command, nodeArg, references);

  // latest suggestions: options and final ones
  if (!isNewExpression && mandatoryArgsAlreadyPresent && currentArgsAreValidForCommand || optionsAlreadyDeclared.length) {
    // suggest some command options
    if (optionsAvailable.length) {
      suggestions.push(...optionsAvailable.map(opt => (0, _factories.buildOptionDefinition)(opt, command.name === 'dissect')));
    }
    if (!optionsAvailable.length || optionsAvailable.every(({
      optional
    }) => optional)) {
      const shouldPushItDown = command.name === 'eval' && !command.args.some(_helpers.isFunctionItem);
      // now suggest pipe or comma
      const finalSuggestions = getFinalSuggestions({
        comma: commandDef.signature.multipleParams && optionsAvailable.length === commandDef.options.length
      }).map(({
        sortText,
        ...rest
      }) => ({
        ...rest,
        sortText: shouldPushItDown ? `Z${sortText}` : sortText
      }));
      suggestions.push(...finalSuggestions);
    }

    // handle recommended queries for from
    if (hasRecommendedQueries) {
      suggestions.push(...(await (0, _suggestions2.getRecommendedQueriesSuggestions)(getFieldsByType)));
    }
  }
  // Due to some logic overlapping functions can be repeated
  // so dedupe here based on text string (it can differ from name)
  return (0, _lodash.uniqBy)(suggestions, suggestion => suggestion.text);
}
async function getBuiltinFunctionNextArgument(queryText, command, option, argDef, nodeArg, nodeArgType, references, getFieldsByType) {
  const suggestions = [];
  const isFnComplete = isFunctionArgComplete(nodeArg, references);
  if (isFnComplete.complete) {
    // i.e. ... | <COMMAND> field > 0 <suggest>
    // i.e. ... | <COMMAND> field + otherN <suggest>
    suggestions.push(...(0, _complete_items.getBuiltinCompatibleFunctionDefinition)(command.name, option === null || option === void 0 ? void 0 : option.name, (0, _types.isParameterType)(nodeArgType) ? nodeArgType : 'any', undefined, workoutBuiltinOptions(nodeArg, references)));
  } else {
    // i.e. ... | <COMMAND> field >= <suggest>
    // i.e. ... | <COMMAND> field + <suggest>
    // i.e. ... | <COMMAND> field and <suggest>

    // Because it's an incomplete function, need to extract the type of the current argument
    // and suggest the next argument based on types

    // pick the last arg and check its type to verify whether is incomplete for the given function
    const cleanedArgs = (0, _context.removeMarkerArgFromArgsList)(nodeArg).args;
    const nestedType = extractTypeFromASTArg(nodeArg.args[cleanedArgs.length - 1], references);
    if (isFnComplete.reason === 'fewArgs') {
      const fnDef = (0, _helpers.getFunctionDefinition)(nodeArg.name);
      if (fnDef !== null && fnDef !== void 0 && fnDef.signatures.every(({
        params
      }) => params.some(({
        type
      }) => (0, _helpers.isArrayType)(type)))) {
        suggestions.push(_complete_items.listCompleteItem);
      } else {
        var _getFunctionDefinitio;
        const finalType = nestedType || nodeArgType || 'any';
        const supportedTypes = (0, _helper.getSupportedTypesForBinaryOperators)(fnDef, finalType);
        suggestions.push(...(await getFieldsOrFunctionsSuggestions(
        // this is a special case with AND/OR
        // <COMMAND> expression AND/OR <suggest>
        // technically another boolean value should be suggested, but it is a better experience
        // to actually suggest a wider set of fields/functions
        finalType === 'boolean' && ((_getFunctionDefinitio = (0, _helpers.getFunctionDefinition)(nodeArg.name)) === null || _getFunctionDefinitio === void 0 ? void 0 : _getFunctionDefinitio.type) === 'builtin' ? ['any'] : supportedTypes, command.name, option === null || option === void 0 ? void 0 : option.name, getFieldsByType, {
          functions: true,
          fields: true,
          variables: references.variables
        })));
      }
    }
    if (isFnComplete.reason === 'wrongTypes') {
      if (nestedType) {
        // suggest something to complete the builtin function
        if (nestedType !== argDef.type && (0, _types.isParameterType)(nestedType) && (0, _types.isReturnType)(argDef.type)) {
          suggestions.push(...(0, _complete_items.getBuiltinCompatibleFunctionDefinition)(command.name, undefined, nestedType, [argDef.type], workoutBuiltinOptions(nodeArg, references)));
        }
      }
    }
  }
  return suggestions.map(s => {
    const overlap = (0, _helper.getOverlapRange)(queryText, s.text);
    const offset = overlap.start === overlap.end ? 1 : 0;
    return {
      ...s,
      rangeToReplace: {
        start: overlap.start + offset,
        end: overlap.end + offset
      }
    };
  });
}
function pushItUpInTheList(suggestions, shouldPromote) {
  if (!shouldPromote) {
    return suggestions;
  }
  return suggestions.map(({
    sortText,
    ...rest
  }) => ({
    ...rest,
    sortText: `1${sortText}`
  }));
}

/**
 * TODO — split this into distinct functions, one for fields, one for functions, one for literals
 */
async function getFieldsOrFunctionsSuggestions(types, commandName, optionName, getFieldsByType, {
  functions,
  fields,
  variables,
  literals = false
}, {
  ignoreFn = [],
  ignoreColumns = []
} = {}) {
  const filteredFieldsByType = pushItUpInTheList(await (fields ? getFieldsByType(types, ignoreColumns, {
    advanceCursor: commandName === 'sort',
    openSuggestions: commandName === 'sort'
  }) : []), functions);
  const filteredVariablesByType = [];
  if (variables) {
    for (const variable of variables.values()) {
      if ((types.includes('any') || types.includes(variable[0].type)) && !ignoreColumns.includes(variable[0].name)) {
        filteredVariablesByType.push(variable[0].name);
      }
    }
    // due to a bug on the ES|QL table side, filter out fields list with underscored variable names (??)
    // avg( numberField ) => avg_numberField_
    const ALPHANUMERIC_REGEXP = /[^a-zA-Z\d]/g;
    if (filteredVariablesByType.length && filteredVariablesByType.some(v => ALPHANUMERIC_REGEXP.test(v))) {
      for (const variable of filteredVariablesByType) {
        const underscoredName = variable.replace(ALPHANUMERIC_REGEXP, '_');
        const index = filteredFieldsByType.findIndex(({
          label
        }) => underscoredName === label || `_${underscoredName}_` === label);
        if (index >= 0) {
          filteredFieldsByType.splice(index);
        }
      }
    }
  }
  // could also be in stats (bucket) but our autocomplete is not great yet
  const displayDateSuggestions = types.includes('date') && ['where', 'eval'].includes(commandName);
  const suggestions = filteredFieldsByType.concat(displayDateSuggestions ? (0, _factories.getDateLiterals)() : [], functions ? (0, _factories.getCompatibleFunctionDefinition)(commandName, optionName, types, ignoreFn) : [], variables ? pushItUpInTheList((0, _factories.buildVariablesDefinitions)(filteredVariablesByType), functions) : [], literals ? (0, _factories.getCompatibleLiterals)(commandName, types) : []);
  return suggestions;
}
const addCommaIf = (condition, text) => condition ? `${text},` : text;
async function getFunctionArgsSuggestions(innerText, commands, {
  command,
  option,
  node
}, getFieldsByType, getFieldsMap, getPolicyMetadata, fullText, offset) {
  const fnDefinition = (0, _helpers.getFunctionDefinition)(node.name);
  // early exit on no hit
  if (!fnDefinition) {
    return [];
  }
  const fieldsMap = await getFieldsMap();
  const anyVariables = (0, _variables.collectVariables)(commands, fieldsMap, innerText);
  const references = {
    fields: fieldsMap,
    variables: anyVariables
  };
  const variablesExcludingCurrentCommandOnes = (0, _variables.excludeVariablesFromCurrentCommand)(commands, command, fieldsMap, innerText);
  const {
    typesToSuggestNext,
    hasMoreMandatoryArgs,
    enrichedArgs,
    argIndex
  } = (0, _helper.getValidSignaturesAndTypesToSuggestNext)(node, references, fnDefinition, fullText, offset);
  const arg = enrichedArgs[argIndex];

  // Whether to prepend comma to suggestion string
  // E.g. if true, "fieldName" -> "fieldName, "
  const isCursorFollowedByComma = fullText ? fullText.slice(offset, fullText.length).trimStart().startsWith(',') : false;
  const canBeBooleanCondition =
  // For `CASE()`, there can be multiple conditions, so keep suggesting fields and functions if possible
  fnDefinition.name === 'case' ||
  // If the type is explicitly a boolean condition
  typesToSuggestNext.some(t => t && t.type === 'boolean' && t.name === 'condition');
  const shouldAddComma = hasMoreMandatoryArgs && fnDefinition.type !== 'builtin' && !isCursorFollowedByComma && !canBeBooleanCondition;
  const shouldAdvanceCursor = hasMoreMandatoryArgs && fnDefinition.type !== 'builtin' && !isCursorFollowedByComma;
  const suggestedConstants = (0, _lodash.uniq)(typesToSuggestNext.map(d => d.literalSuggestions || d.acceptedValues).filter(d => d).flat());
  if (suggestedConstants.length) {
    return (0, _factories.buildValueDefinitions)(suggestedConstants, {
      addComma: shouldAddComma,
      advanceCursorAndOpenSuggestions: hasMoreMandatoryArgs
    });
  }
  const suggestions = [];
  const noArgDefined = !arg;
  const isUnknownColumn = arg && (0, _helpers.isColumnItem)(arg) && !(0, _helpers.getColumnExists)(arg, {
    fields: fieldsMap,
    variables: variablesExcludingCurrentCommandOnes
  });
  if (noArgDefined || isUnknownColumn) {
    // ... | EVAL fn( <suggest>)
    // ... | EVAL fn( field, <suggest>)

    const commandArgIndex = command.args.findIndex(cmdArg => (0, _helpers.isSingleItem)(cmdArg) && cmdArg.location.max >= node.location.max);
    const finalCommandArgIndex = command.name !== 'stats' ? -1 : commandArgIndex < 0 ? Math.max(command.args.length - 1, 0) : commandArgIndex;
    const finalCommandArg = command.args[finalCommandArgIndex];
    const fnToIgnore = [];
    // just ignore the current function
    if (command.name !== 'stats' || (0, _helpers.isOptionItem)(finalCommandArg) && finalCommandArg.name === 'by') {
      fnToIgnore.push(node.name);
    } else {
      fnToIgnore.push(...(0, _helper.getFunctionsToIgnoreForStats)(command, finalCommandArgIndex), ...((0, _helper.isAggFunctionUsedAlready)(command, finalCommandArgIndex) ? (0, _helpers.getAllFunctions)({
        type: 'agg'
      }).map(({
        name
      }) => name) : []));
    }
    // Separate the param definitions into two groups:
    // fields should only be suggested if the param isn't constant-only,
    // and constant suggestions should only be given if it is.
    //
    // TODO - consider incorporating the literalOptions into this
    //
    // TODO — improve this to inherit the constant flag from the outer function
    // (e.g. if func1's first parameter is constant-only, any nested functions should
    // inherit that constraint: func1(func2(shouldBeConstantOnly)))
    //
    const constantOnlyParamDefs = typesToSuggestNext.filter(p => p.constantOnly || /_literal/.test(p.type));
    const getTypesFromParamDefs = paramDefs => {
      return Array.from(new Set(paramDefs.map(({
        type
      }) => type)));
    };

    // Literals
    suggestions.push(...(0, _factories.getCompatibleLiterals)(command.name, getTypesFromParamDefs(constantOnlyParamDefs), undefined, {
      addComma: shouldAddComma,
      advanceCursorAndOpenSuggestions: hasMoreMandatoryArgs
    }));

    // Fields

    suggestions.push(...pushItUpInTheList(await getFieldsByType(
    // For example, in case() where we are expecting a boolean condition
    // we can accept any field types (field1 !== field2)
    canBeBooleanCondition ? ['any'] :
    // @TODO: have a way to better suggest constant only params
    getTypesFromParamDefs(typesToSuggestNext.filter(d => !d.constantOnly)), [], {
      addComma: shouldAddComma,
      advanceCursor: shouldAdvanceCursor,
      openSuggestions: shouldAdvanceCursor
    }), true));

    // Functions
    suggestions.push(...(0, _factories.getCompatibleFunctionDefinition)(command.name, option === null || option === void 0 ? void 0 : option.name, canBeBooleanCondition ? ['any'] : getTypesFromParamDefs(typesToSuggestNext), fnToIgnore).map(suggestion => ({
      ...suggestion,
      text: addCommaIf(shouldAddComma, suggestion.text)
    })));
    // could also be in stats (bucket) but our autocomplete is not great yet
    if (getTypesFromParamDefs(typesToSuggestNext).includes('date') && ['where', 'eval'].includes(command.name) || command.name === 'stats' && typesToSuggestNext.some(t => t && t.type === 'date' && t.constantOnly === true)) suggestions.push(...(0, _factories.getDateLiterals)({
      addComma: shouldAddComma,
      advanceCursorAndOpenSuggestions: hasMoreMandatoryArgs
    }));
  }

  // for eval and row commands try also to complete numeric literals with time intervals where possible
  if (arg) {
    if (command.name !== 'stats') {
      if ((0, _helpers.isLiteralItem)(arg) && (0, _esql_types.isNumericType)(arg.literalType)) {
        // ... | EVAL fn(2 <suggest>)
        suggestions.push(...(0, _factories.getCompatibleLiterals)(command.name, ['time_literal_unit'], undefined, {
          addComma: shouldAddComma,
          advanceCursorAndOpenSuggestions: hasMoreMandatoryArgs
        }));
      }
    }
    // Suggest comparison functions for boolean conditions
    if (canBeBooleanCondition) {
      suggestions.push(..._builtin.comparisonFunctions.map(({
        name,
        description
      }) => ({
        label: name,
        text: name,
        kind: 'Function',
        detail: description
      })));
    }
    if (hasMoreMandatoryArgs) {
      // Suggest a comma if there's another argument for the function
      suggestions.push(_complete_items.commaCompleteItem);
    }
  }

  // For special case of COUNT, suggest * if cursor is in empty spot
  // e.g. count( / ) -> suggest `*`
  if (fnDefinition.name === 'count' && !arg) {
    suggestions.push(_complete_items.allStarConstant);
  }
  return suggestions;
}
async function getListArgsSuggestions(innerText, commands, {
  command,
  node
}, getFieldsByType, getFieldsMaps, getPolicyMetadata) {
  const suggestions = [];
  // node is supposed to be the function who support a list argument (like the "in" operator)
  // so extract the type of the first argument and suggest fields of that type
  if (node && (0, _helpers.isFunctionItem)(node)) {
    const fieldsMap = await getFieldsMaps();
    const anyVariables = (0, _variables.collectVariables)(commands, fieldsMap, innerText);
    // extract the current node from the variables inferred
    anyVariables.forEach((values, key) => {
      if (values.some(v => v.location === node.location)) {
        anyVariables.delete(key);
      }
    });
    const [firstArg] = node.args;
    if ((0, _helpers.isColumnItem)(firstArg)) {
      const argType = extractTypeFromASTArg(firstArg, {
        fields: fieldsMap,
        variables: anyVariables
      });
      if (argType) {
        // do not propose existing columns again
        const otherArgs = node.args.filter(Array.isArray).flat().filter(_helpers.isColumnItem);
        suggestions.push(...(await getFieldsOrFunctionsSuggestions([argType], command.name, undefined, getFieldsByType, {
          functions: true,
          fields: true,
          variables: anyVariables
        }, {
          ignoreColumns: [firstArg.name, ...otherArgs.map(({
            name
          }) => name)]
        })));
      }
    }
  }
  return suggestions;
}
async function getSettingArgsSuggestions(innerText, commands, {
  command,
  node
}, getFieldsByType, getFieldsMaps, getPolicyMetadata) {
  const suggestions = [];
  const settingDefs = (0, _helpers.getCommandDefinition)(command.name).modes;
  if (settingDefs.length) {
    const lastChar = (0, _helpers.getLastCharFromTrimmed)(innerText);
    const matchingSettingDefs = settingDefs.filter(({
      prefix
    }) => lastChar === prefix);
    if (matchingSettingDefs.length) {
      // COMMAND _<here>
      suggestions.push(...matchingSettingDefs.flatMap(_factories.buildSettingDefinitions));
    }
  }
  return suggestions;
}
async function getOptionArgsSuggestions(innerText, commands, {
  command,
  option,
  node
}, getFieldsByType, getFieldsMaps, getPolicyMetadata, getPreferences) {
  let preferences;
  if (getPreferences) {
    preferences = await getPreferences();
  }
  const optionDef = (0, _helpers.getCommandOption)(option.name);
  const {
    nodeArg,
    argIndex,
    lastArg
  } = extractArgMeta(option, node);
  const suggestions = [];
  const isNewExpression = (0, _helpers.isRestartingExpression)(innerText) || option.args.length === 0;
  const fieldsMap = await getFieldsMaps();
  const anyVariables = (0, _variables.collectVariables)(commands, fieldsMap, innerText);
  const references = {
    fields: fieldsMap,
    variables: anyVariables
  };
  if (command.name === 'enrich') {
    if (option.name === 'on') {
      // if it's a new expression, suggest fields to match on
      if (isNewExpression || (0, _helpers.noCaseCompare)((0, _helpers.findPreviousWord)(innerText), 'ON') || option && (0, _helpers.isAssignment)(option.args[0]) && !option.args[1]) {
        const policyName = (0, _helpers.isSourceItem)(command.args[0]) ? command.args[0].name : undefined;
        if (policyName) {
          const policyMetadata = await getPolicyMetadata(policyName);
          if (policyMetadata) {
            suggestions.push(...(0, _factories.buildMatchingFieldsDefinition)(policyMetadata.matchField, Array.from(fieldsMap.keys())));
          }
        }
      } else {
        // propose the with option
        suggestions.push((0, _factories.buildOptionDefinition)((0, _helpers.getCommandOption)('with')), ...getFinalSuggestions({
          comma: false
        }));
      }
    }
    if (option.name === 'with') {
      const policyName = (0, _helpers.isSourceItem)(command.args[0]) ? command.args[0].name : undefined;
      if (policyName) {
        const policyMetadata = await getPolicyMetadata(policyName);
        const anyEnhancedVariables = (0, _variables.collectVariables)(commands, appendEnrichFields(fieldsMap, policyMetadata), innerText);
        if (isNewExpression || (0, _helpers.noCaseCompare)((0, _helpers.findPreviousWord)(innerText), 'WITH')) {
          suggestions.push((0, _factories.buildNewVarDefinition)(findNewVariable(anyEnhancedVariables)));
        }

        // make sure to remove the marker arg from the assign fn
        const assignFn = (0, _helpers.isAssignment)(lastArg) ? (0, _context.removeMarkerArgFromArgsList)(lastArg) : undefined;
        if (policyMetadata) {
          if (isNewExpression || assignFn && !(0, _helpers.isAssignmentComplete)(assignFn)) {
            // ... | ENRICH ... WITH a =
            // ... | ENRICH ... WITH b
            const fieldSuggestions = (0, _factories.buildFieldsDefinitions)(policyMetadata.enrichFields);
            // in this case, we don't want to open the suggestions menu when the field is accepted
            // because we're keeping the suggestions simple here for now. Could always revisit.
            fieldSuggestions.forEach(s => s.command = undefined);

            // attach the replacement range if needed
            const lastWord = (0, _helpers.findFinalWord)(innerText);
            if (lastWord) {
              // ENRICH ... WITH a <suggest>
              const rangeToReplace = {
                start: innerText.length - lastWord.length + 1,
                end: innerText.length + 1
              };
              fieldSuggestions.forEach(s => s.rangeToReplace = rangeToReplace);
            }
            suggestions.push(...fieldSuggestions);
          }
        }
        if (assignFn && hasSameArgBothSides(assignFn) && !isNewExpression && !(0, _helpers.isIncompleteItem)(assignFn)) {
          // ... | ENRICH ... WITH a
          // effectively only assign will apper
          suggestions.push(...pushItUpInTheList((0, _complete_items.getBuiltinCompatibleFunctionDefinition)(command.name, undefined, 'any'), true));
        }
        if (assignFn && ((0, _helpers.isAssignmentComplete)(assignFn) || hasSameArgBothSides(assignFn)) && !isNewExpression) {
          suggestions.push(...getFinalSuggestions({
            comma: true
          }));
        }
      }
    }
  }
  if (command.name === 'rename') {
    if (option.args.length < 2) {
      suggestions.push(...(0, _factories.buildVariablesDefinitions)([findNewVariable(anyVariables)]));
    }
  }
  if (command.name === 'dissect') {
    if (option.args.filter(arg => !((0, _helpers.isSingleItem)(arg) && arg.type === 'unknown')).length < 1 && optionDef) {
      suggestions.push(_complete_items.colonCompleteItem, _complete_items.semiColonCompleteItem);
    }
  }
  if (option.name === 'metadata') {
    const existingFields = new Set(option.args.filter(_helpers.isColumnItem).map(({
      name
    }) => name));
    const filteredMetaFields = _constants.METADATA_FIELDS.filter(name => !existingFields.has(name));
    if (isNewExpression) {
      suggestions.push(...(await handleFragment(innerText, fragment => _constants.METADATA_FIELDS.includes(fragment), (_fragment, rangeToReplace) => (0, _factories.buildFieldsDefinitions)(filteredMetaFields).map(suggestion => ({
        ...suggestion,
        rangeToReplace
      })), (fragment, rangeToReplace) => {
        const _suggestions = [{
          ..._complete_items.pipeCompleteItem,
          text: fragment + ' | ',
          filterText: fragment,
          command: _factories.TRIGGER_SUGGESTION_COMMAND,
          rangeToReplace
        }];
        if (filteredMetaFields.length > 1) {
          _suggestions.push({
            ..._complete_items.commaCompleteItem,
            text: fragment + ', ',
            filterText: fragment,
            command: _factories.TRIGGER_SUGGESTION_COMMAND,
            rangeToReplace
          });
        }
        return _suggestions;
      })));
    } else {
      if (existingFields.size > 0) {
        // METADATA field <suggest>
        if (filteredMetaFields.length > 0) {
          suggestions.push(_complete_items.commaCompleteItem);
        }
        suggestions.push(_complete_items.pipeCompleteItem);
      }
    }
  }
  if (command.name === 'stats') {
    const argDef = optionDef === null || optionDef === void 0 ? void 0 : optionDef.signature.params[argIndex];
    const nodeArgType = extractTypeFromASTArg(nodeArg, references);
    // These cases can happen here, so need to identify each and provide the right suggestion
    // i.e. ... | STATS ... BY field + <suggest>
    // i.e. ... | STATS ... BY field >= <suggest>

    if (nodeArgType) {
      if ((0, _helpers.isFunctionItem)(nodeArg) && !isFunctionArgComplete(nodeArg, references).complete) {
        suggestions.push(...(await getBuiltinFunctionNextArgument(innerText, command, option, {
          type: (argDef === null || argDef === void 0 ? void 0 : argDef.type) || 'unknown'
        }, nodeArg, nodeArgType, {
          fields: references.fields,
          // you can't use a variable defined
          // in the stats command in the by clause
          variables: new Map()
        }, getFieldsByType)));
      }
    }

    // If it's a complete expression then propose some final suggestions
    if (!nodeArgType && option.name === 'by' && option.args.length && !isNewExpression && !(0, _helpers.isAssignment)(lastArg) || (0, _helpers.isAssignment)(lastArg) && (0, _helpers.isAssignmentComplete)(lastArg)) {
      var _optionDef$signature$;
      suggestions.push(...getFinalSuggestions({
        comma: (_optionDef$signature$ = optionDef === null || optionDef === void 0 ? void 0 : optionDef.signature.multipleParams) !== null && _optionDef$signature$ !== void 0 ? _optionDef$signature$ : option.name === 'by'
      }));
    }
  }
  if (optionDef) {
    if (!suggestions.length) {
      const argDefIndex = optionDef.signature.multipleParams ? 0 : Math.max(option.args.length - 1, 0);
      const types = [optionDef.signature.params[argDefIndex].type].filter(_helpers.nonNullable);
      // If it's a complete expression then proposed some final suggestions
      // A complete expression is either a function or a column: <COMMAND> <OPTION> field <here>
      // Or an assignment complete: <COMMAND> <OPTION> field = ... <here>
      if (option.args.length && !isNewExpression && !(0, _helpers.isAssignment)(lastArg) || (0, _helpers.isAssignment)(lastArg) && (0, _helpers.isAssignmentComplete)(lastArg)) {
        suggestions.push(...getFinalSuggestions({
          comma: optionDef.signature.multipleParams
        }));
      } else if (isNewExpression || (0, _helpers.isAssignment)(nodeArg) && !(0, _helpers.isAssignmentComplete)(nodeArg)) {
        suggestions.push(...(await getFieldsByType(types[0] === 'column' ? ['any'] : types, [], {
          advanceCursor: true,
          openSuggestions: true
        })));
        // Checks if cursor is still within function ()
        // by checking if the marker editor/cursor is within an unclosed parenthesis
        const canHaveAssignment = (0, _helpers.countBracketsUnclosed)('(', innerText) === 0;
        if (option.name === 'by') {
          // Add quick snippet for for stats ... by bucket(<>)
          if (command.name === 'stats' && canHaveAssignment) {
            var _preferences;
            suggestions.push({
              label: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.addDateHistogram', {
                defaultMessage: 'Add date histogram'
              }),
              text: (0, _factories.getAddDateHistogramSnippet)((_preferences = preferences) === null || _preferences === void 0 ? void 0 : _preferences.histogramBarTarget),
              asSnippet: true,
              kind: 'Issue',
              detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.addDateHistogramDetail', {
                defaultMessage: 'Add date histogram using bucket()'
              }),
              sortText: '1A',
              command: _factories.TRIGGER_SUGGESTION_COMMAND
            });
          }
          suggestions.push(...(await getFieldsOrFunctionsSuggestions(types[0] === 'column' ? ['any'] : types, command.name, option.name, getFieldsByType, {
            functions: true,
            fields: false
          }, {
            ignoreFn: canHaveAssignment ? [] : ['bucket', 'case']
          })));
        }
        if (command.name === 'stats' && isNewExpression && canHaveAssignment) {
          suggestions.push((0, _factories.buildNewVarDefinition)(findNewVariable(anyVariables)));
        }
      }
    }
  }
  return suggestions;
}

/**
 * This function handles the logic to suggest completions
 * for a given fragment of text in a generic way. A good example is
 * a field name.
 *
 * When typing a field name, there are 2 scenarios
 *
 * 1. field name is incomplete (includes the empty string)
 * KEEP /
 * KEEP fie/
 *
 * 2. field name is complete
 * KEEP field/
 *
 * This function provides a framework for detecting and handling both scenarios in a clean way.
 *
 * @param innerText - the query text before the current cursor position
 * @param isFragmentComplete — return true if the fragment is complete
 * @param getSuggestionsForIncomplete — gets suggestions for an incomplete fragment
 * @param getSuggestionsForComplete - gets suggestions for a complete fragment
 * @returns
 */
function handleFragment(innerText, isFragmentComplete, getSuggestionsForIncomplete, getSuggestionsForComplete) {
  /**
   * @TODO — this string manipulation is crude and can't support all cases
   * Checking for a partial word and computing the replacement range should
   * really be done using the AST node, but we'll have to refactor further upstream
   * to make that available. This is a quick fix to support the most common case.
   */
  const fragment = (0, _helpers.findFinalWord)(innerText);
  if (!fragment) {
    return getSuggestionsForIncomplete('');
  } else {
    const rangeToReplace = {
      start: innerText.length - fragment.length + 1,
      end: innerText.length + 1
    };
    if (isFragmentComplete(fragment)) {
      return getSuggestionsForComplete(fragment, rangeToReplace);
    } else {
      return getSuggestionsForIncomplete(fragment, rangeToReplace);
    }
  }
}
const sortModifierSuggestions = {
  ASC: {
    label: 'ASC',
    text: 'ASC',
    detail: '',
    kind: 'Keyword',
    sortText: '1-ASC',
    command: _factories.TRIGGER_SUGGESTION_COMMAND
  },
  DESC: {
    label: 'DESC',
    text: 'DESC',
    detail: '',
    kind: 'Keyword',
    sortText: '1-DESC',
    command: _factories.TRIGGER_SUGGESTION_COMMAND
  },
  NULLS_FIRST: {
    label: 'NULLS FIRST',
    text: 'NULLS FIRST',
    detail: '',
    kind: 'Keyword',
    sortText: '2-NULLS FIRST',
    command: _factories.TRIGGER_SUGGESTION_COMMAND
  },
  NULLS_LAST: {
    label: 'NULLS LAST',
    text: 'NULLS LAST',
    detail: '',
    kind: 'Keyword',
    sortText: '2-NULLS LAST',
    command: _factories.TRIGGER_SUGGESTION_COMMAND
  }
};
const suggestForSortCmd = async (innerText, getFieldsByType, columnExists) => {
  const prependSpace = s => ({
    ...s,
    text: ' ' + s.text
  });
  const {
    pos,
    nulls
  } = (0, _helper2.getSortPos)(innerText);
  switch (pos) {
    case 'space2':
      {
        return [sortModifierSuggestions.ASC, sortModifierSuggestions.DESC, sortModifierSuggestions.NULLS_FIRST, sortModifierSuggestions.NULLS_LAST, _complete_items.pipeCompleteItem, {
          ..._complete_items.commaCompleteItem,
          text: ', ',
          command: _factories.TRIGGER_SUGGESTION_COMMAND
        }];
      }
    case 'order':
      {
        return handleFragment(innerText, fragment => ['ASC', 'DESC'].some(completeWord => (0, _helpers.noCaseCompare)(completeWord, fragment)), (_fragment, rangeToReplace) => {
          return Object.values(sortModifierSuggestions).map(suggestion => ({
            ...suggestion,
            rangeToReplace
          }));
        }, (fragment, rangeToReplace) => {
          return [{
            ..._complete_items.pipeCompleteItem,
            text: ' | '
          }, {
            ..._complete_items.commaCompleteItem,
            text: ', '
          }, prependSpace(sortModifierSuggestions.NULLS_FIRST), prependSpace(sortModifierSuggestions.NULLS_LAST)].map(suggestion => ({
            ...suggestion,
            filterText: fragment,
            text: fragment + suggestion.text,
            rangeToReplace,
            command: _factories.TRIGGER_SUGGESTION_COMMAND
          }));
        });
      }
    case 'space3':
      {
        return [sortModifierSuggestions.NULLS_FIRST, sortModifierSuggestions.NULLS_LAST, _complete_items.pipeCompleteItem, {
          ..._complete_items.commaCompleteItem,
          text: ', ',
          command: _factories.TRIGGER_SUGGESTION_COMMAND
        }];
      }
    case 'nulls':
      {
        return handleFragment(innerText, fragment => ['FIRST', 'LAST'].some(completeWord => (0, _helpers.noCaseCompare)(completeWord, fragment)), _fragment => {
          const end = innerText.length + 1;
          const start = end - nulls.length;
          return Object.values(sortModifierSuggestions).map(suggestion => ({
            ...suggestion,
            // we can't use the range generated by handleFragment here
            // because it doesn't really support multi-word completions
            rangeToReplace: {
              start,
              end
            }
          }));
        }, (fragment, rangeToReplace) => {
          return [{
            ..._complete_items.pipeCompleteItem,
            text: ' | '
          }, {
            ..._complete_items.commaCompleteItem,
            text: ', '
          }].map(suggestion => ({
            ...suggestion,
            filterText: fragment,
            text: fragment + suggestion.text,
            rangeToReplace,
            command: _factories.TRIGGER_SUGGESTION_COMMAND
          }));
        });
      }
    case 'space4':
      {
        return [_complete_items.pipeCompleteItem, {
          ..._complete_items.commaCompleteItem,
          text: ', ',
          command: _factories.TRIGGER_SUGGESTION_COMMAND
        }];
      }
  }
  const fieldSuggestions = await getFieldsByType('any', [], {
    openSuggestions: true
  });
  const functionSuggestions = await getFieldsOrFunctionsSuggestions(['any'], 'sort', undefined, getFieldsByType, {
    functions: true,
    fields: false
  });
  return await handleFragment(innerText, columnExists, (_fragment, rangeToReplace) => {
    // SORT fie<suggest>
    return [...pushItUpInTheList(fieldSuggestions.map(suggestion => ({
      ...suggestion,
      command: _factories.TRIGGER_SUGGESTION_COMMAND,
      rangeToReplace
    })), true), ...functionSuggestions];
  }, (fragment, rangeToReplace) => {
    // SORT field<suggest>
    return [{
      ..._complete_items.pipeCompleteItem,
      text: ' | '
    }, {
      ..._complete_items.commaCompleteItem,
      text: ', '
    }, prependSpace(sortModifierSuggestions.ASC), prependSpace(sortModifierSuggestions.DESC), prependSpace(sortModifierSuggestions.NULLS_FIRST), prependSpace(sortModifierSuggestions.NULLS_LAST)].map(s => ({
      ...s,
      filterText: fragment,
      text: fragment + s.text,
      command: _factories.TRIGGER_SUGGESTION_COMMAND,
      rangeToReplace
    }));
  });
};
exports.suggestForSortCmd = suggestForSortCmd;