"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isESQLColumnSortable = exports.isESQLColumnGroupable = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const SPATIAL_FIELDS = ['geo_point', 'geo_shape', 'point', 'shape'];
const SOURCE_FIELD = '_source';
const TSDB_COUNTER_FIELDS_PREFIX = 'counter_';
const UNKNOWN_FIELD = 'unknown';

/**
 * Check if a column is sortable.
 *
 * @param column The DatatableColumn of the field.
 * @returns True if the column is sortable, false otherwise.
 */

const isESQLColumnSortable = column => {
  var _column$meta, _column$meta2, _column$meta3, _column$meta4, _column$meta4$esType;
  // We don't allow sorting on spatial fields
  if (SPATIAL_FIELDS.includes((_column$meta = column.meta) === null || _column$meta === void 0 ? void 0 : _column$meta.type)) {
    return false;
  }

  // we don't allow sorting on the _source field
  if (((_column$meta2 = column.meta) === null || _column$meta2 === void 0 ? void 0 : _column$meta2.type) === SOURCE_FIELD) {
    return false;
  }

  // we don't allow sorting on tsdb counter fields
  if ((_column$meta3 = column.meta) !== null && _column$meta3 !== void 0 && _column$meta3.esType && ((_column$meta4 = column.meta) === null || _column$meta4 === void 0 ? void 0 : (_column$meta4$esType = _column$meta4.esType) === null || _column$meta4$esType === void 0 ? void 0 : _column$meta4$esType.indexOf(TSDB_COUNTER_FIELDS_PREFIX)) !== -1) {
    return false;
  }
  return true;
};

/**
 * Check if a column is groupable (| STATS ... BY <column>).
 *
 * @param column The DatatableColumn of the field.
 * @returns True if the column is groupable, false otherwise.
 */
exports.isESQLColumnSortable = isESQLColumnSortable;
const isESQLColumnGroupable = column => {
  var _column$meta5, _column$meta6, _column$meta7, _column$meta7$esType;
  // we don't allow grouping on the unknown field types
  if (((_column$meta5 = column.meta) === null || _column$meta5 === void 0 ? void 0 : _column$meta5.type) === UNKNOWN_FIELD) {
    return false;
  }
  // we don't allow grouping on tsdb counter fields
  if ((_column$meta6 = column.meta) !== null && _column$meta6 !== void 0 && _column$meta6.esType && ((_column$meta7 = column.meta) === null || _column$meta7 === void 0 ? void 0 : (_column$meta7$esType = _column$meta7.esType) === null || _column$meta7$esType === void 0 ? void 0 : _column$meta7$esType.indexOf(TSDB_COUNTER_FIELDS_PREFIX)) !== -1) {
    return false;
  }
  return true;
};
exports.isESQLColumnGroupable = isESQLColumnGroupable;