"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.collectAllAggFields = collectAllAggFields;
exports.collectAllColumnIdentifiers = collectAllColumnIdentifiers;
exports.collectAllFields = collectAllFields;
exports.collectAllSourceIdentifiers = collectAllSourceIdentifiers;
exports.collectBooleanExpression = collectBooleanExpression;
exports.collectLogicalExpression = collectLogicalExpression;
exports.getEnrichClauses = getEnrichClauses;
exports.getMatchField = getMatchField;
exports.getPolicyName = getPolicyName;
exports.visitByOption = visitByOption;
exports.visitDissect = visitDissect;
exports.visitField = visitField;
exports.visitGrok = visitGrok;
exports.visitOrderExpressions = visitOrderExpressions;
exports.visitPrimaryExpression = visitPrimaryExpression;
exports.visitRenameClauses = visitRenameClauses;
var _antlr = require("antlr4");
var _esql_parser = _interopRequireWildcard(require("../antlr/esql_parser"));
var _factories = require("./factories");
var _helpers = require("./helpers");
var _utils = require("../visitor/utils");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function collectAllSourceIdentifiers(ctx) {
  const fromContexts = ctx.getTypedRuleContexts(_esql_parser.IndexPatternContext);
  return fromContexts.map(sourceCtx => (0, _factories.createSource)(sourceCtx));
}
function terminalNodeToParserRuleContext(node) {
  const context = new _antlr.ParserRuleContext();
  context.start = node.symbol;
  context.stop = node.symbol;
  context.children = [node];
  return context;
}
function extractIdentifiers(ctx) {
  if (ctx instanceof _esql_parser.MetadataOptionContext) {
    return ctx.UNQUOTED_SOURCE_list().map(node => {
      return terminalNodeToParserRuleContext(node);
    }).flat();
  }
  if (ctx instanceof _esql_parser.MvExpandCommandContext) {
    return (0, _factories.wrapIdentifierAsArray)(ctx.qualifiedName());
  }
  return (0, _factories.wrapIdentifierAsArray)(ctx.qualifiedNamePatterns().qualifiedNamePattern_list());
}
function makeColumnsOutOfIdentifiers(identifiers) {
  var _identifiers$filter$m;
  const args = (_identifiers$filter$m = identifiers.filter(child => (0, _factories.textExistsAndIsValid)(child.getText())).map(sourceContext => {
    return (0, _factories.createColumn)(sourceContext);
  })) !== null && _identifiers$filter$m !== void 0 ? _identifiers$filter$m : [];
  return args;
}
function collectAllColumnIdentifiers(ctx) {
  const identifiers = extractIdentifiers(ctx);
  return makeColumnsOutOfIdentifiers(identifiers);
}
function getPolicyName(ctx) {
  if (!ctx._policyName || !(0, _factories.textExistsAndIsValid)(ctx._policyName.text)) {
    return [];
  }
  const policyComponents = ctx._policyName.text.split(':');
  if (policyComponents.length > 1) {
    const [setting, policyName] = policyComponents;
    return [(0, _factories.createSetting)(ctx._policyName, setting), (0, _factories.createPolicy)(ctx._policyName, policyName)];
  }
  return [(0, _factories.createPolicy)(ctx._policyName, policyComponents[0])];
}
function getMatchField(ctx) {
  if (!ctx._matchField) {
    return [];
  }
  const identifier = ctx.qualifiedNamePattern();
  if (identifier) {
    const fn = (0, _factories.createOption)(ctx.ON().getText().toLowerCase(), ctx);
    let max = ctx.ON().symbol.stop;
    if ((0, _factories.textExistsAndIsValid)(identifier.getText())) {
      const column = (0, _factories.createColumn)(identifier);
      fn.args.push(column);
      max = column.location.max;
    }
    fn.location.min = ctx.ON().symbol.start;
    fn.location.max = max;
    return [fn];
  }
  return [];
}
function getEnrichClauses(ctx) {
  const ast = [];
  const withCtx = ctx.WITH();
  if (withCtx) {
    const option = (0, _factories.createOption)(withCtx.getText().toLowerCase(), ctx);
    ast.push(option);
    const clauses = ctx.enrichWithClause_list();
    for (const clause of clauses) {
      var _lastArg$location$max, _lastArg$location;
      if (clause._enrichField) {
        const args = [];
        if (clause.ASSIGN()) {
          var _clause$_enrichField;
          args.push((0, _factories.createColumn)(clause._newName));
          if ((0, _factories.textExistsAndIsValid)((_clause$_enrichField = clause._enrichField) === null || _clause$_enrichField === void 0 ? void 0 : _clause$_enrichField.getText())) {
            args.push((0, _factories.createColumn)(clause._enrichField));
          }
        } else {
          var _clause$_enrichField2;
          // if an explicit assign is not set, create a fake assign with
          // both left and right value with the same column
          if ((0, _factories.textExistsAndIsValid)((_clause$_enrichField2 = clause._enrichField) === null || _clause$_enrichField2 === void 0 ? void 0 : _clause$_enrichField2.getText())) {
            args.push((0, _factories.createColumn)(clause._enrichField), (0, _factories.createColumn)(clause._enrichField));
          }
        }
        if (args.length) {
          const fn = (0, _factories.createFunction)('=', clause, undefined, 'binary-expression');
          fn.args.push(args[0], args[1] ? [args[1]] : []);
          option.args.push(fn);
        }
      }
      const location = option.location;
      const lastArg = (0, _utils.lastItem)(option.args);
      location.min = withCtx.symbol.start;
      location.max = (_lastArg$location$max = lastArg === null || lastArg === void 0 ? void 0 : (_lastArg$location = lastArg.location) === null || _lastArg$location === void 0 ? void 0 : _lastArg$location.max) !== null && _lastArg$location$max !== void 0 ? _lastArg$location$max : withCtx.symbol.stop;
    }
  }
  return ast;
}
function visitLogicalNot(ctx) {
  const fn = (0, _factories.createFunction)('not', ctx, undefined, 'unary-expression');
  fn.args.push(...collectBooleanExpression(ctx.booleanExpression()));
  // update the location of the assign based on arguments
  const argsLocationExtends = (0, _factories.computeLocationExtends)(fn);
  fn.location = argsLocationExtends;
  return fn;
}
function visitLogicalAndsOrs(ctx) {
  const fn = (0, _factories.createFunction)(ctx.AND() ? 'and' : 'or', ctx, undefined, 'binary-expression');
  fn.args.push(...collectBooleanExpression(ctx._left), ...collectBooleanExpression(ctx._right));
  // update the location of the assign based on arguments
  const argsLocationExtends = (0, _factories.computeLocationExtends)(fn);
  fn.location = argsLocationExtends;
  return fn;
}
function visitLogicalIns(ctx) {
  const fn = (0, _factories.createFunction)(ctx.NOT() ? 'not_in' : 'in', ctx, undefined, 'binary-expression');
  const [left, ...list] = ctx.valueExpression_list();
  const leftArg = visitValueExpression(left);
  if (leftArg) {
    fn.args.push(...(Array.isArray(leftArg) ? leftArg : [leftArg]));
    const values = list.map(ve => visitValueExpression(ve));
    const listArgs = values.filter(_factories.nonNullable).flatMap(arg => Array.isArray(arg) ? arg.filter(_factories.nonNullable) : arg);
    // distinguish between missing brackets (missing text error) and an empty list
    if ((0, _factories.textExistsAndIsValid)(ctx.getText())) {
      fn.args.push(listArgs);
    }
  }
  // update the location of the assign based on arguments
  const argsLocationExtends = (0, _factories.computeLocationExtends)(fn);
  fn.location = argsLocationExtends;
  return fn;
}
function getMathOperation(ctx) {
  return (ctx.PLUS() || ctx.MINUS() || ctx.ASTERISK() || ctx.SLASH() || ctx.PERCENT()).getText() || '';
}
function getComparisonName(ctx) {
  return (ctx.EQ() || ctx.NEQ() || ctx.LT() || ctx.LTE() || ctx.GT() || ctx.GTE()).getText() || '';
}
function visitValueExpression(ctx) {
  if (!(0, _factories.textExistsAndIsValid)(ctx.getText())) {
    return [];
  }
  if (ctx instanceof _esql_parser.ValueExpressionDefaultContext) {
    return visitOperatorExpression(ctx.operatorExpression());
  }
  if (ctx instanceof _esql_parser.ComparisonContext) {
    const comparisonNode = ctx.comparisonOperator();
    const comparisonFn = (0, _factories.createFunction)(getComparisonName(comparisonNode), comparisonNode, undefined, 'binary-expression');
    comparisonFn.args.push(visitOperatorExpression(ctx._left), visitOperatorExpression(ctx._right));
    // update the location of the comparisonFn based on arguments
    const argsLocationExtends = (0, _factories.computeLocationExtends)(comparisonFn);
    comparisonFn.location = argsLocationExtends;
    return comparisonFn;
  }
}
function visitOperatorExpression(ctx) {
  if (ctx instanceof _esql_parser.ArithmeticUnaryContext) {
    const arg = visitOperatorExpression(ctx.operatorExpression());
    // this is a number sign thing
    const fn = (0, _factories.createFunction)('*', ctx, undefined, 'binary-expression');
    fn.args.push((0, _factories.createFakeMultiplyLiteral)(ctx, 'integer'));
    if (arg) {
      fn.args.push(arg);
    }
    return fn;
  }
  if (ctx instanceof _esql_parser.ArithmeticBinaryContext) {
    const fn = (0, _factories.createFunction)(getMathOperation(ctx), ctx, undefined, 'binary-expression');
    const args = [visitOperatorExpression(ctx._left), visitOperatorExpression(ctx._right)];
    for (const arg of args) {
      if (arg) {
        fn.args.push(arg);
      }
    }
    // update the location of the assign based on arguments
    const argsLocationExtends = (0, _factories.computeLocationExtends)(fn);
    fn.location = argsLocationExtends;
    return fn;
  }
  if (ctx instanceof _esql_parser.OperatorExpressionDefaultContext) {
    return visitPrimaryExpression(ctx.primaryExpression());
  }
}
function getBooleanValue(ctx) {
  const parentNode = ctx instanceof _esql_parser.BooleanLiteralContext ? ctx.booleanValue() : ctx;
  const booleanTerminalNode = parentNode.TRUE() || parentNode.FALSE();
  return (0, _factories.createLiteral)('boolean', booleanTerminalNode);
}
function getConstant(ctx) {
  if (ctx instanceof _esql_parser.NullLiteralContext) {
    return (0, _factories.createLiteral)('null', ctx.NULL());
  }
  if (ctx instanceof _esql_parser.QualifiedIntegerLiteralContext) {
    // despite the generic name, this is a date unit constant:
    // e.g. 1 year, 15 months
    return (0, _factories.createTimeUnit)(ctx);
  }

  // Decimal type covers multiple ES|QL types: long, double, etc.
  if (ctx instanceof _esql_parser.DecimalLiteralContext) {
    return (0, _factories.createNumericLiteral)(ctx.decimalValue(), 'double');
  }

  // Integer type encompasses integer
  if (ctx instanceof _esql_parser.IntegerLiteralContext) {
    return (0, _factories.createNumericLiteral)(ctx.integerValue(), 'integer');
  }
  if (ctx instanceof _esql_parser.BooleanLiteralContext) {
    return getBooleanValue(ctx);
  }
  if (ctx instanceof _esql_parser.StringLiteralContext) {
    // String literal covers multiple ES|QL types: text and keyword types
    return (0, _factories.createLiteral)('keyword', ctx.string_().QUOTED_STRING());
  }
  if (ctx instanceof _esql_parser.NumericArrayLiteralContext || ctx instanceof _esql_parser.BooleanArrayLiteralContext || ctx instanceof _esql_parser.StringArrayLiteralContext) {
    const values = [];
    for (const numericValue of ctx.getTypedRuleContexts(_esql_parser.NumericValueContext)) {
      const isDecimal = numericValue.decimalValue() !== null && numericValue.decimalValue() !== undefined;
      const value = numericValue.decimalValue() || numericValue.integerValue();
      values.push((0, _factories.createNumericLiteral)(value, isDecimal ? 'double' : 'integer'));
    }
    for (const booleanValue of ctx.getTypedRuleContexts(_esql_parser.BooleanValueContext)) {
      values.push(getBooleanValue(booleanValue));
    }
    for (const string of ctx.getTypedRuleContexts(_esql_parser.StringContext)) {
      // String literal covers multiple ES|QL types: text and keyword types
      const literal = (0, _factories.createLiteral)('keyword', string.QUOTED_STRING());
      if (literal) {
        values.push(literal);
      }
    }
    return (0, _factories.createList)(ctx, values);
  }
  if (ctx instanceof _esql_parser.InputParameterContext && ctx.children) {
    const values = [];
    for (const child of ctx.children) {
      if (child instanceof _esql_parser.InputParamContext) {
        const literal = {
          type: 'literal',
          literalType: 'param',
          paramType: 'unnamed',
          text: ctx.getText(),
          name: '',
          value: '',
          location: (0, _helpers.getPosition)(ctx.start, ctx.stop),
          incomplete: Boolean(ctx.exception)
        };
        values.push(literal);
      } else if (child instanceof _esql_parser.InputNamedOrPositionalParamContext) {
        const text = child.getText();
        const value = text.slice(1);
        const valueAsNumber = Number(value);
        const isPositional = String(valueAsNumber) === value;
        if (isPositional) {
          const literal = {
            type: 'literal',
            literalType: 'param',
            paramType: 'positional',
            value: valueAsNumber,
            text,
            name: '',
            location: (0, _helpers.getPosition)(ctx.start, ctx.stop),
            incomplete: Boolean(ctx.exception)
          };
          values.push(literal);
        } else {
          const literal = {
            type: 'literal',
            literalType: 'param',
            paramType: 'named',
            value,
            text,
            name: '',
            location: (0, _helpers.getPosition)(ctx.start, ctx.stop),
            incomplete: Boolean(ctx.exception)
          };
          values.push(literal);
        }
      }
    }
    return values;
  }
  return (0, _factories.createUnknownItem)(ctx);
}
function visitRenameClauses(clausesCtx) {
  return clausesCtx.map(clause => {
    var _clause$_oldName;
    const asToken = clause.getToken(_esql_parser.default.AS, 0);
    if (asToken && (0, _factories.textExistsAndIsValid)(asToken.getText())) {
      const option = (0, _factories.createOption)(asToken.getText().toLowerCase(), clause);
      for (const arg of [clause._oldName, clause._newName]) {
        if ((0, _factories.textExistsAndIsValid)(arg.getText())) {
          option.args.push((0, _factories.createColumn)(arg));
        }
      }
      const firstArg = (0, _utils.firstItem)(option.args);
      const lastArg = (0, _utils.lastItem)(option.args);
      const location = option.location;
      if (firstArg) location.min = firstArg.location.min;
      if (lastArg) location.max = lastArg.location.max;
      return option;
    } else if ((0, _factories.textExistsAndIsValid)((_clause$_oldName = clause._oldName) === null || _clause$_oldName === void 0 ? void 0 : _clause$_oldName.getText())) {
      return (0, _factories.createColumn)(clause._oldName);
    }
  }).filter(_factories.nonNullable);
}
function visitPrimaryExpression(ctx) {
  if (ctx instanceof _esql_parser.ConstantDefaultContext) {
    return getConstant(ctx.constant());
  }
  if (ctx instanceof _esql_parser.DereferenceContext) {
    return (0, _factories.createColumn)(ctx.qualifiedName());
  }
  if (ctx instanceof _esql_parser.ParenthesizedExpressionContext) {
    return collectBooleanExpression(ctx.booleanExpression());
  }
  if (ctx instanceof _esql_parser.FunctionContext) {
    const functionExpressionCtx = ctx.functionExpression();
    const functionNameContext = functionExpressionCtx.functionName().MATCH() ? functionExpressionCtx.functionName().MATCH() : functionExpressionCtx.functionName().identifierOrParameter();
    const fn = (0, _factories.createFunction)(functionNameContext.getText().toLowerCase(), ctx, undefined, 'variadic-call');
    const asteriskArg = functionExpressionCtx.ASTERISK() ? (0, _factories.createColumnStar)(functionExpressionCtx.ASTERISK()) : undefined;
    if (asteriskArg) {
      fn.args.push(asteriskArg);
    }
    const functionArgs = functionExpressionCtx.booleanExpression_list().flatMap(collectBooleanExpression).filter(_factories.nonNullable);
    if (functionArgs.length) {
      fn.args.push(...functionArgs);
    }
    return fn;
  }
  if (ctx instanceof _esql_parser.InlineCastContext) {
    return collectInlineCast(ctx);
  }
  return (0, _factories.createUnknownItem)(ctx);
}
function collectInlineCast(ctx) {
  const primaryExpression = visitPrimaryExpression(ctx.primaryExpression());
  return (0, _factories.createInlineCast)(ctx, primaryExpression);
}
function collectLogicalExpression(ctx) {
  if (ctx instanceof _esql_parser.LogicalNotContext) {
    return [visitLogicalNot(ctx)];
  }
  if (ctx instanceof _esql_parser.LogicalBinaryContext) {
    return [visitLogicalAndsOrs(ctx)];
  }
  if (ctx instanceof _esql_parser.LogicalInContext) {
    return [visitLogicalIns(ctx)];
  }
  return [];
}
function collectRegexExpression(ctx) {
  const regexes = ctx.getTypedRuleContexts(_esql_parser.RegexBooleanExpressionContext);
  const ret = [];
  return ret.concat(regexes.map(regex => {
    var _regex$_kind$text;
    const negate = regex.NOT();
    const likeType = ((_regex$_kind$text = regex._kind.text) === null || _regex$_kind$text === void 0 ? void 0 : _regex$_kind$text.toLowerCase()) || '';
    const fnName = `${negate ? 'not_' : ''}${likeType}`;
    const fn = (0, _factories.createFunction)(fnName, regex, undefined, 'binary-expression');
    const arg = visitValueExpression(regex.valueExpression());
    if (arg) {
      fn.args.push(arg);
      const literal = (0, _factories.createLiteral)('keyword', regex._pattern.QUOTED_STRING());
      if (literal) {
        fn.args.push(literal);
      }
    }
    return fn;
  }));
}
function collectIsNullExpression(ctx) {
  if (!(ctx instanceof _esql_parser.IsNullContext)) {
    return [];
  }
  const negate = ctx.NOT();
  const fnName = `is${negate ? ' not ' : ' '}null`;
  const fn = (0, _factories.createFunction)(fnName, ctx, undefined, 'postfix-unary-expression');
  const arg = visitValueExpression(ctx.valueExpression());
  if (arg) {
    fn.args.push(arg);
  }
  return [fn];
}
function collectDefaultExpression(ctx) {
  if (!(ctx instanceof _esql_parser.BooleanDefaultContext)) {
    return [];
  }
  const arg = visitValueExpression(ctx.valueExpression());
  return arg ? [arg] : [];
}
function collectBooleanExpression(ctx) {
  const ast = [];
  if (!ctx) {
    return ast;
  }
  return ast.concat(collectLogicalExpression(ctx), collectRegexExpression(ctx), collectIsNullExpression(ctx), collectDefaultExpression(ctx)).flat();
}
function visitField(ctx) {
  if (ctx.qualifiedName() && ctx.ASSIGN()) {
    const fn = (0, _factories.createFunction)(ctx.ASSIGN().getText(), ctx, undefined, 'binary-expression');
    fn.args.push((0, _factories.createColumn)(ctx.qualifiedName()), collectBooleanExpression(ctx.booleanExpression()));
    // update the location of the assign based on arguments
    const argsLocationExtends = (0, _factories.computeLocationExtends)(fn);
    fn.location = argsLocationExtends;
    return [fn];
  }
  return collectBooleanExpression(ctx.booleanExpression());
}
function collectAllAggFields(ctx) {
  const ast = [];
  if (!ctx) {
    return ast;
  }
  try {
    for (const aggField of ctx.aggField_list()) {
      ast.push(...visitField(aggField.field()));
    }
  } catch (e) {
    // do nothing
  }
  return ast;
}
function collectAllFields(ctx) {
  const ast = [];
  if (!ctx) {
    return ast;
  }
  try {
    for (const field of ctx.field_list()) {
      ast.push(...visitField(field));
    }
  } catch (e) {
    // do nothing
  }
  return ast;
}
function visitByOption(ctx, expr) {
  const byCtx = ctx.BY();
  if (!byCtx || !expr) {
    return [];
  }
  const option = (0, _factories.createOption)(byCtx.getText().toLowerCase(), ctx);
  option.args.push(...collectAllFields(expr));
  option.location.min = byCtx.symbol.start;
  const lastArg = (0, _utils.lastItem)(option.args);
  if (lastArg) option.location.max = lastArg.location.max;
  return [option];
}
const visitOrderExpression = ctx => {
  var _ctx$_ordering, _ctx$_ordering$text, _ctx$_nullOrdering, _ctx$_nullOrdering$te;
  const arg = collectBooleanExpression(ctx.booleanExpression())[0];
  let order = '';
  let nulls = '';
  const ordering = (_ctx$_ordering = ctx._ordering) === null || _ctx$_ordering === void 0 ? void 0 : (_ctx$_ordering$text = _ctx$_ordering.text) === null || _ctx$_ordering$text === void 0 ? void 0 : _ctx$_ordering$text.toUpperCase();
  if (ordering) order = ordering;
  const nullOrdering = (_ctx$_nullOrdering = ctx._nullOrdering) === null || _ctx$_nullOrdering === void 0 ? void 0 : (_ctx$_nullOrdering$te = _ctx$_nullOrdering.text) === null || _ctx$_nullOrdering$te === void 0 ? void 0 : _ctx$_nullOrdering$te.toUpperCase();
  switch (nullOrdering) {
    case 'LAST':
      nulls = 'NULLS LAST';
      break;
    case 'FIRST':
      nulls = 'NULLS FIRST';
      break;
  }
  if (!order && !nulls) {
    return arg;
  }
  return (0, _factories.createOrderExpression)(ctx, arg, order, nulls);
};
function visitOrderExpressions(ctx) {
  const ast = [];
  for (const orderCtx of ctx) {
    ast.push(visitOrderExpression(orderCtx));
  }
  return ast;
}
function visitDissect(ctx) {
  const pattern = ctx.string_().getToken(_esql_parser.default.QUOTED_STRING, 0);
  return [visitPrimaryExpression(ctx.primaryExpression()), ...(pattern && (0, _factories.textExistsAndIsValid)(pattern.getText()) ? [(0, _factories.createLiteral)('keyword', pattern), ...visitDissectOptions(ctx.commandOptions())] : [])].filter(_factories.nonNullable);
}
function visitGrok(ctx) {
  const pattern = ctx.string_().getToken(_esql_parser.default.QUOTED_STRING, 0);
  return [visitPrimaryExpression(ctx.primaryExpression()), ...(pattern && (0, _factories.textExistsAndIsValid)(pattern.getText()) ? [(0, _factories.createLiteral)('keyword', pattern)] : [])].filter(_factories.nonNullable);
}
function visitDissectOptions(ctx) {
  if (!ctx) {
    return [];
  }
  const options = [];
  for (const optionCtx of ctx.commandOption_list()) {
    const option = (0, _factories.createOption)((0, _factories.sanitizeIdentifierString)(optionCtx.identifier()).toLowerCase(), optionCtx);
    options.push(option);
    // it can throw while accessing constant for incomplete commands, so try catch it
    try {
      const optionValue = getConstant(optionCtx.constant());
      if (optionValue != null) {
        option.args.push(optionValue);
      }
    } catch (e) {
      // do nothing here
    }
  }
  return options;
}