"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.computeLocationExtends = computeLocationExtends;
exports.createAstBaseItem = createAstBaseItem;
exports.createColumn = createColumn;
exports.createColumnStar = createColumnStar;
exports.createCommand = void 0;
exports.createError = createError;
exports.createFakeMultiplyLiteral = createFakeMultiplyLiteral;
exports.createFunction = createFunction;
exports.createList = exports.createInlineCast = void 0;
exports.createLiteral = createLiteral;
exports.createLiteralString = createLiteralString;
exports.createNumericLiteral = void 0;
exports.createOption = createOption;
exports.createOrderExpression = void 0;
exports.createPolicy = createPolicy;
exports.createSetting = createSetting;
exports.createSource = createSource;
exports.createTimeUnit = createTimeUnit;
exports.createUnknownItem = createUnknownItem;
exports.nonNullable = nonNullable;
exports.sanitizeIdentifierString = sanitizeIdentifierString;
exports.textExistsAndIsValid = textExistsAndIsValid;
exports.wrapIdentifierAsArray = wrapIdentifierAsArray;
var _esql_parser = require("../antlr/esql_parser");
var _constants = require("./constants");
var _helpers = require("./helpers");
var _builder = require("../builder");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * In case of changes in the grammar, this script should be updated: esql_update_ast_script.js
 */

function nonNullable(v) {
  return v != null;
}
function createAstBaseItem(name, ctx) {
  return {
    name,
    text: ctx.getText(),
    location: (0, _helpers.getPosition)(ctx.start, ctx.stop),
    incomplete: Boolean(ctx.exception)
  };
}
const createParserFields = ctx => ({
  text: ctx.getText(),
  location: (0, _helpers.getPosition)(ctx.start, ctx.stop),
  incomplete: Boolean(ctx.exception)
});
const createCommand = (name, ctx) => _builder.Builder.command({
  name,
  args: []
}, createParserFields(ctx));
exports.createCommand = createCommand;
const createInlineCast = (ctx, value) => _builder.Builder.expression.inlineCast({
  castType: ctx.dataType().getText().toLowerCase(),
  value
}, createParserFields(ctx));
exports.createInlineCast = createInlineCast;
const createList = (ctx, values) => _builder.Builder.expression.literal.list({
  values
}, createParserFields(ctx));
exports.createList = createList;
const createNumericLiteral = (ctx, literalType) => _builder.Builder.expression.literal.numeric({
  value: Number(ctx.getText()),
  literalType
}, createParserFields(ctx));
exports.createNumericLiteral = createNumericLiteral;
function createFakeMultiplyLiteral(ctx, literalType) {
  return {
    type: 'literal',
    literalType,
    text: ctx.getText(),
    name: ctx.getText(),
    value: ctx.PLUS() ? 1 : -1,
    location: (0, _helpers.getPosition)(ctx.start, ctx.stop),
    incomplete: Boolean(ctx.exception)
  };
}
function createLiteralString(token) {
  const text = token.text;
  return {
    type: 'literal',
    literalType: 'keyword',
    text,
    name: text,
    value: text,
    location: (0, _helpers.getPosition)(token),
    incomplete: Boolean(token.text === '')
  };
}
function isMissingText(text) {
  return /<missing /.test(text);
}
function textExistsAndIsValid(text) {
  return !!(text && !isMissingText(text));
}
function createLiteral(type, node) {
  if (!node) {
    return {
      type: 'literal',
      name: 'unknown',
      text: 'unknown',
      value: 'unknown',
      literalType: type,
      location: {
        min: 0,
        max: 0
      },
      incomplete: false
    };
  }
  const text = node.getText();
  const partialLiteral = {
    type: 'literal',
    text,
    name: text,
    location: (0, _helpers.getPosition)(node.symbol),
    incomplete: isMissingText(text)
  };
  if (type === 'double' || type === 'integer') {
    return {
      ...partialLiteral,
      literalType: type,
      value: Number(text),
      paramType: 'number'
    };
  } else if (type === 'param') {
    throw new Error('Should never happen');
  }
  return {
    ...partialLiteral,
    literalType: type,
    value: text
  };
}
function createTimeUnit(ctx) {
  return {
    type: 'timeInterval',
    quantity: Number(ctx.integerValue().INTEGER_LITERAL().getText()),
    unit: ctx.UNQUOTED_IDENTIFIER().symbol.text,
    text: ctx.getText(),
    location: (0, _helpers.getPosition)(ctx.start, ctx.stop),
    name: `${ctx.integerValue().INTEGER_LITERAL().getText()} ${ctx.UNQUOTED_IDENTIFIER().symbol.text}`,
    incomplete: Boolean(ctx.exception)
  };
}
function createFunction(name, ctx, customPosition, subtype) {
  const node = {
    type: 'function',
    name,
    text: ctx.getText(),
    location: customPosition !== null && customPosition !== void 0 ? customPosition : (0, _helpers.getPosition)(ctx.start, ctx.stop),
    args: [],
    incomplete: Boolean(ctx.exception)
  };
  if (subtype) {
    node.subtype = subtype;
  }
  return node;
}
const createOrderExpression = (ctx, arg, order, nulls) => {
  const node = {
    type: 'order',
    name: '',
    order,
    nulls,
    args: [arg],
    text: ctx.getText(),
    location: (0, _helpers.getPosition)(ctx.start, ctx.stop),
    incomplete: Boolean(ctx.exception)
  };
  return node;
};
exports.createOrderExpression = createOrderExpression;
function walkFunctionStructure(args, initialLocation, prop, getNextItemIndex) {
  let nextArg = args[getNextItemIndex(args)];
  const location = {
    ...initialLocation
  };
  while (Array.isArray(nextArg) || nextArg) {
    if (Array.isArray(nextArg)) {
      nextArg = nextArg[getNextItemIndex(nextArg)];
    } else {
      location[prop] = Math[prop](location[prop], nextArg.location[prop]);
      if (nextArg.type === 'function') {
        nextArg = nextArg.args[getNextItemIndex(nextArg.args)];
      } else {
        nextArg = undefined;
      }
    }
  }
  return location[prop];
}
function computeLocationExtends(fn) {
  const location = fn.location;
  if (fn.args) {
    // get min location navigating in depth keeping the left/first arg
    location.min = walkFunctionStructure(fn.args, location, 'min', () => 0);
    // get max location navigating in depth keeping the right/last arg
    location.max = walkFunctionStructure(fn.args, location, 'max', args => args.length - 1);
    // in case of empty array as last arg, bump the max location by 3 chars (empty brackets)
    if (Array.isArray(fn.args[fn.args.length - 1]) && !fn.args[fn.args.length - 1].length) {
      location.max += 3;
    }
  }
  return location;
}

// Note: do not import esql_parser or bundle size will grow up by ~500 kb
/**
 * Do not touch this piece of code as it is auto-generated by a script
 */

/* SCRIPT_MARKER_START */
function getQuotedText(ctx) {
  return [27 /* esql_parser.QUOTED_STRING */, 69 /* esql_parser.QUOTED_IDENTIFIER */].map(keyCode => ctx.getToken(keyCode, 0)).filter(nonNullable)[0];
}
function getUnquotedText(ctx) {
  return [68 /* esql_parser.UNQUOTED_IDENTIFIER */, 77 /* esql_parser.UNQUOTED_SOURCE */].map(keyCode => ctx.getToken(keyCode, 0)).filter(nonNullable)[0];
}
/* SCRIPT_MARKER_END */

function isQuoted(text) {
  return text && /^(`)/.test(text);
}

/**
 * Follow a similar logic to the ES one:
 * * remove backticks at the beginning and at the end
 * * remove double backticks
 */
function safeBackticksRemoval(text) {
  return (text === null || text === void 0 ? void 0 : text.replace(_constants.TICKS_REGEX, '').replace(_constants.DOUBLE_TICKS_REGEX, _constants.SINGLE_BACKTICK)) || '';
}
function sanitizeSourceString(ctx) {
  const contextText = ctx.getText();
  // If wrapped by triple quote, remove
  if (contextText.startsWith(`"""`) && contextText.endsWith(`"""`)) {
    return contextText.replace(/\"\"\"/g, '');
  }
  // If wrapped by single quote, remove
  if (contextText.startsWith(`"`) && contextText.endsWith(`"`)) {
    return contextText.slice(1, -1);
  }
  return contextText;
}
const unquoteIndexString = indexString => {
  const isStringQuoted = indexString[0] === '"';
  if (!isStringQuoted) {
    return indexString;
  }

  // If wrapped by triple double quotes, simply remove them.
  if (indexString.startsWith(`"""`) && indexString.endsWith(`"""`)) {
    return indexString.slice(3, -3);
  }

  // If wrapped by double quote, remove them and unescape the string.
  if (indexString[indexString.length - 1] === '"') {
    indexString = indexString.slice(1, -1);
    indexString = indexString.replace(/\\"/g, '"').replace(/\\r/g, '\r').replace(/\\n/g, '\n').replace(/\\t/g, '\t').replace(/\\\\/g, '\\');
    return indexString;
  }

  // This should never happen, but if it does, return the original string.
  return indexString;
};
function sanitizeIdentifierString(ctx) {
  var _getUnquotedText, _getQuotedText;
  const result = ((_getUnquotedText = getUnquotedText(ctx)) === null || _getUnquotedText === void 0 ? void 0 : _getUnquotedText.getText()) || safeBackticksRemoval((_getQuotedText = getQuotedText(ctx)) === null || _getQuotedText === void 0 ? void 0 : _getQuotedText.getText()) || safeBackticksRemoval(ctx.getText()); // for some reason some quoted text is not detected correctly by the parser
  // TODO - understand why <missing null> is now returned as the match text for the FROM command
  return result === '<missing null>' ? '' : result;
}
function wrapIdentifierAsArray(identifierCtx) {
  return Array.isArray(identifierCtx) ? identifierCtx : [identifierCtx];
}
function createSetting(policyName, mode) {
  return {
    type: 'mode',
    name: mode.replace('_', '').toLowerCase(),
    text: mode,
    location: (0, _helpers.getPosition)(policyName, {
      stop: policyName.start + mode.length - 1
    }),
    // unfortunately this is the only location we have
    incomplete: false
  };
}

/**
 * In https://github.com/elastic/elasticsearch/pull/103949 the ENRICH policy name
 * changed from rule to token type so we need to handle this specifically
 */
function createPolicy(token, policy) {
  return {
    type: 'source',
    name: policy,
    text: policy,
    sourceType: 'policy',
    location: (0, _helpers.getPosition)({
      start: token.stop - policy.length + 1,
      stop: token.stop
    }),
    // take into account ccq modes
    incomplete: false
  };
}
function createSource(ctx, type = 'index') {
  const text = sanitizeSourceString(ctx);
  let cluster = '';
  let index = '';
  if (ctx instanceof _esql_parser.IndexPatternContext) {
    const clusterString = ctx.clusterString();
    const indexString = ctx.indexString();
    if (clusterString) {
      cluster = clusterString.getText();
    }
    if (indexString) {
      index = indexString.getText();
      index = unquoteIndexString(index);
    }
  }
  return {
    type: 'source',
    cluster,
    index,
    name: text,
    sourceType: type,
    location: (0, _helpers.getPosition)(ctx.start, ctx.stop),
    incomplete: Boolean(ctx.exception || text === ''),
    text: ctx === null || ctx === void 0 ? void 0 : ctx.getText()
  };
}
function createColumnStar(ctx) {
  const text = ctx.getText();
  return {
    type: 'column',
    name: text,
    parts: [text],
    text,
    location: (0, _helpers.getPosition)(ctx.symbol),
    incomplete: ctx.getText() === '',
    quoted: false
  };
}
function createColumn(ctx) {
  const parts = [];
  if (ctx instanceof _esql_parser.QualifiedNamePatternContext) {
    parts.push(...ctx.identifierPattern_list().map(identifier => (0, _helpers.parseIdentifier)(identifier.getText())));
  } else if (ctx instanceof _esql_parser.QualifiedNameContext) {
    parts.push(...ctx.identifierOrParameter_list().map(identifier => (0, _helpers.parseIdentifier)(identifier.getText())));
  } else {
    parts.push(sanitizeIdentifierString(ctx));
  }
  const text = sanitizeIdentifierString(ctx);
  const hasQuotes = Boolean(getQuotedText(ctx) || isQuoted(ctx.getText()));
  return {
    type: 'column',
    name: text,
    parts,
    text: ctx.getText(),
    location: (0, _helpers.getPosition)(ctx.start, ctx.stop),
    incomplete: Boolean(ctx.exception || text === ''),
    quoted: hasQuotes
  };
}
function createOption(name, ctx) {
  var _ctx$children;
  return {
    type: 'option',
    name,
    text: ctx.getText(),
    location: (0, _helpers.getPosition)(ctx.start, ctx.stop),
    args: [],
    incomplete: Boolean(ctx.exception || ((_ctx$children = ctx.children) === null || _ctx$children === void 0 ? void 0 : _ctx$children.some(c => {
      // @ts-expect-error not exposed in type but exists see https://github.com/antlr/antlr4/blob/v4.11.1/runtime/JavaScript/src/antlr4/tree/ErrorNodeImpl.js#L19
      return Boolean(c.isErrorNode);
    })))
  };
}
function createUnknownItem(ctx) {
  return {
    type: 'unknown',
    name: 'unknown',
    text: ctx.getText(),
    location: (0, _helpers.getPosition)(ctx.start, ctx.stop),
    incomplete: Boolean(ctx.exception)
  };
}
function createError(exception) {
  const token = exception.offendingToken;
  return {
    type: 'error',
    text: `SyntaxError: ${exception.message}`,
    location: (0, _helpers.getPosition)(token)
  };
}