"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.removeCommandOption = exports.removeCommandArgument = exports.removeCommand = exports.listCommands = exports.findCommandOptionByName = exports.findCommandOption = exports.findCommandByName = exports.findCommand = exports.appendCommandOption = exports.appendCommandArgument = exports.appendCommand = void 0;
var _util = require("../ast/util");
var _builder = require("../builder");
var _visitor = require("../visitor");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Returns an iterator for all command AST nodes in the query. If a predicate is
 * provided, only commands that satisfy the predicate will be returned.
 *
 * @param ast Root AST node to search for commands.
 * @param predicate Optional predicate to filter commands.
 * @returns A list of commands found in the AST.
 */
const listCommands = (ast, predicate) => {
  return new _visitor.Visitor().on('visitQuery', function* (ctx) {
    for (const cmd of ctx.commands()) {
      if (!predicate || predicate(cmd)) {
        yield cmd;
      }
    }
  }).visitQuery(ast);
};

/**
 * Returns the first command AST node at a given index in the query that
 * satisfies the predicate. If no index is provided, the first command found
 * will be returned.
 *
 * @param ast Root AST node to search for commands.
 * @param predicate Optional predicate to filter commands.
 * @param index The index of the command to return.
 * @returns The command found in the AST, if any.
 */
exports.listCommands = listCommands;
const findCommand = (ast, predicate, index = 0) => {
  for (const cmd of listCommands(ast, predicate)) {
    if (!index) {
      return cmd;
    }
    index--;
  }
  return undefined;
};

/**
 * Returns the first command option AST node that satisfies the predicate.
 *
 * @param command The command AST node to search for options.
 * @param predicate The predicate to filter options.
 * @returns The option found in the command, if any.
 */
exports.findCommand = findCommand;
const findCommandOption = (command, predicate) => {
  return new _visitor.Visitor().on('visitCommand', ctx => {
    for (const opt of ctx.options()) {
      if (predicate(opt)) {
        return opt;
      }
    }
    return undefined;
  }).visitCommand(command);
};

/**
 * Returns the first command AST node at a given index with a given name in the
 * query. If no index is provided, the first command found will be returned.
 *
 * @param ast Root AST node to search for commands.
 * @param commandName The name of the command to find.
 * @param index The index of the command to return.
 * @returns The command found in the AST, if any.
 */
exports.findCommandOption = findCommandOption;
const findCommandByName = (ast, commandName, index = 0) => {
  return findCommand(ast, cmd => cmd.name === commandName, index);
};

/**
 * Returns the first command option AST node with a given name in the query.
 *
 * @param ast The root AST node to search for command options.
 * @param commandName Command name to search for.
 * @param optionName Option name to search for.
 * @returns The option found in the command, if any.
 */
exports.findCommandByName = findCommandByName;
const findCommandOptionByName = (ast, commandName, optionName) => {
  const command = findCommand(ast, cmd => cmd.name === commandName);
  if (!command) {
    return undefined;
  }
  return findCommandOption(command, opt => opt.name === optionName);
};

/**
 * Adds a new command to the query AST node.
 *
 * @param ast The root AST node to append the command to.
 * @param command The command AST node to append.
 */
exports.findCommandOptionByName = findCommandOptionByName;
const appendCommand = (ast, command) => {
  ast.commands.push(command);
};

/**
 * Inserts a command option into the command's arguments list. The option can
 * be specified as a string or an AST node.
 *
 * @param command The command AST node to insert the option into.
 * @param option The option to insert.
 * @returns The inserted option.
 */
exports.appendCommand = appendCommand;
const appendCommandOption = (command, option) => {
  if (typeof option === 'string') {
    option = _builder.Builder.option({
      name: option
    });
  }
  command.args.push(option);
  return option;
};
exports.appendCommandOption = appendCommandOption;
const appendCommandArgument = (command, expression) => {
  if (expression.type === 'option') {
    command.args.push(expression);
    return command.args.length - 1;
  }
  const index = command.args.findIndex(arg => (0, _util.isOptionNode)(arg));
  if (index > -1) {
    command.args.splice(index, 0, expression);
    return index;
  }
  command.args.push(expression);
  return command.args.length - 1;
};
exports.appendCommandArgument = appendCommandArgument;
const removeCommand = (ast, command) => {
  const cmds = ast.commands;
  const length = cmds.length;
  for (let i = 0; i < length; i++) {
    if (cmds[i] === command) {
      cmds.splice(i, 1);
      return true;
    }
  }
  return false;
};

/**
 * Removes the first command option from the command's arguments list that
 * satisfies the predicate.
 *
 * @param command The command AST node to remove the option from.
 * @param predicate The predicate to filter options.
 * @returns The removed option, if any.
 */
exports.removeCommand = removeCommand;
const removeCommandOption = (ast, option) => {
  return new _visitor.Visitor().on('visitCommandOption', ctx => {
    return ctx.node === option;
  }).on('visitCommand', ctx => {
    let target;
    for (const opt of ctx.options()) {
      if (opt === option) {
        target = opt;
        break;
      }
    }
    if (!target) {
      return false;
    }
    const index = ctx.node.args.indexOf(target);
    if (index === -1) {
      return false;
    }
    ctx.node.args.splice(index, 1);
    return true;
  }).on('visitQuery', ctx => {
    for (const success of ctx.visitCommands()) {
      if (success) {
        return true;
      }
    }
    return false;
  }).visitQuery(ast);
};

/**
 * Searches all command arguments in the query AST node and removes the node
 * from the command's arguments list.
 *
 * @param ast The root AST node to search for command arguments.
 * @param node The argument AST node to remove.
 * @returns Returns true if the argument was removed, false otherwise.
 */
exports.removeCommandOption = removeCommandOption;
const removeCommandArgument = (ast, node) => {
  return new _visitor.Visitor().on('visitCommand', ctx => {
    const args = ctx.node.args;
    const length = args.length;
    for (let i = 0; i < length; i++) {
      if (args[i] === node) {
        args.splice(i, 1);
        return true;
      }
    }
    return false;
  }).on('visitQuery', ctx => {
    for (const success of ctx.visitCommands()) {
      if (success) {
        return true;
      }
    }
    return false;
  }).visitQuery(ast);
};
exports.removeCommandArgument = removeCommandArgument;