"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.upsert = exports.set = exports.remove = exports.list = exports.find = exports.byIndex = void 0;
var _builder = require("../../../builder");
var generic = _interopRequireWildcard(require("../../generic"));
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Lists all "LIMIT" commands in the query AST.
 *
 * @param ast The root AST node to search for "LIMIT" commands.
 * @returns A collection of "LIMIT" commands.
 */
const list = ast => {
  return generic.listCommands(ast, cmd => cmd.name === 'limit');
};

/**
 * Retrieves the "LIMIT" command at the specified index in order of appearance.
 *
 * @param ast The root AST node to search for "LIMIT" commands.
 * @param index The index of the "LIMIT" command to retrieve.
 * @returns The "LIMIT" command at the specified index, if any.
 */
exports.list = list;
const byIndex = (ast, index) => {
  return [...list(ast)][index];
};

/**
 * Finds the first "LIMIT" command that satisfies the provided predicate.
 *
 * @param ast The root AST node to search for "LIMIT" commands.
 * @param predicate The predicate function to apply to each "LIMIT" command.
 * @returns The first "LIMIT" command that satisfies the predicate, if any.
 */
exports.byIndex = byIndex;
const find = (ast, predicate) => {
  return [...list(ast)].find(predicate);
};

/**
 * Deletes the specified "LIMIT" command from the query AST.
 *
 * @param ast The root AST node to search for "LIMIT" commands.
 * @param index The index of the "LIMIT" command to remove.
 * @returns The removed "LIMIT" command, if any.
 */
exports.find = find;
const remove = (ast, index = 0) => {
  const command = generic.findCommandByName(ast, 'limit', index);
  if (!command) {
    return;
  }
  const success = generic.removeCommand(ast, command);
  if (!success) {
    return;
  }
  return command;
};

/**
 * Sets the value of the specified "LIMIT" command. If `indexOrPredicate` is not
 * specified will update the first "LIMIT" command found, if any.
 *
 * @param ast The root AST node to search for "LIMIT" commands.
 * @param value The new value to set.
 * @param indexOrPredicate The index of the "LIMIT" command to update, or a
 *     predicate function.
 * @returns The updated "LIMIT" command, if any.
 */
exports.remove = remove;
const set = (ast, value, indexOrPredicate = 0) => {
  const node = typeof indexOrPredicate === 'number' ? byIndex(ast, indexOrPredicate) : find(ast, indexOrPredicate);
  if (!node) {
    return;
  }
  const literal = _builder.Builder.expression.literal.numeric({
    literalType: 'integer',
    value
  });
  node.args = [literal];
  return node;
};

/**
 * Updates the value of the specified "LIMIT" command. If the "LIMIT" command
 * is not found, a new one will be created and appended to the query AST.
 *
 * @param ast The root AST node to search for "LIMIT" commands.
 * @param value The new value to set.
 * @param indexOrPredicate The index of the "LIMIT" command to update, or a
 *     predicate function.
 * @returns The updated or newly created "LIMIT" command.
 */
exports.set = set;
const upsert = (ast, value, indexOrPredicate = 0) => {
  const node = set(ast, value, indexOrPredicate);
  if (node) {
    return node;
  }
  const literal = _builder.Builder.expression.literal.numeric({
    literalType: 'integer',
    value
  });
  const command = _builder.Builder.command({
    name: 'limit',
    args: [literal]
  });
  generic.appendCommand(ast, command);
  return command;
};
exports.upsert = upsert;