"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateEntityDefinitionRoute = void 0;
var _entitiesSchema = require("@kbn/entities-schema");
var _zod = require("@kbn/zod");
var _entity_security_exception = require("../../lib/entities/errors/entity_security_exception");
var _invalid_transform_error = require("../../lib/entities/errors/invalid_transform_error");
var _create_entity_manager_server_route = require("../create_entity_manager_server_route");
var _entity_not_found = require("../../lib/entities/errors/entity_not_found");
var _entity_definition_update_conflict = require("../../lib/entities/errors/entity_definition_update_conflict");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * @openapi
 * /internal/entities/definition:
 *   patch:
 *     description: Update an entity definition.
 *     tags:
 *       - definitions
 *     parameters:
 *       - in: path
 *         name: id
 *         description: The entity definition ID
 *         schema:
 *           type: string
 *         required: true
 *     requestBody:
 *       description: The definition properties to update
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/entityDefinitionUpdateSchema'
 *     responses:
 *       200:
 *         description: Success
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/entityDefinitionSchema'
 *       400:
 *         description: The entity definition cannot be installed; see the error for more details
 *       404:
 *         description: The entity definition does not exist
 *       403:
 *         description: User is not allowed to update the entity definition
 *       409:
 *         description: The entity definition is being updated by another request
 */
const updateEntityDefinitionRoute = exports.updateEntityDefinitionRoute = (0, _create_entity_manager_server_route.createEntityManagerServerRoute)({
  endpoint: 'PATCH /internal/entities/definition/{id}',
  params: _zod.z.object({
    path: _zod.z.object({
      id: _zod.z.string()
    }),
    body: _entitiesSchema.entityDefinitionUpdateSchema
  }),
  handler: async ({
    request,
    response,
    params,
    logger,
    getScopedClient
  }) => {
    const entityClient = await getScopedClient({
      request
    });
    try {
      const updatedDefinition = await entityClient.updateEntityDefinition({
        id: params.path.id,
        definitionUpdate: params.body
      });
      return response.ok({
        body: updatedDefinition
      });
    } catch (e) {
      logger.error(e);
      if (e instanceof _entity_not_found.EntityDefinitionNotFound) {
        return response.notFound({
          body: {
            message: `Entity definition [${params.path.id}] not found`
          }
        });
      }
      if (e instanceof _entity_definition_update_conflict.EntityDefinitionUpdateConflict) {
        return response.conflict({
          body: {
            message: `Entity definition [${params.path.id}] has changes in progress`
          }
        });
      }
      if (e instanceof _entity_security_exception.EntitySecurityException || e instanceof _invalid_transform_error.InvalidTransformError) {
        return response.customError({
          body: e,
          statusCode: 400
        });
      }
      return response.customError({
        body: e,
        statusCode: 500
      });
    }
  }
});