"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.resetEntityDefinitionRoute = void 0;
var _entitiesSchema = require("@kbn/entities-schema");
var _zod = require("@kbn/zod");
var _entity_security_exception = require("../../lib/entities/errors/entity_security_exception");
var _invalid_transform_error = require("../../lib/entities/errors/invalid_transform_error");
var _read_entity_definition = require("../../lib/entities/read_entity_definition");
var _delete_ingest_pipeline = require("../../lib/entities/delete_ingest_pipeline");
var _delete_index = require("../../lib/entities/delete_index");
var _create_and_install_ingest_pipeline = require("../../lib/entities/create_and_install_ingest_pipeline");
var _create_and_install_transform = require("../../lib/entities/create_and_install_transform");
var _start_transforms = require("../../lib/entities/start_transforms");
var _entity_not_found = require("../../lib/entities/errors/entity_not_found");
var _create_entity_manager_server_route = require("../create_entity_manager_server_route");
var _delete_transforms = require("../../lib/entities/delete_transforms");
var _stop_transforms = require("../../lib/entities/stop_transforms");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const resetEntityDefinitionRoute = exports.resetEntityDefinitionRoute = (0, _create_entity_manager_server_route.createEntityManagerServerRoute)({
  endpoint: 'POST /internal/entities/definition/{id}/_reset',
  params: _zod.z.object({
    path: _entitiesSchema.resetEntityDefinitionParamsSchema
  }),
  handler: async ({
    context,
    response,
    params,
    logger
  }) => {
    try {
      const soClient = (await context.core).savedObjects.client;
      const esClient = (await context.core).elasticsearch.client.asCurrentUser;
      const definition = await (0, _read_entity_definition.readEntityDefinition)(soClient, params.path.id, logger);

      // Delete the transform and ingest pipeline
      await (0, _stop_transforms.stopTransforms)(esClient, definition, logger);
      await (0, _delete_transforms.deleteTransforms)(esClient, definition, logger);
      await (0, _delete_ingest_pipeline.deleteIngestPipelines)(esClient, definition, logger);
      await (0, _delete_index.deleteIndices)(esClient, definition, logger);

      // Recreate everything
      await (0, _create_and_install_ingest_pipeline.createAndInstallIngestPipelines)(esClient, definition, logger);
      await (0, _create_and_install_transform.createAndInstallTransforms)(esClient, definition, logger);
      await (0, _start_transforms.startTransforms)(esClient, definition, logger);
      return response.ok({
        body: {
          acknowledged: true
        }
      });
    } catch (e) {
      logger.error(e);
      if (e instanceof _entity_not_found.EntityDefinitionNotFound) {
        return response.notFound({
          body: e
        });
      }
      if (e instanceof _entity_security_exception.EntitySecurityException || e instanceof _invalid_transform_error.InvalidTransformError) {
        return response.customError({
          body: e,
          statusCode: 400
        });
      }
      return response.customError({
        body: e,
        statusCode: 500
      });
    }
  }
});