"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.checkEntityDiscoveryEnabledRoute = void 0;
var _semver = _interopRequireDefault(require("semver"));
var _errors = require("../../../common/errors");
var _auth = require("../../lib/auth");
var _built_in = require("../../lib/entities/built_in");
var _find_entity_definition = require("../../lib/entities/find_entity_definition");
var _utils = require("../../lib/utils");
var _create_entity_manager_server_route = require("../create_entity_manager_server_route");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * @openapi
 * /internal/entities/managed/enablement:
 *   get:
 *     description: Check if managed (built-in) entity discovery is enabled. Enabled entity discovery requires a valid api key and the latest version of the builtin definitions installed and running.
 *     tags:
 *       - management
 *     responses:
 *       200:
 *         description: OK - Verify result in response body
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               required: enabled
 *               properties:
 *                 enabled:
 *                  type: boolean
 *                  example: false
 *                 reason:
 *                  type: string
 *                  example: api_key_not_found
 */
const checkEntityDiscoveryEnabledRoute = exports.checkEntityDiscoveryEnabledRoute = (0, _create_entity_manager_server_route.createEntityManagerServerRoute)({
  endpoint: 'GET /internal/entities/managed/enablement',
  handler: async ({
    response,
    logger,
    server
  }) => {
    try {
      logger.debug('reading entity discovery API key from saved object');
      const apiKey = await (0, _auth.readEntityDiscoveryAPIKey)(server);
      if (apiKey === undefined) {
        return response.ok({
          body: {
            enabled: false,
            reason: _errors.ERROR_API_KEY_NOT_FOUND
          }
        });
      }
      logger.debug('validating existing entity discovery API key');
      const isValid = await (0, _auth.checkIfEntityDiscoveryAPIKeyIsValid)(server, apiKey);
      if (!isValid) {
        return response.ok({
          body: {
            enabled: false,
            reason: _errors.ERROR_API_KEY_NOT_VALID
          }
        });
      }
      const {
        clusterClient,
        soClient
      } = (0, _utils.getClientsFromAPIKey)({
        apiKey,
        server
      });
      const entityDiscoveryState = await Promise.all(_built_in.builtInDefinitions.map(async builtInDefinition => {
        const definitions = await (0, _find_entity_definition.findEntityDefinitions)({
          soClient,
          esClient: clusterClient.asSecondaryAuthUser,
          id: builtInDefinition.id,
          includeState: true
        });
        return {
          installedDefinition: definitions[0],
          builtInDefinition
        };
      })).then(results => results.reduce((state, {
        installedDefinition,
        builtInDefinition
      }) => {
        return {
          installed: Boolean(state.installed && (installedDefinition === null || installedDefinition === void 0 ? void 0 : installedDefinition.state.installed)),
          running: Boolean(state.running && (installedDefinition === null || installedDefinition === void 0 ? void 0 : installedDefinition.state.running)),
          outdated: state.outdated || installedDefinition && _semver.default.neq(installedDefinition.version, builtInDefinition.version)
        };
      }, {
        installed: true,
        running: true,
        outdated: false
      }));
      if (!entityDiscoveryState.installed) {
        return response.ok({
          body: {
            enabled: false,
            reason: _errors.ERROR_PARTIAL_BUILTIN_INSTALLATION
          }
        });
      }
      if (!entityDiscoveryState.running) {
        return response.ok({
          body: {
            enabled: false,
            reason: _errors.ERROR_DEFINITION_STOPPED
          }
        });
      }
      if (entityDiscoveryState.outdated) {
        return response.ok({
          body: {
            enabled: false,
            reason: _errors.ERROR_BUILTIN_UPGRADE_REQUIRED
          }
        });
      }
      return response.ok({
        body: {
          enabled: true
        }
      });
    } catch (err) {
      logger.error(err);
      return response.customError({
        statusCode: 500,
        body: err
      });
    }
  }
});