"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LogRetentionLogic = void 0;
var _kea = require("kea");
var _flash_messages = require("../../../shared/flash_messages");
var _http = require("../../../shared/http");
var _convert_log_retention = require("./utils/convert_log_retention");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const LogRetentionLogic = exports.LogRetentionLogic = (0, _kea.kea)({
  path: ['enterprise_search', 'app_search', 'log_retention_logic'],
  actions: () => ({
    clearLogRetentionUpdating: true,
    closeModals: true,
    fetchLogRetention: true,
    saveLogRetention: (option, enabled) => ({
      enabled,
      option
    }),
    setOpenedModal: option => ({
      option
    }),
    toggleLogRetention: option => ({
      option
    }),
    updateLogRetention: logRetention => ({
      logRetention
    })
  }),
  reducers: () => ({
    logRetention: [null, {
      // @ts-expect-error upgrade typescript v5.1.6
      updateLogRetention: (_, {
        logRetention
      }) => logRetention
    }],
    isLogRetentionUpdating: [false, {
      clearLogRetentionUpdating: () => false,
      closeModals: () => false,
      fetchLogRetention: () => true,
      toggleLogRetention: () => true
    }],
    openedModal: [null, {
      closeModals: () => null,
      saveLogRetention: () => null,
      // @ts-expect-error upgrade typescript v5.1.6
      setOpenedModal: (_, {
        option
      }) => option
    }]
  }),
  listeners: ({
    actions,
    values
  }) => ({
    fetchLogRetention: async (_, breakpoint) => {
      await breakpoint(100); // Prevents duplicate calls to the API (e.g., when a tooltip & callout are on the same page)

      try {
        const {
          http
        } = _http.HttpLogic.values;
        const response = await http.get('/internal/app_search/log_settings');
        actions.updateLogRetention((0, _convert_log_retention.convertLogRetentionFromServerToClient)(response));
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      } finally {
        actions.clearLogRetentionUpdating();
      }
    },
    saveLogRetention: async ({
      enabled,
      option
    }) => {
      const updateData = {
        [option]: {
          enabled
        }
      };
      try {
        const {
          http
        } = _http.HttpLogic.values;
        const response = await http.put('/internal/app_search/log_settings', {
          body: JSON.stringify(updateData)
        });
        actions.updateLogRetention((0, _convert_log_retention.convertLogRetentionFromServerToClient)(response));
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      } finally {
        actions.clearLogRetentionUpdating();
      }
    },
    toggleLogRetention: ({
      option
    }) => {
      var _values$logRetention;
      const logRetention = (_values$logRetention = values.logRetention) === null || _values$logRetention === void 0 ? void 0 : _values$logRetention[option];

      // If the user has found a way to call this before we've retrieved
      // log retention settings from the server, short circuit this and return early
      if (!logRetention) {
        return;
      }
      const optionIsAlreadyEnabled = logRetention.enabled;
      if (optionIsAlreadyEnabled) {
        actions.setOpenedModal(option);
      } else {
        actions.saveLogRetention(option, true);
      }
    }
  })
});