"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EngineCreationSteps = exports.EngineCreationLogic = void 0;
var _kea = require("kea");
var _i18n = require("@kbn/i18n");
var _flash_messages = require("../../../shared/flash_messages");
var _http = require("../../../shared/http");
var _kibana = require("../../../shared/kibana");
var _format_api_name = require("../../utils/format_api_name");
var _constants = require("./constants");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
let EngineCreationSteps = exports.EngineCreationSteps = /*#__PURE__*/function (EngineCreationSteps) {
  EngineCreationSteps["SelectStep"] = "Select Engine Type";
  EngineCreationSteps["ConfigureStep"] = "Configure Engine";
  EngineCreationSteps["ReviewStep"] = "Review";
  return EngineCreationSteps;
}({});
const EngineCreationLogic = exports.EngineCreationLogic = (0, _kea.kea)({
  path: ['enterprise_search', 'app_search', 'engine_creation_logic'],
  actions: {
    onEngineCreationSuccess: true,
    setIngestionMethod: method => ({
      method
    }),
    setLanguage: language => ({
      language
    }),
    setRawName: rawName => ({
      rawName
    }),
    setAliasRawName: aliasRawName => ({
      aliasRawName
    }),
    submitEngine: true,
    onSubmitError: true,
    loadIndices: true,
    onLoadIndicesSuccess: indices => ({
      indices
    }),
    setSelectedIndex: selectedIndexName => ({
      selectedIndexName
    }),
    setEngineType: engineType => ({
      engineType
    }),
    setCreationStep: currentEngineCreationStep => currentEngineCreationStep,
    setIsAliasAllowed: isAliasAllowed => ({
      isAliasAllowed
    }),
    initializeWithESIndex: indexName => ({
      indexName
    })
  },
  reducers: {
    ingestionMethod: ['', {
      // @ts-expect-error upgrade typescript v5.1.6
      setIngestionMethod: (_, {
        method
      }) => method
    }],
    isLoading: [false, {
      submitEngine: () => true,
      onSubmitError: () => false
    }],
    language: [_constants.DEFAULT_LANGUAGE, {
      // @ts-expect-error upgrade typescript v5.1.6
      setLanguage: (_, {
        language
      }) => language
    }],
    rawName: ['', {
      // @ts-expect-error upgrade typescript v5.1.6
      setRawName: (_, {
        rawName
      }) => rawName
    }],
    aliasRawName: ['', {
      // @ts-expect-error upgrade typescript v5.1.6
      setAliasRawName: (_, {
        aliasRawName
      }) => aliasRawName,
      // @ts-expect-error upgrade typescript v5.1.6
      setSelectedIndex: (_, {
        selectedIndexName
      }) => {
        return selectedIndexName.length === 0 || selectedIndexName.startsWith('search-') ? '' : `search-${selectedIndexName}-alias`;
      },
      // @ts-expect-error upgrade typescript v5.1.6
      initializeWithESIndex: (_, {
        indexName
      }) => indexName.length === 0 || indexName.startsWith('search-') ? '' : `search-${indexName}-alias`
    }],
    isAliasAllowed: [true, {
      // @ts-expect-error upgrade typescript v5.1.6
      setIsAliasAllowed: (_, {
        isAliasAllowed
      }) => isAliasAllowed
    }],
    isLoadingIndices: [false, {
      loadIndices: () => true,
      onLoadIndicesSuccess: () => false,
      onSubmitError: () => false
    }],
    indices: [[], {
      // @ts-expect-error upgrade typescript v5.1.6
      onLoadIndicesSuccess: (_, {
        indices
      }) => indices
    }],
    selectedIndex: ['', {
      // @ts-expect-error upgrade typescript v5.1.6
      setSelectedIndex: (_, {
        selectedIndexName
      }) => selectedIndexName,
      onSubmitError: () => '',
      // @ts-expect-error upgrade typescript v5.1.6
      initializeWithESIndex: (_, {
        indexName
      }) => indexName
    }],
    engineType: ['appSearch', {
      // @ts-expect-error upgrade typescript v5.1.6
      setEngineType: (_, {
        engineType
      }) => engineType,
      // @ts-expect-error upgrade typescript v5.1.6
      initializeWithESIndex: () => 'elasticsearch'
    }],
    currentEngineCreationStep: [EngineCreationSteps.SelectStep, {
      // @ts-expect-error upgrade typescript v5.1.6
      setCreationStep: (_, currentEngineCreationStep) => currentEngineCreationStep,
      initializeWithESIndex: () => EngineCreationSteps.ConfigureStep
    }]
  },
  selectors: ({
    selectors
  }) => ({
    name: [() => [selectors.rawName], rawName => (0, _format_api_name.formatApiName)(rawName)],
    aliasName: [() => [selectors.aliasRawName], aliasRawName => (0, _format_api_name.formatApiName)(aliasRawName)],
    indicesFormatted: [() => [selectors.indices, selectors.selectedIndex], (indices, selectedIndexName) => (0, _utils.formatIndicesToSelectable)(indices, selectedIndexName)],
    isSubmitDisabled: [() => [selectors.name, selectors.engineType, selectors.selectedIndex, selectors.aliasName, selectors.showAliasNameErrorMessages], (name, engineType, selectedIndex, aliasName, showAliasNameErrorMessages) => {
      if (name.length === 0 || showAliasNameErrorMessages) {
        return true;
      }
      if (engineType === 'elasticsearch') {
        if (selectedIndex.length === 0) {
          return true;
        }
        if (aliasName.length === 0) {
          return !selectedIndex.startsWith('search-');
        } else {
          return !aliasName.startsWith('search-');
        }
      }
      return false;
    }],
    isAliasRequired: [() => [selectors.selectedIndex], selectedIndex => selectedIndex.length > 0 && !(selectedIndex !== null && selectedIndex !== void 0 && selectedIndex.startsWith('search-'))],
    selectedIndexFormatted: [() => [selectors.selectedIndex, selectors.indicesFormatted], (selectedIndex, indicesFormatted) => {
      return indicesFormatted.find(el => el.label === selectedIndex);
    }],
    aliasNameErrorMessage: [() => [selectors.aliasName, selectors.indices], (aliasName, indices) => {
      const existingAlias = indices.find(el => el.name === aliasName);
      if (existingAlias) {
        return _i18n.i18n.translate('xpack.enterpriseSearch.appSearch.engineCreation.configureForm.aliasName.errorText', {
          // ugly, but cannot use dedent here and pass Kibana's Checks
          defaultMessage: `
There is an existing index or alias with the name {aliasName}.
Please choose another alias name.
`,
          values: {
            aliasName
          }
        });
      } else {
        return '';
      }
    }],
    showAliasNameErrorMessages: [() => [selectors.aliasNameErrorMessage], aliasNameErrorMessage => aliasNameErrorMessage.length > 0]
  }),
  listeners: ({
    values,
    actions
  }) => ({
    submitEngine: async () => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        name,
        language,
        engineType,
        selectedIndex,
        aliasName
      } = values;
      try {
        if (engineType === 'appSearch') {
          const body = JSON.stringify({
            name,
            language
          });
          await http.post('/internal/app_search/engines', {
            body
          });
        } else {
          const body = JSON.stringify({
            name,
            search_index: {
              type: 'elasticsearch',
              index_name: selectedIndex,
              ...(aliasName.length === 0 ? {} : {
                alias_name: aliasName
              })
            }
          });
          await http.post('/internal/app_search/elasticsearch/engines', {
            body
          });
        }
        actions.onEngineCreationSuccess();
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
        actions.onSubmitError();
      }
    },
    onEngineCreationSuccess: () => {
      const {
        ingestionMethod,
        name
      } = values;
      const {
        navigateToUrl
      } = _kibana.KibanaLogic.values;
      const toUrl = (0, _utils.getRedirectToAfterEngineCreation)({
        ingestionMethod,
        engineName: name
      });
      (0, _flash_messages.flashSuccessToast)((0, _constants.ENGINE_CREATION_SUCCESS_MESSAGE)(name));
      navigateToUrl(toUrl);
    },
    loadIndices: async () => {
      const {
        http
      } = _http.HttpLogic.values;
      try {
        const indices = await http.get('/internal/enterprise_search/search_indices');
        actions.onLoadIndicesSuccess(indices);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
        actions.onSubmitError();
      }
    }
  })
});