"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EncryptedSavedObjectAttributesDefinition = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Represents the definition of the attributes of the specific saved object that are supposed to be
 * encrypted. The definition also dictates which attributes should be included in AAD and/or
 * stripped from response.
 */
class EncryptedSavedObjectAttributesDefinition {
  constructor(typeRegistration) {
    (0, _defineProperty2.default)(this, "attributesToEncrypt", void 0);
    (0, _defineProperty2.default)(this, "attributesToIncludeInAAD", void 0);
    (0, _defineProperty2.default)(this, "attributesToStrip", void 0);
    if (typeRegistration.attributesToIncludeInAAD) {
      const invalidAttributeKeys = new Array();
      typeRegistration.attributesToEncrypt.forEach(attribute => {
        var _typeRegistration$att;
        const attributeKey = typeof attribute !== 'string' ? attribute.key : attribute;
        if ((_typeRegistration$att = typeRegistration.attributesToIncludeInAAD) !== null && _typeRegistration$att !== void 0 && _typeRegistration$att.has(attributeKey)) {
          invalidAttributeKeys.push(attributeKey);
        }
      });
      if (invalidAttributeKeys.length > 0) {
        throw new Error(`Invalid EncryptedSavedObjectTypeRegistration for type '${typeRegistration.type}'. ` + `attributesToIncludeInAAD must not contain any values in attributesToEncrypt: ${invalidAttributeKeys}`);
      }
    }
    const attributesToEncrypt = new Set();
    const attributesToStrip = new Set();
    for (const attribute of typeRegistration.attributesToEncrypt) {
      if (typeof attribute === 'string') {
        attributesToEncrypt.add(attribute);
        attributesToStrip.add(attribute);
      } else {
        attributesToEncrypt.add(attribute.key);
        if (!attribute.dangerouslyExposeValue) {
          attributesToStrip.add(attribute.key);
        }
      }
    }
    this.attributesToEncrypt = attributesToEncrypt;
    this.attributesToStrip = attributesToStrip;
    this.attributesToIncludeInAAD = typeRegistration.attributesToIncludeInAAD;
  }

  /**
   * Determines whether particular attribute should be encrypted. Full list of attributes that
   * should be encrypted can be retrieved via `attributesToEncrypt` property.
   * @param attributeName Name of the attribute.
   */
  shouldBeEncrypted(attributeName) {
    return this.attributesToEncrypt.has(attributeName);
  }

  /**
   * Determines whether particular attribute should be included in AAD.
   * @param attributeName Name of the attribute.
   */
  shouldBeIncludedInAAD(attributeName) {
    return !this.shouldBeEncrypted(attributeName) && this.attributesToIncludeInAAD != null && this.attributesToIncludeInAAD.has(attributeName);
  }

  /**
   * Determines whether particular attribute should be stripped from the attribute list.
   * @param attributeName Name of the attribute.
   */
  shouldBeStripped(attributeName) {
    return this.attributesToStrip.has(attributeName);
  }

  /**
   * Collects all attributes (both keys and values) that should contribute to AAD.
   * @param attributes Attributes of the saved object
   */
  collectAttributesForAAD(attributes) {
    const aadAttributes = {};
    if (this.attributesToIncludeInAAD) {
      for (const attributeKey of this.attributesToIncludeInAAD) {
        if (!this.shouldBeEncrypted(attributeKey) && Object.hasOwn(attributes, attributeKey)) {
          aadAttributes[attributeKey] = attributes[attributeKey];
        }
      }
    }
    return aadAttributes;
  }
}
exports.EncryptedSavedObjectAttributesDefinition = EncryptedSavedObjectAttributesDefinition;