"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.postKnowledgeBaseRoute = void 0;
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _common = require("@kbn/elastic-assistant-common/impl/schemas/common");
var _build_response = require("../../lib/build_response");
var _helpers = require("../helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Since we're awaiting on ELSER setup, this could take a bit (especially if ML needs to autoscale)
// Consider just returning if attempt was successful, and switch to client polling
const ROUTE_HANDLER_TIMEOUT = 10 * 60 * 1000; // 10 * 60 seconds = 10 minutes

/**
 * Load Knowledge Base index, pipeline, and resources (collection of documents)
 * @param router
 */
const postKnowledgeBaseRoute = router => {
  router.versioned.post({
    access: 'internal',
    path: _elasticAssistantCommon.ELASTIC_AI_ASSISTANT_KNOWLEDGE_BASE_URL,
    options: {
      tags: ['access:elasticAssistant'],
      timeout: {
        idleSocket: ROUTE_HANDLER_TIMEOUT
      }
    }
  }).addVersion({
    version: _elasticAssistantCommon.ELASTIC_AI_ASSISTANT_INTERNAL_API_VERSION,
    validate: {
      request: {
        params: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.CreateKnowledgeBaseRequestParams),
        query: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.CreateKnowledgeBaseRequestQuery)
      }
    }
  }, async (context, request, response) => {
    const resp = (0, _build_response.buildResponse)(response);
    const ctx = await context.resolve(['core', 'elasticAssistant', 'licensing']);
    const assistantContext = ctx.elasticAssistant;
    const core = ctx.core;
    const soClient = core.savedObjects.getClient();

    // FF Check for V2 KB
    const v2KnowledgeBaseEnabled = (0, _helpers.isV2KnowledgeBaseEnabled)({
      context: ctx,
      request
    });
    // Only allow modelId override if FF is enabled as this will re-write the ingest pipeline and break any previous KB entries
    // This is only really needed for API integration tests
    const modelIdOverride = v2KnowledgeBaseEnabled ? request.query.modelId : undefined;
    const ignoreSecurityLabs = request.query.ignoreSecurityLabs;
    try {
      const knowledgeBaseDataClient = await assistantContext.getAIAssistantKnowledgeBaseDataClient({
        modelIdOverride,
        v2KnowledgeBaseEnabled
      });
      if (!knowledgeBaseDataClient) {
        return response.custom({
          body: {
            success: false
          },
          statusCode: 500
        });
      }
      await knowledgeBaseDataClient.setupKnowledgeBase({
        soClient,
        v2KnowledgeBaseEnabled,
        ignoreSecurityLabs
      });
      return response.ok({
        body: {
          success: true
        }
      });
    } catch (error) {
      return resp.error({
        body: error.message,
        statusCode: 500
      });
    }
  });
};
exports.postKnowledgeBaseRoute = postKnowledgeBaseRoute;