"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getKnowledgeBaseStatusRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _common = require("@kbn/elastic-assistant-common/impl/schemas/common");
var _build_response = require("../../lib/build_response");
var _helpers = require("../helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Get the status of the Knowledge Base index, pipeline, and resources (collection of documents)
 *
 * @param router IRouter for registering routes
 */
const getKnowledgeBaseStatusRoute = router => {
  router.versioned.get({
    access: 'internal',
    path: _elasticAssistantCommon.ELASTIC_AI_ASSISTANT_KNOWLEDGE_BASE_URL,
    options: {
      tags: ['access:elasticAssistant']
    }
  }).addVersion({
    version: _elasticAssistantCommon.ELASTIC_AI_ASSISTANT_INTERNAL_API_VERSION,
    validate: {
      request: {
        params: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.ReadKnowledgeBaseRequestParams)
      }
    }
  }, async (context, request, response) => {
    const resp = (0, _build_response.buildResponse)(response);
    const ctx = await context.resolve(['core', 'elasticAssistant', 'licensing']);
    const assistantContext = ctx.elasticAssistant;
    const logger = ctx.elasticAssistant.logger;
    try {
      // FF Check for V2 KB
      const v2KnowledgeBaseEnabled = (0, _helpers.isV2KnowledgeBaseEnabled)({
        context: ctx,
        request
      });
      const kbDataClient = await assistantContext.getAIAssistantKnowledgeBaseDataClient({
        v2KnowledgeBaseEnabled
      });
      if (!kbDataClient) {
        return response.custom({
          body: {
            success: false
          },
          statusCode: 500
        });
      }
      const indexExists = true; // Installed at startup, always true
      const pipelineExists = true; // Installed at startup, always true
      const modelExists = await kbDataClient.isModelInstalled();
      const setupAvailable = await kbDataClient.isSetupAvailable();
      const isModelDeployed = await kbDataClient.isModelDeployed();
      const body = {
        elser_exists: modelExists,
        index_exists: indexExists,
        is_setup_in_progress: kbDataClient.isSetupInProgress,
        is_setup_available: setupAvailable,
        pipeline_exists: pipelineExists
      };
      if (indexExists && isModelDeployed) {
        const securityLabsExists = v2KnowledgeBaseEnabled ? await kbDataClient.isSecurityLabsDocsLoaded() : true;
        const userDataExists = v2KnowledgeBaseEnabled ? await kbDataClient.isUserDataExists() : true;
        return response.ok({
          body: {
            ...body,
            security_labs_exists: securityLabsExists,
            user_data_exists: userDataExists
          }
        });
      }
      return response.ok({
        body
      });
    } catch (err) {
      logger.error(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return resp.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.getKnowledgeBaseStatusRoute = getKnowledgeBaseStatusRoute;