"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateConversationWithUserInput = exports.performChecks = exports.langChainExecute = exports.isV2KnowledgeBaseEnabled = exports.hasAIAssistantLicense = exports.getSystemPromptFromUserConversation = exports.getPluginNameFromRequest = exports.getMessageFromRawResponse = exports.getIsKnowledgeBaseInstalled = exports.generateTitleForNewChatConversation = exports.createConversationWithUserInput = exports.appendMessageToConversation = exports.appendAssistantMessageToConversation = exports.UPGRADE_LICENSE_MESSAGE = exports.NEW_CHAT = exports.DEFAULT_PLUGIN_NAME = void 0;
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _i18n = require("@kbn/i18n");
var _langsmith = require("@kbn/langchain/server/tracers/langsmith");
var _event_based_telemetry = require("../lib/telemetry/event_based_telemetry");
var _constants = require("../../common/constants");
var _constants2 = require("./knowledge_base/constants");
var _utils = require("./utils");
var _executor = require("../lib/executor");
var _helpers = require("../lib/langchain/helpers");
var _default_assistant_graph = require("../lib/langchain/graphs/default_assistant_graph");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_PLUGIN_NAME = exports.DEFAULT_PLUGIN_NAME = 'securitySolutionUI';
const NEW_CHAT = exports.NEW_CHAT = _i18n.i18n.translate('xpack.elasticAssistantPlugin.server.newChat', {
  defaultMessage: 'New chat'
});

/**
 * Attempts to extract the plugin name the request originated from using the request headers.
 *
 * Note from Kibana Core: This is not a 100% fit solution, though, because plugins can run in the background,
 * or even use other plugins’ helpers (ie, APM can use the infra helpers to call a third plugin)
 *
 * Should suffice for our purposes here with where the Elastic Assistant is currently used, but if needing a
 * dedicated solution, the core folks said to reach out.
 *
 * @param logger optional logger to log any errors
 * @param defaultPluginName default plugin name to use if unable to determine from request
 * @param request Kibana Request
 *
 * @returns plugin name
 */
const getPluginNameFromRequest = ({
  logger,
  defaultPluginName,
  request
}) => {
  try {
    const contextHeader = request.headers['x-kbn-context'];
    if (contextHeader != null) {
      var _JSON$parse;
      return (_JSON$parse = JSON.parse(decodeURIComponent(Array.isArray(contextHeader) ? contextHeader[0] : contextHeader))) === null || _JSON$parse === void 0 ? void 0 : _JSON$parse.name;
    }
  } catch (err) {
    logger === null || logger === void 0 ? void 0 : logger.error(`Error determining source plugin for selecting tools, using ${defaultPluginName}.`);
  }
  return defaultPluginName;
};
exports.getPluginNameFromRequest = getPluginNameFromRequest;
const getMessageFromRawResponse = ({
  rawContent,
  isError,
  traceData
}) => {
  const dateTimeString = new Date().toISOString();
  if (rawContent) {
    return {
      role: 'assistant',
      content: rawContent,
      timestamp: dateTimeString,
      isError,
      traceData
    };
  } else {
    return {
      role: 'assistant',
      content: 'Error: Response from LLM API is empty or undefined.',
      timestamp: dateTimeString,
      isError: true
    };
  }
};
exports.getMessageFromRawResponse = getMessageFromRawResponse;
const hasAIAssistantLicense = license => license.hasAtLeast(_constants.MINIMUM_AI_ASSISTANT_LICENSE);
exports.hasAIAssistantLicense = hasAIAssistantLicense;
const UPGRADE_LICENSE_MESSAGE = exports.UPGRADE_LICENSE_MESSAGE = 'Your license does not support AI Assistant. Please upgrade your license.';
const generateTitleForNewChatConversation = async ({
  message,
  model,
  actionTypeId,
  connectorId,
  logger,
  actionsClient,
  responseLanguage = 'English'
}) => {
  try {
    const autoTitle = await (0, _executor.executeAction)({
      actionsClient,
      connectorId,
      actionTypeId,
      params: {
        subAction: 'invokeAI',
        subActionParams: {
          model,
          messages: [{
            role: 'system',
            content: `You are a helpful assistant for Elastic Security. Assume the following message is the start of a conversation between you and a user; give this conversation a title based on the content below. DO NOT UNDER ANY CIRCUMSTANCES wrap this title in single or double quotes. This title is shown in a list of conversations to the user, so title it for the user, not for you. Please create the title in ${responseLanguage}.`
          }, {
            role: message.role,
            content: message.content
          }],
          ...(actionTypeId === '.gen-ai' ? {
            n: 1,
            stop: null,
            temperature: 0.2
          } : {
            temperature: 0,
            stopSequences: []
          })
        }
      },
      logger
    }); // TODO: Use function overloads in executeAction to avoid this cast when sending subAction: 'invokeAI',
    if (autoTitle.status === 'ok') {
      // This regular expression captures a string enclosed in single or double quotes.
      // It extracts the string content without the quotes.
      // Example matches:
      // - "Hello, World!" => Captures: Hello, World!
      // - 'Another Example' => Captures: Another Example
      // - JustTextWithoutQuotes => Captures: JustTextWithoutQuotes
      const match = autoTitle.data.match(/^["']?([^"']+)["']?$/);
      const title = match ? match[1] : autoTitle.data;
      return title;
    }
  } catch (e) {
    /* empty */
  }
};
exports.generateTitleForNewChatConversation = generateTitleForNewChatConversation;
const appendMessageToConversation = async ({
  conversationsDataClient,
  messages,
  replacements,
  conversation
}) => {
  const updatedConversation = await (conversationsDataClient === null || conversationsDataClient === void 0 ? void 0 : conversationsDataClient.appendConversationMessages({
    existingConversation: conversation,
    messages: messages.map(m => {
      var _m$role;
      return {
        ...{
          content: (0, _elasticAssistantCommon.replaceAnonymizedValuesWithOriginalValues)({
            messageContent: m.content,
            replacements
          }),
          role: (_m$role = m.role) !== null && _m$role !== void 0 ? _m$role : 'user'
        },
        timestamp: new Date().toISOString()
      };
    })
  }));
  return updatedConversation;
};
exports.appendMessageToConversation = appendMessageToConversation;
const extractPromptFromESResult = result => {
  if (result.total > 0 && result.data.hits.hits.length > 0) {
    var _result$data$hits$hit;
    return (_result$data$hits$hit = result.data.hits.hits[0]._source) === null || _result$data$hits$hit === void 0 ? void 0 : _result$data$hits$hit.content;
  }
  return undefined;
};
const getSystemPromptFromUserConversation = async ({
  conversationsDataClient,
  conversationId,
  promptsDataClient
}) => {
  var _conversation$apiConf;
  const conversation = await conversationsDataClient.getConversation({
    id: conversationId
  });
  if (!conversation) {
    return undefined;
  }
  const currentSystemPromptId = (_conversation$apiConf = conversation.apiConfig) === null || _conversation$apiConf === void 0 ? void 0 : _conversation$apiConf.defaultSystemPromptId;
  if (!currentSystemPromptId) {
    return undefined;
  }
  const result = await promptsDataClient.findDocuments({
    perPage: 1,
    page: 1,
    filter: `_id: "${currentSystemPromptId}"`
  });
  return extractPromptFromESResult(result);
};
exports.getSystemPromptFromUserConversation = getSystemPromptFromUserConversation;
const appendAssistantMessageToConversation = async ({
  conversationsDataClient,
  messageContent,
  replacements,
  conversationId,
  isError = false,
  traceData = {}
}) => {
  const conversation = await conversationsDataClient.getConversation({
    id: conversationId
  });
  if (!conversation) {
    return;
  }
  await conversationsDataClient.appendConversationMessages({
    existingConversation: conversation,
    messages: [getMessageFromRawResponse({
      rawContent: (0, _elasticAssistantCommon.replaceAnonymizedValuesWithOriginalValues)({
        messageContent,
        replacements
      }),
      traceData,
      isError
    })]
  });
  if (Object.keys(replacements).length > 0) {
    await (conversationsDataClient === null || conversationsDataClient === void 0 ? void 0 : conversationsDataClient.updateConversation({
      conversationUpdateProps: {
        id: conversation.id,
        replacements
      }
    }));
  }
};
exports.appendAssistantMessageToConversation = appendAssistantMessageToConversation;
const langChainExecute = async ({
  messages,
  replacements,
  onNewReplacements,
  abortSignal,
  telemetry,
  actionTypeId,
  connectorId,
  isOssModel,
  context,
  actionsClient,
  inference,
  request,
  logger,
  conversationId,
  onLlmResponse,
  getElser,
  response,
  responseLanguage,
  isStream = true,
  systemPrompt
}) => {
  var _await$assistantConte;
  // Fetch any tools registered by the request's originating plugin
  const pluginName = getPluginNameFromRequest({
    request,
    defaultPluginName: DEFAULT_PLUGIN_NAME,
    logger
  });
  const assistantContext = context.elasticAssistant;
  const assistantTools = assistantContext.getRegisteredTools(pluginName).filter(x => x.id !== 'attack-discovery'); // We don't (yet) support asking the assistant for NEW attack discoveries from a conversation
  const v2KnowledgeBaseEnabled = assistantContext.getRegisteredFeatures(pluginName).assistantKnowledgeBaseByDefault;

  // get a scoped esClient for assistant memory
  const esClient = context.core.elasticsearch.client.asCurrentUser;

  // convert the assistant messages to LangChain messages:
  const langChainMessages = (0, _helpers.getLangChainMessages)(messages);
  const anonymizationFieldsDataClient = await assistantContext.getAIAssistantAnonymizationFieldsDataClient();
  const conversationsDataClient = await assistantContext.getAIAssistantConversationsDataClient();

  // Create an ElasticsearchStore for KB interactions
  const kbDataClient = (_await$assistantConte = await assistantContext.getAIAssistantKnowledgeBaseDataClient({
    v2KnowledgeBaseEnabled
  })) !== null && _await$assistantConte !== void 0 ? _await$assistantConte : undefined;
  const dataClients = {
    anonymizationFieldsDataClient: anonymizationFieldsDataClient !== null && anonymizationFieldsDataClient !== void 0 ? anonymizationFieldsDataClient : undefined,
    conversationsDataClient: conversationsDataClient !== null && conversationsDataClient !== void 0 ? conversationsDataClient : undefined,
    kbDataClient
  };
  const isKnowledgeBaseInstalled = await getIsKnowledgeBaseInstalled(kbDataClient);
  // Shared executor params
  const executorParams = {
    abortSignal,
    dataClients,
    alertsIndexPattern: request.body.alertsIndexPattern,
    actionsClient,
    assistantTools,
    conversationId,
    connectorId,
    esClient,
    inference,
    isStream,
    llmType: (0, _utils.getLlmType)(actionTypeId),
    isOssModel,
    langChainMessages,
    logger,
    onNewReplacements,
    onLlmResponse,
    request,
    replacements,
    responseLanguage,
    size: request.body.size,
    systemPrompt,
    telemetry,
    telemetryParams: {
      actionTypeId,
      model: request.body.model,
      assistantStreamingEnabled: isStream,
      isEnabledKnowledgeBase: isKnowledgeBaseInstalled,
      eventType: _event_based_telemetry.INVOKE_ASSISTANT_SUCCESS_EVENT.eventType
    },
    traceOptions: {
      projectName: request.body.langSmithProject,
      tracers: (0, _langsmith.getLangSmithTracer)({
        apiKey: request.body.langSmithApiKey,
        projectName: request.body.langSmithProject,
        logger
      })
    }
  };
  const result = await (0, _default_assistant_graph.callAssistantGraph)(executorParams);
  return response.ok(result);
};
exports.langChainExecute = langChainExecute;
const createConversationWithUserInput = async ({
  conversationsDataClient,
  replacements,
  conversationId,
  actionTypeId,
  promptId,
  connectorId,
  newMessages,
  model
}) => {
  if (!conversationId) {
    if (newMessages && newMessages.length > 0) {
      return conversationsDataClient.createConversation({
        conversation: {
          title: NEW_CHAT,
          messages: newMessages.map(m => ({
            content: m.content,
            role: m.role,
            timestamp: new Date().toISOString()
          })),
          replacements,
          apiConfig: {
            connectorId,
            actionTypeId,
            model,
            defaultSystemPromptId: promptId
          }
        }
      });
    }
  }
};
exports.createConversationWithUserInput = createConversationWithUserInput;
const updateConversationWithUserInput = async ({
  logger,
  conversationsDataClient,
  replacements,
  conversationId,
  actionTypeId,
  connectorId,
  actionsClient,
  newMessages,
  model
}) => {
  var _updatedConversation, _updatedConversation$, _newMessages$;
  const conversation = await (conversationsDataClient === null || conversationsDataClient === void 0 ? void 0 : conversationsDataClient.getConversation({
    id: conversationId
  }));
  if (conversation == null) {
    throw new Error(`conversation id: "${conversationId}" not found`);
  }
  let updatedConversation = conversation;
  const messages = (_updatedConversation = updatedConversation) === null || _updatedConversation === void 0 ? void 0 : (_updatedConversation$ = _updatedConversation.messages) === null || _updatedConversation$ === void 0 ? void 0 : _updatedConversation$.map(c => ({
    role: c.role,
    content: c.content,
    timestamp: c.timestamp
  }));
  const lastMessage = (_newMessages$ = newMessages === null || newMessages === void 0 ? void 0 : newMessages[0]) !== null && _newMessages$ !== void 0 ? _newMessages$ : messages === null || messages === void 0 ? void 0 : messages[0];
  if ((conversation === null || conversation === void 0 ? void 0 : conversation.title) === NEW_CHAT && lastMessage) {
    const title = await generateTitleForNewChatConversation({
      message: lastMessage,
      actionsClient,
      actionTypeId,
      connectorId,
      logger,
      model
    });
    const res = await conversationsDataClient.updateConversation({
      conversationUpdateProps: {
        id: conversationId,
        title
      }
    });
    if (res) {
      updatedConversation = res;
    }
  }
  if (newMessages) {
    return appendMessageToConversation({
      conversation: updatedConversation,
      conversationsDataClient,
      messages: newMessages,
      replacements
    });
  }
  return updatedConversation;
};

/**
 * Helper to perform checks for authenticated user, license, and optionally capability.
 * Check order is license, authenticated user, then capability.
 *
 * Returns either a successful check with an AuthenticatedUser or
 * an unsuccessful check with an error IKibanaResponse.
 *
 * @param capability - Specific capability to check if enabled, e.g. `assistantModelEvaluation`
 * @param context - Route context
 * @param request - Route KibanaRequest
 * @param response - Route KibanaResponseFactory
 * @returns PerformChecks
 */
exports.updateConversationWithUserInput = updateConversationWithUserInput;
const performChecks = ({
  capability,
  context,
  request,
  response
}) => {
  const assistantResponse = (0, _utils.buildResponse)(response);
  if (!hasAIAssistantLicense(context.licensing.license)) {
    return {
      isSuccess: false,
      response: response.forbidden({
        body: {
          message: UPGRADE_LICENSE_MESSAGE
        }
      })
    };
  }
  const currentUser = context.elasticAssistant.getCurrentUser();
  if (currentUser == null) {
    return {
      isSuccess: false,
      response: assistantResponse.error({
        body: `Authenticated user not found`,
        statusCode: 401
      })
    };
  }
  if (capability) {
    const pluginName = getPluginNameFromRequest({
      request,
      defaultPluginName: DEFAULT_PLUGIN_NAME
    });
    const registeredFeatures = context.elasticAssistant.getRegisteredFeatures(pluginName);
    if (!registeredFeatures[capability]) {
      return {
        isSuccess: false,
        response: response.notFound()
      };
    }
  }
  return {
    isSuccess: true,
    currentUser
  };
};

/**
 * Returns whether the v2 KB is enabled
 *
 * @param context - Route context
 * @param request - Route KibanaRequest

 */
exports.performChecks = performChecks;
const isV2KnowledgeBaseEnabled = ({
  context,
  request
}) => {
  const pluginName = getPluginNameFromRequest({
    request,
    defaultPluginName: DEFAULT_PLUGIN_NAME
  });
  return context.elasticAssistant.getRegisteredFeatures(pluginName).assistantKnowledgeBaseByDefault;
};

/**
 * Telemetry function to determine whether knowledge base has been installed
 * @param kbDataClient
 */
exports.isV2KnowledgeBaseEnabled = isV2KnowledgeBaseEnabled;
const getIsKnowledgeBaseInstalled = async kbDataClient => {
  let securityLabsDocsExist = false;
  let isModelDeployed = false;
  if (kbDataClient != null) {
    try {
      isModelDeployed = await kbDataClient.isModelDeployed();
      if (isModelDeployed) {
        securityLabsDocsExist = (await kbDataClient.getKnowledgeBaseDocumentEntries({
          kbResource: _constants2.SECURITY_LABS_RESOURCE,
          query: _constants2.SECURITY_LABS_LOADED_QUERY
        })).length > 0;
      }
    } catch (e) {
      /* if telemetry related requests fail, fallback to default values */
    }
  }
  return isModelDeployed && securityLabsDocsExist;
};
exports.getIsKnowledgeBaseInstalled = getIsKnowledgeBaseInstalled;