"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchLangSmithDatasets = exports.fetchLangSmithDataset = void 0;
var _langsmith = require("langsmith");
var _langsmith2 = require("@kbn/langchain/server/tracers/langsmith");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Fetches a dataset from LangSmith. Note that `client` will use env vars unless langSmithApiKey is specified
 *
 * @param datasetName
 * @param logger
 * @param langSmithApiKey
 */
const fetchLangSmithDataset = async (datasetName, logger, langSmithApiKey) => {
  if (datasetName === undefined || langSmithApiKey == null && !(0, _langsmith2.isLangSmithEnabled)()) {
    throw new Error('LangSmith dataset name not provided or LangSmith not enabled');
  }
  try {
    const client = new _langsmith.Client({
      apiKey: langSmithApiKey
    });
    const examples = [];
    for await (const example of client.listExamples({
      datasetName
    })) {
      examples.push(example);
    }
    return examples;
  } catch (e) {
    logger.error(`Error fetching dataset from LangSmith: ${e.message}`);
    return [];
  }
};

/**
 * Fetches all LangSmith datasets.  Note that `client` will use env vars unless langSmithApiKey is specified
 *
 * @param logger
 * @param langSmithApiKey
 */
exports.fetchLangSmithDataset = fetchLangSmithDataset;
const fetchLangSmithDatasets = async ({
  logger,
  langSmithApiKey
}) => {
  try {
    const client = new _langsmith.Client({
      apiKey: langSmithApiKey
    });
    const datasets = [];
    for await (const dataset of client.listDatasets()) {
      datasets.push(dataset);
    }
    return datasets.map(d => d.name).sort();
  } catch (e) {
    logger.error(`Error fetching datasets from LangSmith: ${e.message}`);
    return [];
  }
};
exports.fetchLangSmithDatasets = fetchLangSmithDatasets;