"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.postAttackDiscoveryRoute = void 0;
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _common = require("@kbn/elastic-assistant-common/impl/schemas/common");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _moment = _interopRequireDefault(require("moment/moment"));
var _constants = require("../../../../common/constants");
var _handle_graph_error = require("./helpers/handle_graph_error");
var _helpers = require("../helpers/helpers");
var _build_response = require("../../../lib/build_response");
var _invoke_attack_discovery_graph = require("./helpers/invoke_attack_discovery_graph");
var _request_is_valid = require("./helpers/request_is_valid");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ROUTE_HANDLER_TIMEOUT = 10 * 60 * 1000; // 10 * 60 seconds = 10 minutes
const LANG_CHAIN_TIMEOUT = ROUTE_HANDLER_TIMEOUT - 10_000; // 9 minutes 50 seconds
const CONNECTOR_TIMEOUT = LANG_CHAIN_TIMEOUT - 10_000; // 9 minutes 40 seconds

const postAttackDiscoveryRoute = router => {
  router.versioned.post({
    access: 'internal',
    path: _constants.ATTACK_DISCOVERY,
    options: {
      tags: ['access:elasticAssistant'],
      timeout: {
        idleSocket: ROUTE_HANDLER_TIMEOUT
      }
    }
  }).addVersion({
    version: _elasticAssistantCommon.ELASTIC_AI_ASSISTANT_INTERNAL_API_VERSION,
    validate: {
      request: {
        body: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.AttackDiscoveryPostRequestBody)
      },
      response: {
        200: {
          body: {
            custom: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.AttackDiscoveryPostResponse)
          }
        }
      }
    }
  }, async (context, request, response) => {
    const startTime = (0, _moment.default)(); // start timing the generation
    const resp = (0, _build_response.buildResponse)(response);
    const assistantContext = await context.elasticAssistant;
    const logger = assistantContext.logger;
    const telemetry = assistantContext.telemetry;
    try {
      // get the actions plugin start contract from the request context:
      const actions = (await context.elasticAssistant).actions;
      const actionsClient = await actions.getActionsClientWithRequest(request);
      const dataClient = await assistantContext.getAttackDiscoveryDataClient();
      const authenticatedUser = assistantContext.getCurrentUser();
      if (authenticatedUser == null) {
        return resp.error({
          body: `Authenticated user not found`,
          statusCode: 401
        });
      }
      if (!dataClient) {
        return resp.error({
          body: `Attack discovery data client not initialized`,
          statusCode: 500
        });
      }

      // get parameters from the request body
      const alertsIndexPattern = decodeURIComponent(request.body.alertsIndexPattern);
      const {
        apiConfig,
        anonymizationFields,
        langSmithApiKey,
        langSmithProject,
        replacements,
        size
      } = request.body;
      if (!(0, _request_is_valid.requestIsValid)({
        alertsIndexPattern,
        request,
        size
      })) {
        return resp.error({
          body: 'Bad Request',
          statusCode: 400
        });
      }

      // get an Elasticsearch client for the authenticated user:
      const esClient = (await context.core).elasticsearch.client.asCurrentUser;

      // callback to accumulate the latest replacements:
      let latestReplacements = {
        ...replacements
      };
      const onNewReplacements = newReplacements => {
        latestReplacements = {
          ...latestReplacements,
          ...newReplacements
        };
      };
      const {
        currentAd,
        attackDiscoveryId
      } = await (0, _helpers.updateAttackDiscoveryStatusToRunning)(dataClient, authenticatedUser, apiConfig, size);

      // Don't await the results of invoking the graph; (just the metadata will be returned from the route handler):
      (0, _invoke_attack_discovery_graph.invokeAttackDiscoveryGraph)({
        actionsClient,
        alertsIndexPattern,
        anonymizationFields,
        apiConfig,
        connectorTimeout: CONNECTOR_TIMEOUT,
        esClient,
        langSmithProject,
        langSmithApiKey,
        latestReplacements,
        logger,
        onNewReplacements,
        size
      }).then(({
        anonymizedAlerts,
        attackDiscoveries
      }) => (0, _helpers.updateAttackDiscoveries)({
        anonymizedAlerts,
        apiConfig,
        attackDiscoveries,
        attackDiscoveryId,
        authenticatedUser,
        dataClient,
        latestReplacements,
        logger,
        size,
        startTime,
        telemetry
      })).catch(err => (0, _handle_graph_error.handleGraphError)({
        apiConfig,
        attackDiscoveryId,
        authenticatedUser,
        dataClient,
        err,
        latestReplacements,
        logger,
        telemetry
      }));
      return response.ok({
        body: currentAd
      });
    } catch (err) {
      logger.error(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return resp.error({
        body: {
          success: false,
          error: error.message
        },
        statusCode: error.statusCode
      });
    }
  });
};
exports.postAttackDiscoveryRoute = postAttackDiscoveryRoute;