"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseGeminiStream = exports.handleStreamStorage = void 0;
var _server = require("@kbn/langchain/server");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const handleStreamStorage = async ({
  abortSignal,
  responseStream,
  actionTypeId,
  onMessageSent,
  logger
}) => {
  try {
    const parser = actionTypeId === '.bedrock' ? _server.parseBedrockStream : actionTypeId === '.gemini' ? parseGeminiStream : parseOpenAIStream;
    const parsedResponse = await parser(responseStream, logger, abortSignal);
    if (onMessageSent) {
      onMessageSent(parsedResponse);
    }
  } catch (e) {
    if (onMessageSent) {
      onMessageSent(`An error occurred while streaming the response:\n\n${e.message}`);
    }
  }
};
exports.handleStreamStorage = handleStreamStorage;
const parseOpenAIStream = async (stream, logger, abortSignal) => {
  let responseBody = '';
  stream.on('data', chunk => {
    responseBody += chunk.toString();
  });
  return new Promise((resolve, reject) => {
    stream.on('end', () => {
      resolve(parseOpenAIResponse(responseBody));
    });
    stream.on('error', err => {
      reject(err);
    });
    if (abortSignal) {
      abortSignal.addEventListener('abort', () => {
        stream.destroy();
        resolve(parseOpenAIResponse(responseBody));
      });
    }
  });
};
const parseOpenAIResponse = responseBody => responseBody.split('\n').filter(line => {
  return line.startsWith('data: ') && !line.endsWith('[DONE]');
}).map(line => {
  return JSON.parse(line.replace('data: ', ''));
}).filter(line => {
  return 'object' in line && line.object === 'chat.completion.chunk' && line.choices.length > 0;
}).reduce((prev, line) => {
  const msg = line.choices[0].delta;
  return prev + (msg.content || '');
}, '');
const parseGeminiStream = async (stream, logger, abortSignal) => {
  let responseBody = '';
  stream.on('data', chunk => {
    responseBody += chunk.toString();
  });
  return new Promise((resolve, reject) => {
    stream.on('end', () => {
      resolve((0, _server.parseGeminiResponse)(responseBody));
    });
    stream.on('error', err => {
      reject(err);
    });
    if (abortSignal) {
      abortSignal.addEventListener('abort', () => {
        stream.destroy();
        logger.info('Gemini stream parsing was aborted.');
        resolve((0, _server.parseGeminiResponse)(responseBody));
      });
    }
  });
};
exports.parseGeminiStream = parseGeminiStream;