"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.callAssistantGraph = void 0;
var _server = require("@kbn/langchain/server");
var _agents = require("langchain/agents");
var _apm = require("@kbn/langchain/server/tracers/apm");
var _telemetry = require("@kbn/langchain/server/tracers/telemetry");
var _utils = require("../../../../routes/utils");
var _prompts = require("./prompts");
var _graph = require("./graph");
var _helpers = require("./helpers");
var _helpers2 = require("../../../../ai_assistant_data_clients/anonymization_fields/helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const callAssistantGraph = async ({
  abortSignal,
  actionsClient,
  alertsIndexPattern,
  assistantTools = [],
  connectorId,
  conversationId,
  dataClients,
  esClient,
  inference,
  langChainMessages,
  llmType,
  isOssModel,
  logger: parentLogger,
  isStream = false,
  onLlmResponse,
  onNewReplacements,
  replacements,
  request,
  size,
  systemPrompt,
  telemetry,
  telemetryParams,
  traceOptions,
  responseLanguage = 'English'
}) => {
  var _dataClients$anonymiz, _await$dataClients$kb, _dataClients$kbDataCl, _dataClients$kbDataCl2, _traceOptions$project, _latestMessage$;
  const logger = parentLogger.get('defaultAssistantGraph');
  const isOpenAI = llmType === 'openai' && !isOssModel;
  const llmClass = (0, _utils.getLlmClass)(llmType);

  /**
   * Creates a new instance of llmClass.
   *
   * This function ensures that a new llmClass instance is created every time it is called.
   * This is necessary to avoid any potential side effects from shared state. By always
   * creating a new instance, we prevent other uses of llm from binding and changing
   * the state unintentionally. For this reason, never assign this value to a variable (ex const llm = createLlmInstance())
   */
  const createLlmInstance = () => new llmClass({
    actionsClient,
    connectorId,
    llmType,
    logger,
    // possible client model override,
    // let this be undefined otherwise so the connector handles the model
    model: request.body.model,
    // ensure this is defined because we default to it in the language_models
    // This is where the LangSmith logs (Metadata > Invocation Params) are set
    temperature: (0, _server.getDefaultArguments)(llmType).temperature,
    signal: abortSignal,
    streaming: isStream,
    // prevents the agent from retrying on failure
    // failure could be due to bad connector, we should deliver that result to the client asap
    maxRetries: 0
  });
  const anonymizationFieldsRes = await (dataClients === null || dataClients === void 0 ? void 0 : (_dataClients$anonymiz = dataClients.anonymizationFieldsDataClient) === null || _dataClients$anonymiz === void 0 ? void 0 : _dataClients$anonymiz.findDocuments({
    perPage: 1000,
    page: 1
  }));
  const anonymizationFields = anonymizationFieldsRes ? (0, _helpers2.transformESSearchToAnonymizationFields)(anonymizationFieldsRes.data) : undefined;
  const latestMessage = langChainMessages.slice(-1); // the last message

  // Check if KB is available
  const isEnabledKnowledgeBase = (_await$dataClients$kb = await (dataClients === null || dataClients === void 0 ? void 0 : (_dataClients$kbDataCl = dataClients.kbDataClient) === null || _dataClients$kbDataCl === void 0 ? void 0 : _dataClients$kbDataCl.isModelDeployed())) !== null && _await$dataClients$kb !== void 0 ? _await$dataClients$kb : false;

  // Fetch any applicable tools that the source plugin may have registered
  const assistantToolParams = {
    alertsIndexPattern,
    anonymizationFields,
    connectorId,
    esClient,
    inference,
    isEnabledKnowledgeBase,
    kbDataClient: dataClients === null || dataClients === void 0 ? void 0 : dataClients.kbDataClient,
    logger,
    onNewReplacements,
    replacements,
    request,
    size,
    telemetry
  };
  const tools = assistantTools.flatMap(tool => {
    var _tool$getTool;
    return (_tool$getTool = tool.getTool({
      ...assistantToolParams,
      llm: createLlmInstance(),
      isOssModel
    })) !== null && _tool$getTool !== void 0 ? _tool$getTool : [];
  });

  // If KB enabled, fetch for any KB IndexEntries and generate a tool for each
  if (isEnabledKnowledgeBase && dataClients !== null && dataClients !== void 0 && (_dataClients$kbDataCl2 = dataClients.kbDataClient) !== null && _dataClients$kbDataCl2 !== void 0 && _dataClients$kbDataCl2.isV2KnowledgeBaseEnabled) {
    var _dataClients$kbDataCl3;
    const kbTools = await (dataClients === null || dataClients === void 0 ? void 0 : (_dataClients$kbDataCl3 = dataClients.kbDataClient) === null || _dataClients$kbDataCl3 === void 0 ? void 0 : _dataClients$kbDataCl3.getAssistantTools({
      assistantToolParams,
      esClient
    }));
    if (kbTools) {
      tools.push(...kbTools);
    }
  }
  const agentRunnable = isOpenAI ? await (0, _agents.createOpenAIFunctionsAgent)({
    llm: createLlmInstance(),
    tools,
    prompt: (0, _prompts.formatPrompt)(_prompts.systemPrompts.openai, systemPrompt),
    streamRunnable: isStream
  }) : llmType && ['bedrock', 'gemini'].includes(llmType) ? await (0, _agents.createToolCallingAgent)({
    llm: createLlmInstance(),
    tools,
    prompt: llmType === 'bedrock' ? (0, _prompts.formatPrompt)(_prompts.systemPrompts.bedrock, systemPrompt) : (0, _prompts.formatPrompt)(_prompts.systemPrompts.gemini, systemPrompt),
    streamRunnable: isStream
  }) :
  // used with OSS models
  await (0, _agents.createStructuredChatAgent)({
    llm: createLlmInstance(),
    tools,
    prompt: (0, _prompts.formatPromptStructured)(_prompts.systemPrompts.structuredChat, systemPrompt),
    streamRunnable: isStream
  });
  const apmTracer = new _apm.APMTracer({
    projectName: (_traceOptions$project = traceOptions === null || traceOptions === void 0 ? void 0 : traceOptions.projectName) !== null && _traceOptions$project !== void 0 ? _traceOptions$project : 'default'
  }, logger);
  const telemetryTracer = telemetryParams ? new _telemetry.TelemetryTracer({
    elasticTools: assistantTools.map(({
      name
    }) => name),
    totalTools: tools.length,
    telemetry,
    telemetryParams
  }, logger) : undefined;
  const assistantGraph = (0, _graph.getDefaultAssistantGraph)({
    agentRunnable,
    dataClients,
    // we need to pass it like this or streaming does not work for bedrock
    createLlmInstance,
    logger,
    tools,
    replacements
  });
  const inputs = {
    responseLanguage,
    conversationId,
    llmType,
    isStream,
    isOssModel,
    input: (_latestMessage$ = latestMessage[0]) === null || _latestMessage$ === void 0 ? void 0 : _latestMessage$.content
  };
  if (isStream) {
    return (0, _helpers.streamGraph)({
      apmTracer,
      assistantGraph,
      inputs,
      logger,
      onLlmResponse,
      request,
      telemetryTracer,
      traceOptions
    });
  }
  const graphResponse = await (0, _helpers.invokeGraph)({
    apmTracer,
    assistantGraph,
    inputs,
    onLlmResponse,
    telemetryTracer,
    traceOptions
  });
  return {
    body: {
      connector_id: connectorId,
      data: graphResponse.output,
      trace_data: graphResponse.traceData,
      replacements,
      status: 'ok',
      ...(graphResponse.conversationId ? {
        conversationId: graphResponse.conversationId
      } : {})
    },
    headers: {
      'content-type': 'application/json'
    }
  };
};
exports.callAssistantGraph = callAssistantGraph;