"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDefaultAttackDiscoveryGraph = void 0;
var _langgraph = require("@langchain/langgraph");
var _constants = require("./constants");
var _generate_or_end = require("./edges/generate_or_end");
var _generate_or_refine_or_end = require("./edges/generate_or_refine_or_end");
var _refine_or_end = require("./edges/refine_or_end");
var _retrieve_anonymized_alerts_or_generate = require("./edges/retrieve_anonymized_alerts_or_generate");
var _state = require("./state");
var _generate = require("./nodes/generate");
var _refine = require("./nodes/refine");
var _retriever = require("./nodes/retriever");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * This function returns a compiled state graph that represents the default
 * Attack discovery graph.
 *
 * Refer to the following diagram for this graph:
 * x-pack/plugins/elastic_assistant/docs/img/default_attack_discovery_graph.png
 */
const getDefaultAttackDiscoveryGraph = ({
  alertsIndexPattern,
  anonymizationFields,
  esClient,
  llm,
  logger,
  onNewReplacements,
  replacements,
  size
}) => {
  try {
    const graphState = (0, _state.getDefaultGraphState)();

    // get nodes:
    const retrieveAnonymizedAlertsNode = (0, _retriever.getRetrieveAnonymizedAlertsNode)({
      alertsIndexPattern,
      anonymizationFields,
      esClient,
      logger,
      onNewReplacements,
      replacements,
      size
    });
    const generateNode = (0, _generate.getGenerateNode)({
      llm,
      logger
    });
    const refineNode = (0, _refine.getRefineNode)({
      llm,
      logger
    });

    // get edges:
    const generateOrEndEdge = (0, _generate_or_end.getGenerateOrEndEdge)(logger);
    const generatOrRefineOrEndEdge = (0, _generate_or_refine_or_end.getGenerateOrRefineOrEndEdge)(logger);
    const refineOrEndEdge = (0, _refine_or_end.getRefineOrEndEdge)(logger);
    const retrieveAnonymizedAlertsOrGenerateEdge = (0, _retrieve_anonymized_alerts_or_generate.getRetrieveAnonymizedAlertsOrGenerateEdge)(logger);

    // create the graph:
    const graph = new _langgraph.StateGraph({
      channels: graphState
    }).addNode(_constants.NodeType.RETRIEVE_ANONYMIZED_ALERTS_NODE, retrieveAnonymizedAlertsNode).addNode(_constants.NodeType.GENERATE_NODE, generateNode).addNode(_constants.NodeType.REFINE_NODE, refineNode).addConditionalEdges(_langgraph.START, retrieveAnonymizedAlertsOrGenerateEdge, {
      generate: _constants.NodeType.GENERATE_NODE,
      retrieve_anonymized_alerts: _constants.NodeType.RETRIEVE_ANONYMIZED_ALERTS_NODE
    }).addConditionalEdges(_constants.NodeType.RETRIEVE_ANONYMIZED_ALERTS_NODE, generateOrEndEdge, {
      end: _langgraph.END,
      generate: _constants.NodeType.GENERATE_NODE
    }).addConditionalEdges(_constants.NodeType.GENERATE_NODE, generatOrRefineOrEndEdge, {
      end: _langgraph.END,
      generate: _constants.NodeType.GENERATE_NODE,
      refine: _constants.NodeType.REFINE_NODE
    }).addConditionalEdges(_constants.NodeType.REFINE_NODE, refineOrEndEdge, {
      end: _langgraph.END,
      refine: _constants.NodeType.REFINE_NODE
    });

    // compile the graph:
    return graph.compile();
  } catch (e) {
    throw new Error(`Unable to compile AttackDiscoveryGraph\n${e}`);
  }
};
exports.getDefaultAttackDiscoveryGraph = getDefaultAttackDiscoveryGraph;