"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AIAssistantKnowledgeBaseDataClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _document = require("langchain/document");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _pRetry = _interopRequireDefault(require("p-retry"));
var _server = require("@kbn/data-views-plugin/server");
var _lodash = require("lodash");
var _ = require("..");
var _create_knowledge_base_entry = require("./create_knowledge_base_entry");
var _transforms = require("./transforms");
var _constants = require("../../routes/knowledge_base/constants");
var _helpers = require("./helpers");
var _utils = require("../../routes/knowledge_base/entries/utils");
var _security_labs_loader = require("../../lib/langchain/content_loaders/security_labs_loader");
var _field_maps_configuration = require("./field_maps_configuration");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Params for when creating KbDataClient in Request Context Factory. Useful if needing to modify
 * configuration after initial plugin start
 */

class AIAssistantKnowledgeBaseDataClient extends _.AIAssistantDataClient {
  constructor(options) {
    super(options);
    /**
     * Returns whether setup of the Knowledge Base can be performed (essentially an ML features check)
     *
     */
    (0, _defineProperty2.default)(this, "isSetupAvailable", async () => {
      // ML plugin requires request to retrieve capabilities, which are in turn scoped to the user from the request,
      // so we just test the API for a 404 instead to determine if ML is 'available'
      // TODO: expand to include memory check, see https://github.com/elastic/ml-team/issues/1208#issuecomment-2115770318
      try {
        const esClient = await this.options.elasticsearchClientPromise;
        await esClient.ml.getMemoryStats({
          human: true
        });
      } catch (error) {
        return false;
      }
      return true;
    });
    /**
     * Downloads and installs ELSER model if not already installed
     *
     * @param soClient SavedObjectsClientContract for installing ELSER so that ML SO's are in sync
     */
    (0, _defineProperty2.default)(this, "installModel", async ({
      soClient
    }) => {
      const elserId = await this.options.getElserId();
      this.options.logger.debug(`Installing ELSER model '${elserId}'...`);
      try {
        await this.options.ml
        // TODO: Potentially plumb soClient through DataClient from pluginStart
        .trainedModelsProvider({}, soClient).installElasticModel(elserId);
      } catch (error) {
        this.options.logger.error(`Error installing ELSER model '${elserId}':\n${error}`);
      }
    });
    /**
     * Returns whether ELSER is installed/ready to deploy
     *
     * @returns Promise<boolean> indicating whether the model is installed
     */
    (0, _defineProperty2.default)(this, "isModelInstalled", async () => {
      const elserId = await this.options.getElserId();
      this.options.logger.debug(`Checking if ELSER model '${elserId}' is installed...`);
      try {
        var _getResponse$trained_;
        const esClient = await this.options.elasticsearchClientPromise;
        const getResponse = await esClient.ml.getTrainedModels({
          model_id: elserId,
          include: 'definition_status'
        });
        return Boolean((_getResponse$trained_ = getResponse.trained_model_configs[0]) === null || _getResponse$trained_ === void 0 ? void 0 : _getResponse$trained_.fully_defined);
      } catch (error) {
        if (!(0, _helpers.isModelAlreadyExistsError)(error)) {
          this.options.logger.error(`Error checking if ELSER model '${elserId}' is installed:\n${error}`);
        }
        return false;
      }
    });
    /**
     * Deploy the ELSER model with default configuration
     */
    (0, _defineProperty2.default)(this, "deployModel", async () => {
      const elserId = await this.options.getElserId();
      this.options.logger.debug(`Deploying ELSER model '${elserId}'...`);
      try {
        const esClient = await this.options.elasticsearchClientPromise;
        await esClient.ml.startTrainedModelDeployment({
          model_id: elserId,
          wait_for: 'fully_allocated'
        });
      } catch (error) {
        this.options.logger.error(`Error deploying ELSER model '${elserId}':\n${error}`);
        throw new Error(`Error deploying ELSER model '${elserId}':\n${error}`);
      }
    });
    /**
     * Checks if the provided model is deployed and allocated in Elasticsearch
     *
     * @returns Promise<boolean> indicating whether the model is deployed
     */
    (0, _defineProperty2.default)(this, "isModelDeployed", async () => {
      const elserId = await this.options.getElserId();
      this.options.logger.debug(`Checking if ELSER model '${elserId}' is deployed...`);
      try {
        if (this.isV2KnowledgeBaseEnabled) {
          return await this.isInferenceEndpointExists();
        } else {
          var _getResponse$trained_2;
          const esClient = await this.options.elasticsearchClientPromise;
          const getResponse = await esClient.ml.getTrainedModelsStats({
            model_id: elserId
          });

          // For standardized way of checking deployment status see: https://github.com/elastic/elasticsearch/issues/106986
          const isReadyESS = stats => {
            var _stats$deployment_sta, _stats$deployment_sta2;
            return ((_stats$deployment_sta = stats.deployment_stats) === null || _stats$deployment_sta === void 0 ? void 0 : _stats$deployment_sta.state) === 'started' && ((_stats$deployment_sta2 = stats.deployment_stats) === null || _stats$deployment_sta2 === void 0 ? void 0 : _stats$deployment_sta2.allocation_status.state) === 'fully_allocated';
          };
          const isReadyServerless = stats => {
            var _stats$deployment_sta3, _stats$deployment_sta4;
            return (_stats$deployment_sta3 = stats.deployment_stats) === null || _stats$deployment_sta3 === void 0 ? void 0 : (_stats$deployment_sta4 = _stats$deployment_sta3.nodes) === null || _stats$deployment_sta4 === void 0 ? void 0 : _stats$deployment_sta4.some(node => node.routing_state.routing_state === 'started');
          };
          return (_getResponse$trained_2 = getResponse.trained_model_stats) === null || _getResponse$trained_2 === void 0 ? void 0 : _getResponse$trained_2.some(stats => isReadyESS(stats) || isReadyServerless(stats));
        }
      } catch (e) {
        this.options.logger.debug(`Error checking if ELSER model '${elserId}' is deployed: ${e}`);
        // Returns 404 if it doesn't exist
        return false;
      }
    });
    (0, _defineProperty2.default)(this, "isInferenceEndpointExists", async () => {
      try {
        const esClient = await this.options.elasticsearchClientPromise;
        return !!(await esClient.inference.get({
          inference_id: _field_maps_configuration.ASSISTANT_ELSER_INFERENCE_ID,
          task_type: 'sparse_embedding'
        }));
      } catch (error) {
        this.options.logger.debug(`Error checking if Inference endpoint ${_field_maps_configuration.ASSISTANT_ELSER_INFERENCE_ID} exists: ${error}`);
        return false;
      }
    });
    (0, _defineProperty2.default)(this, "createInferenceEndpoint", async () => {
      const elserId = await this.options.getElserId();
      this.options.logger.debug(`Deploying ELSER model '${elserId}'...`);
      const esClient = await this.options.elasticsearchClientPromise;
      const inferenceEndpointExists = await this.isInferenceEndpointExists();
      if (inferenceEndpointExists) {
        try {
          await esClient.inference.delete({
            inference_id: _field_maps_configuration.ASSISTANT_ELSER_INFERENCE_ID,
            // it's being used in the mapping so we need to force delete
            force: true
          });
          this.options.logger.debug(`Deleted existing inference endpoint for ELSER model '${elserId}'`);
        } catch (error) {
          this.options.logger.error(`Error deleting inference endpoint for ELSER model '${elserId}':\n${error}`);
        }
      }
      try {
        await esClient.inference.put({
          task_type: 'sparse_embedding',
          inference_id: _field_maps_configuration.ASSISTANT_ELSER_INFERENCE_ID,
          inference_config: {
            service: 'elasticsearch',
            service_settings: {
              adaptive_allocations: {
                enabled: true,
                min_number_of_allocations: 0,
                max_number_of_allocations: 8
              },
              task_settings: {}
            },
            task_settings: {}
          }
        });

        // await for the model to be deployed
        await this.isInferenceEndpointExists();
      } catch (error) {
        this.options.logger.error(`Error creating inference endpoint for ELSER model '${elserId}':\n${error}`);
        throw new Error(`Error creating inference endpoint for ELSER model '${elserId}':\n${error}`);
      }
    });
    /**
     * Downloads and deploys recommended ELSER (if not already), then loads ES|QL docs
     *
     * NOTE: Before automatically installing ELSER in the background, we should perform deployment resource checks
     * Only necessary for ESS, as Serverless can always auto-install if `productTier === complete`
     * See ml-team issue for providing 'dry run' flag to perform these checks: https://github.com/elastic/ml-team/issues/1208
     *
     * @param options
     * @param options.soClient SavedObjectsClientContract for installing ELSER so that ML SO's are in sync
     *
     * @returns Promise<void>
     */
    (0, _defineProperty2.default)(this, "setupKnowledgeBase", async ({
      soClient,
      v2KnowledgeBaseEnabled = true,
      ignoreSecurityLabs = false
    }) => {
      if (this.options.getIsKBSetupInProgress()) {
        this.options.logger.debug('Knowledge Base setup already in progress');
        return;
      }
      this.options.logger.debug('Starting Knowledge Base setup...');
      this.options.setIsKBSetupInProgress(true);
      const elserId = await this.options.getElserId();
      if (v2KnowledgeBaseEnabled) {
        // Delete legacy ESQL knowledge base docs if they exist, and silence the error if they do not
        try {
          const esClient = await this.options.elasticsearchClientPromise;
          const legacyESQL = await esClient.deleteByQuery({
            index: this.indexTemplateAndPattern.alias,
            query: {
              bool: {
                must: [{
                  terms: {
                    'metadata.kbResource': ['esql', 'unknown']
                  }
                }]
              }
            }
          });
          if ((legacyESQL === null || legacyESQL === void 0 ? void 0 : legacyESQL.total) != null && (legacyESQL === null || legacyESQL === void 0 ? void 0 : legacyESQL.total) > 0) {
            this.options.logger.info(`Removed ${legacyESQL === null || legacyESQL === void 0 ? void 0 : legacyESQL.total} ESQL knowledge base docs from knowledge base data stream: ${this.indexTemplateAndPattern.alias}.`);
          }
          // Delete any existing Security Labs content
          const securityLabsDocs = await esClient.deleteByQuery({
            index: this.indexTemplateAndPattern.alias,
            query: {
              bool: {
                must: [{
                  terms: {
                    kb_resource: [_constants.SECURITY_LABS_RESOURCE]
                  }
                }]
              }
            }
          });
          if (securityLabsDocs !== null && securityLabsDocs !== void 0 && securityLabsDocs.total) {
            this.options.logger.info(`Removed ${securityLabsDocs === null || securityLabsDocs === void 0 ? void 0 : securityLabsDocs.total} Security Labs knowledge base docs from knowledge base data stream: ${this.indexTemplateAndPattern.alias}.`);
          }
        } catch (e) {
          this.options.logger.info('No legacy ESQL or Security Labs knowledge base docs to delete');
        }
      }
      try {
        const isInstalled = await this.isModelInstalled();
        if (!isInstalled) {
          await this.installModel({
            soClient
          });
          await (0, _pRetry.default)(async () => (await this.isModelInstalled()) ? Promise.resolve() : Promise.reject(new Error('Model not installed')), {
            minTimeout: 10000,
            maxTimeout: 10000,
            retries: 10
          });
          this.options.logger.debug(`ELSER model '${elserId}' successfully installed!`);
        } else {
          this.options.logger.debug(`ELSER model '${elserId}' is already installed`);
        }
        if (!this.isV2KnowledgeBaseEnabled) {
          const isDeployed = await this.isModelDeployed();
          if (!isDeployed) {
            await this.deployModel();
            await (0, _pRetry.default)(async () => (await this.isModelDeployed()) ? Promise.resolve() : Promise.reject(new Error('Model not deployed')), {
              minTimeout: 2000,
              retries: 10
            });
            this.options.logger.debug(`ELSER model '${elserId}' successfully deployed!`);
          } else {
            this.options.logger.debug(`ELSER model '${elserId}' is already deployed`);
          }
        } else {
          const inferenceExists = await this.isInferenceEndpointExists();
          if (!inferenceExists) {
            await this.createInferenceEndpoint();
            this.options.logger.debug(`Inference endpoint for ELSER model '${elserId}' successfully deployed!`);
          } else {
            this.options.logger.debug(`Inference endpoint for ELSER model '${elserId}' is already deployed`);
          }
        }
        this.options.logger.debug(`Checking if Knowledge Base docs have been loaded...`);
        if (v2KnowledgeBaseEnabled && !ignoreSecurityLabs) {
          const labsDocsLoaded = await this.isSecurityLabsDocsLoaded();
          if (!labsDocsLoaded) {
            this.options.logger.debug(`Loading Security Labs KB docs...`);
            await (0, _security_labs_loader.loadSecurityLabs)(this, this.options.logger);
          } else {
            this.options.logger.debug(`Security Labs Knowledge Base docs already loaded!`);
          }
        }
      } catch (e) {
        this.options.setIsKBSetupInProgress(false);
        this.options.logger.error(`Error setting up Knowledge Base: ${e.message}`);
        throw new Error(`Error setting up Knowledge Base: ${e.message}`);
      } finally {
        this.options.setIsKBSetupInProgress(false);
      }
    });
    /**
     * Adds LangChain Documents to the knowledge base
     *
     * @param {Array<Document<Metadata>>} documents - LangChain Documents to add to the knowledge base
     * @param global whether these entries should be added globally, i.e. empty users[]
     */
    (0, _defineProperty2.default)(this, "addKnowledgeBaseDocuments", async ({
      documents,
      global = false
    }) => {
      var _created$data$hits$hi;
      const writer = await this.getWriter();
      const changedAt = new Date().toISOString();
      const authenticatedUser = this.options.currentUser;
      if (authenticatedUser == null) {
        throw new Error('Authenticated user not found! Ensure kbDataClient was initialized from a request.');
      }
      if (global && !this.options.manageGlobalKnowledgeBaseAIAssistant) {
        throw new Error('User lacks privileges to create global knowledge base entries');
      }
      const {
        errors,
        docs_created: docsCreated
      } = await writer.bulk({
        documentsToCreate: documents.map(doc => {
          // v1 schema has metadata nested in a `metadata` object
          if (this.options.v2KnowledgeBaseEnabled) {
            var _doc$metadata$kbResou, _doc$metadata$require, _doc$metadata$source;
            return (0, _create_knowledge_base_entry.transformToCreateSchema)({
              createdAt: changedAt,
              spaceId: this.spaceId,
              user: authenticatedUser,
              entry: {
                type: _elasticAssistantCommon.DocumentEntryType.value,
                name: 'unknown',
                text: doc.pageContent,
                kbResource: (_doc$metadata$kbResou = doc.metadata.kbResource) !== null && _doc$metadata$kbResou !== void 0 ? _doc$metadata$kbResou : 'unknown',
                required: (_doc$metadata$require = doc.metadata.required) !== null && _doc$metadata$require !== void 0 ? _doc$metadata$require : false,
                source: (_doc$metadata$source = doc.metadata.source) !== null && _doc$metadata$source !== void 0 ? _doc$metadata$source : 'unknown'
              },
              global
            });
          } else {
            var _doc$metadata$kbResou2, _doc$metadata$require2, _doc$metadata$source2;
            return (0, _create_knowledge_base_entry.transformToLegacyCreateSchema)({
              createdAt: changedAt,
              spaceId: this.spaceId,
              user: authenticatedUser,
              entry: {
                type: _elasticAssistantCommon.DocumentEntryType.value,
                name: 'unknown',
                text: doc.pageContent,
                metadata: {
                  kbResource: (_doc$metadata$kbResou2 = doc.metadata.kbResource) !== null && _doc$metadata$kbResou2 !== void 0 ? _doc$metadata$kbResou2 : 'unknown',
                  required: (_doc$metadata$require2 = doc.metadata.required) !== null && _doc$metadata$require2 !== void 0 ? _doc$metadata$require2 : false,
                  source: (_doc$metadata$source2 = doc.metadata.source) !== null && _doc$metadata$source2 !== void 0 ? _doc$metadata$source2 : 'unknown'
                }
              },
              global
            });
          }
        }),
        authenticatedUser
      });
      const created = docsCreated.length > 0 ? await this.findDocuments({
        page: 1,
        perPage: 10000,
        filter: docsCreated.map(c => `_id:${c}`).join(' OR ')
      }) : undefined;
      // Intentionally no telemetry here - this path only used to install security docs
      // Plans to make this function private in a different PR so no user entry ever is created in this path
      this.options.logger.debug(`created: ${(_created$data$hits$hi = created === null || created === void 0 ? void 0 : created.data.hits.hits.length) !== null && _created$data$hits$hi !== void 0 ? _created$data$hits$hi : '0'}`);
      this.options.logger.debug(() => `errors: ${JSON.stringify(errors, null, 2)}`);
      return created !== null && created !== void 0 && created.data ? (0, _transforms.transformESSearchToKnowledgeBaseEntry)(created === null || created === void 0 ? void 0 : created.data) : [];
    });
    /**
     * Returns if ES|QL KB docs have been loaded
     */
    (0, _defineProperty2.default)(this, "isESQLDocsLoaded", async () => {
      const esqlDocs = await this.getKnowledgeBaseDocumentEntries({
        query: _constants.ESQL_DOCS_LOADED_QUERY,
        // kbResource, // Note: `8.15` installs have kbResource as `unknown`, so don't filter yet
        required: true
      });
      return esqlDocs.length > 0;
    });
    /**
     * Returns if user's KB docs exists
     */
    (0, _defineProperty2.default)(this, "isUserDataExists", async () => {
      const user = this.options.currentUser;
      if (user == null) {
        throw new Error('Authenticated user not found! Ensure kbDataClient was initialized from a request.');
      }
      const esClient = await this.options.elasticsearchClientPromise;
      const modelId = await this.options.getElserId();
      try {
        var _result$hits;
        const vectorSearchQuery = (0, _helpers.getKBVectorSearchQuery)({
          kbResource: _constants.USER_RESOURCE,
          required: false,
          user,
          modelId,
          v2KnowledgeBaseEnabled: this.options.v2KnowledgeBaseEnabled
        });
        const result = await esClient.search({
          index: this.indexTemplateAndPattern.alias,
          size: 0,
          query: vectorSearchQuery,
          track_total_hits: true
        });
        return !!((_result$hits = result.hits) === null || _result$hits === void 0 ? void 0 : _result$hits.total).value;
      } catch (e) {
        this.options.logger.debug(`Error checking if user's KB docs exist: ${e.message}`);
        return false;
      }
    });
    /**
     * Returns if allSecurity Labs KB docs have been loaded
     */
    (0, _defineProperty2.default)(this, "isSecurityLabsDocsLoaded", async () => {
      const user = this.options.currentUser;
      if (user == null) {
        throw new Error('Authenticated user not found! Ensure kbDataClient was initialized from a request.');
      }
      const expectedDocsCount = await (0, _security_labs_loader.getSecurityLabsDocsCount)({
        logger: this.options.logger
      });
      const esClient = await this.options.elasticsearchClientPromise;
      const modelId = await this.options.getElserId();
      try {
        var _result$hits2;
        const vectorSearchQuery = (0, _helpers.getKBVectorSearchQuery)({
          kbResource: _constants.SECURITY_LABS_RESOURCE,
          required: false,
          user,
          modelId,
          v2KnowledgeBaseEnabled: this.options.v2KnowledgeBaseEnabled
        });
        const result = await esClient.search({
          index: this.indexTemplateAndPattern.alias,
          size: 0,
          query: vectorSearchQuery,
          track_total_hits: true
        });
        const existingDocs = ((_result$hits2 = result.hits) === null || _result$hits2 === void 0 ? void 0 : _result$hits2.total).value;
        if (existingDocs !== expectedDocsCount) {
          this.options.logger.debug(`Security Labs docs are not loaded, existing docs: ${existingDocs}, expected docs: ${expectedDocsCount}`);
        }
        return existingDocs === expectedDocsCount;
      } catch (e) {
        this.options.logger.info(`Error checking if Security Labs docs are loaded: ${e.message}`);
        return false;
      }
    });
    /**
     * Performs similarity search to retrieve LangChain Documents from the knowledge base
     */
    (0, _defineProperty2.default)(this, "getKnowledgeBaseDocumentEntries", async ({
      filter,
      kbResource,
      query,
      required
    }) => {
      const user = this.options.currentUser;
      if (user == null) {
        throw new Error('Authenticated user not found! Ensure kbDataClient was initialized from a request.');
      }
      const esClient = await this.options.elasticsearchClientPromise;
      const modelId = await this.options.getElserId();
      const vectorSearchQuery = (0, _helpers.getKBVectorSearchQuery)({
        filter,
        kbResource,
        query,
        required,
        user,
        modelId,
        v2KnowledgeBaseEnabled: this.options.v2KnowledgeBaseEnabled
      });
      try {
        const result = await esClient.search({
          index: this.indexTemplateAndPattern.alias,
          size: 10,
          query: vectorSearchQuery
        });
        const results = result.hits.hits.map(hit => {
          var _hit$_source, _hit$_source2, _hit$_source3, _hit$_source$metadata, _hit$_source4, _hit$_source$text, _hit$_source5;
          const metadata = this.options.v2KnowledgeBaseEnabled ? {
            source: hit === null || hit === void 0 ? void 0 : (_hit$_source = hit._source) === null || _hit$_source === void 0 ? void 0 : _hit$_source.source,
            required: hit === null || hit === void 0 ? void 0 : (_hit$_source2 = hit._source) === null || _hit$_source2 === void 0 ? void 0 : _hit$_source2.required,
            kbResource: hit === null || hit === void 0 ? void 0 : (_hit$_source3 = hit._source) === null || _hit$_source3 === void 0 ? void 0 : _hit$_source3.kb_resource
          } : // @ts-ignore v1 schema has metadata nested in a `metadata` object and kbResource vs kb_resource
          (_hit$_source$metadata = hit === null || hit === void 0 ? void 0 : (_hit$_source4 = hit._source) === null || _hit$_source4 === void 0 ? void 0 : _hit$_source4.metadata) !== null && _hit$_source$metadata !== void 0 ? _hit$_source$metadata : {};
          return new _document.Document({
            pageContent: (_hit$_source$text = hit === null || hit === void 0 ? void 0 : (_hit$_source5 = hit._source) === null || _hit$_source5 === void 0 ? void 0 : _hit$_source5.text) !== null && _hit$_source$text !== void 0 ? _hit$_source$text : '',
            metadata
          });
        });
        this.options.logger.debug(() => `getKnowledgeBaseDocuments() - Similarity Search Query:\n ${JSON.stringify(vectorSearchQuery)}`);
        this.options.logger.debug(() => `getKnowledgeBaseDocuments() - Similarity Search returned [${JSON.stringify(results.length)}] results`);
        return results;
      } catch (e) {
        this.options.logger.error(`Error performing KB Similarity Search: ${e.message}`);
        return [];
      }
    });
    /**
     * Returns all global and current user's private `required` document entries.
     */
    (0, _defineProperty2.default)(this, "getRequiredKnowledgeBaseDocumentEntries", async () => {
      const user = this.options.currentUser;
      if (user == null) {
        throw new Error('Authenticated user not found! Ensure kbDataClient was initialized from a request.');
      }
      try {
        const userFilter = (0, _utils.getKBUserFilter)(user);
        const results = await this.findDocuments({
          // Note: This is a magic number to set some upward bound as to not blow the context with too
          // many historical KB entries. Ideally we'd query for all and token trim.
          perPage: 100,
          page: 1,
          sortField: 'created_at',
          sortOrder: 'asc',
          filter: `${userFilter} AND type:document AND kb_resource:user AND required:true`
        });
        this.options.logger.debug(`kbDataClient.getRequiredKnowledgeBaseDocumentEntries() - results:\n${JSON.stringify(results)}`);
        if (results) {
          return (0, _transforms.transformESSearchToKnowledgeBaseEntry)(results.data);
        }
      } catch (e) {
        this.options.logger.error(`kbDataClient.getRequiredKnowledgeBaseDocumentEntries() - Failed to fetch DocumentEntries`);
        return [];
      }
      return [];
    });
    /**
     * Creates a new Knowledge Base Entry.
     *
     * @param knowledgeBaseEntry
     * @param global
     */
    (0, _defineProperty2.default)(this, "createKnowledgeBaseEntry", async ({
      knowledgeBaseEntry,
      telemetry,
      global = false
    }) => {
      const authenticatedUser = this.options.currentUser;
      if (authenticatedUser == null) {
        throw new Error('Authenticated user not found! Ensure kbDataClient was initialized from a request.');
      }
      if (global && !this.options.manageGlobalKnowledgeBaseAIAssistant) {
        throw new Error('User lacks privileges to create global knowledge base entries');
      }
      this.options.logger.debug(() => `Creating Knowledge Base Entry:\n ${JSON.stringify(knowledgeBaseEntry, null, 2)}`);
      this.options.logger.debug(`kbIndex: ${this.indexTemplateAndPattern.alias}`);
      const esClient = await this.options.elasticsearchClientPromise;
      return (0, _create_knowledge_base_entry.createKnowledgeBaseEntry)({
        esClient,
        knowledgeBaseIndex: this.indexTemplateAndPattern.alias,
        logger: this.options.logger,
        spaceId: this.spaceId,
        user: authenticatedUser,
        knowledgeBaseEntry,
        global,
        telemetry,
        isV2: this.options.v2KnowledgeBaseEnabled
      });
    });
    /**
     * Returns AssistantTools for any 'relevant' KB IndexEntries that exist in the knowledge base.
     *
     * Note: Accepts esClient so retrieval can be scoped to the current user as esClient on kbDataClient
     * is scoped to system user.
     */
    (0, _defineProperty2.default)(this, "getAssistantTools", async ({
      assistantToolParams,
      esClient
    }) => {
      const user = this.options.currentUser;
      if (user == null) {
        throw new Error('Authenticated user not found! Ensure kbDataClient was initialized from a request.');
      }
      try {
        const elserId = this.isV2KnowledgeBaseEnabled ? _field_maps_configuration.ASSISTANT_ELSER_INFERENCE_ID : await this.options.getElserId();
        const userFilter = (0, _utils.getKBUserFilter)(user);
        const results = await this.findDocuments({
          // Note: This is a magic number to set some upward bound as to not blow the context with too
          // many registered tools. As discussed in review, this will initially be mitigated by caps on
          // the IndexEntries field lengths, context trimming at the graph layer (before compilation),
          // and eventually some sort of tool discovery sub-graph or generic retriever to scale tool usage.
          perPage: 23,
          page: 1,
          sortField: 'created_at',
          sortOrder: 'asc',
          filter: `${userFilter} AND type:index`
        });
        this.options.logger.debug(`kbDataClient.getAssistantTools() - results:\n${JSON.stringify(results, null, 2)}`);
        if (results) {
          const entries = (0, _transforms.transformESSearchToKnowledgeBaseEntry)(results.data);
          const indexPatternFetcher = new _server.IndexPatternsFetcher(esClient);
          const existingIndices = await indexPatternFetcher.getExistingIndices((0, _lodash.map)(entries, 'index'));
          return entries
          // Filter out any IndexEntries that don't have an existing index
          .filter(entry => existingIndices.includes(entry.index)).map(indexEntry => {
            return (0, _helpers.getStructuredToolForIndexEntry)({
              indexEntry,
              esClient,
              logger: this.options.logger,
              elserId
            });
          });
        }
      } catch (e) {
        this.options.logger.error(`kbDataClient.getAssistantTools() - Failed to fetch IndexEntries`);
        return [];
      }
      return [];
    });
    this.options = options;
  }
  get isSetupInProgress() {
    return this.options.getIsKBSetupInProgress();
  }
  get isV2KnowledgeBaseEnabled() {
    return this.options.v2KnowledgeBaseEnabled;
  }
}
exports.AIAssistantKnowledgeBaseDataClient = AIAssistantKnowledgeBaseDataClient;