"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isModelAlreadyExistsError = exports.getStructuredToolForIndexEntry = exports.getKBVectorSearchQuery = void 0;
var _zod = require("@kbn/zod");
var _lodash = require("lodash");
var _tools = require("@langchain/core/tools");
var _elasticsearch = require("@elastic/elasticsearch");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isModelAlreadyExistsError = error => {
  return error instanceof _elasticsearch.errors.ResponseError && (error.body.error.type === 'resource_not_found_exception' || error.body.error.type === 'status_exception');
};

/**
 * Returns an Elasticsearch query DSL that performs a vector search against the Knowledge Base for the given query/user/filter. Searches only for DocumentEntries, not IndexEntries as they have no content.
 *
 * @param filter - Optional filter to apply to the search
 * @param kbResource - Specific resource tag to filter for, e.g. 'esql' or 'user'
 * @param modelId - ID of the model to search with, e.g. `.elser_model_2`
 * @param query - The search query provided by the user
 * @param required - Whether to only include required entries
 * @param user - The authenticated user
 * @param v2KnowledgeBaseEnabled whether the new v2 KB is enabled
 * @returns
 */
exports.isModelAlreadyExistsError = isModelAlreadyExistsError;
const getKBVectorSearchQuery = ({
  filter,
  kbResource,
  modelId,
  query,
  required,
  user,
  v2KnowledgeBaseEnabled = false
}) => {
  const kbResourceKey = v2KnowledgeBaseEnabled ? 'kb_resource' : 'metadata.kbResource';
  const requiredKey = v2KnowledgeBaseEnabled ? 'required' : 'metadata.required';
  const resourceFilter = kbResource ? [{
    term: {
      [kbResourceKey]: kbResource
    }
  }] : [];
  const requiredFilter = required ? [{
    term: {
      [requiredKey]: required
    }
  }] : [];
  const userFilter = {
    should: [{
      nested: {
        path: 'users',
        query: {
          bool: {
            minimum_should_match: 1,
            should: [{
              match: user.profile_uid ? {
                'users.id': user.profile_uid
              } : {
                'users.name': user.username
              }
            }]
          }
        }
      }
    }, {
      bool: {
        must_not: [{
          nested: {
            path: 'users',
            query: {
              bool: {
                filter: {
                  exists: {
                    field: 'users'
                  }
                }
              }
            }
          }
        }]
      }
    }]
  };
  let semanticTextFilter = [];
  if (v2KnowledgeBaseEnabled && query) {
    semanticTextFilter = [{
      semantic: {
        field: 'semantic_text',
        query
      }
    }];
  } else if (!v2KnowledgeBaseEnabled) {
    semanticTextFilter = [{
      text_expansion: {
        'vector.tokens': {
          model_id: modelId,
          model_text: query
        }
      }
    }];
  }
  return {
    bool: {
      must: [...semanticTextFilter, ...requiredFilter, ...resourceFilter],
      ...userFilter,
      filter,
      minimum_should_match: 1
    }
  };
};

/**
 * Returns a StructuredTool for a given IndexEntry
 */
exports.getKBVectorSearchQuery = getKBVectorSearchQuery;
const getStructuredToolForIndexEntry = ({
  indexEntry,
  esClient,
  logger,
  elserId
}) => {
  var _indexEntry$inputSche;
  const inputSchema = (_indexEntry$inputSche = indexEntry.inputSchema) === null || _indexEntry$inputSche === void 0 ? void 0 : _indexEntry$inputSche.reduce((prev, input) => {
    const fieldType = input.fieldType === 'string' ? _zod.z.string() : input.fieldType === 'number' ? _zod.z.number() : input.fieldType === 'boolean' ? _zod.z.boolean() : _zod.z.any();
    return {
      ...prev,
      [input.fieldName]: fieldType.describe(input.description)
    };
  }, {});
  return new _tools.DynamicStructuredTool({
    name: indexEntry.name.replace(/[^a-zA-Z0-9-]/g, ''),
    // // Tool names expects a string that matches the pattern '^[a-zA-Z0-9-]+$'
    description: indexEntry.description,
    schema: _zod.z.object({
      query: _zod.z.string().describe(indexEntry.queryDescription),
      ...inputSchema
    }),
    func: async (input, _, cbManager) => {
      var _indexEntry$inputSche2, _indexEntry$inputSche3;
      logger.debug(() => `Generated ${indexEntry.name} Tool:input\n ${JSON.stringify(input, null, 2)}`);

      // Generate filters for inputSchema fields
      const filter = (_indexEntry$inputSche2 = (_indexEntry$inputSche3 = indexEntry.inputSchema) === null || _indexEntry$inputSche3 === void 0 ? void 0 : _indexEntry$inputSche3.reduce((prev, i) => {
        return [...prev,
        // @ts-expect-error Possible to override types with dynamic input schema?
        {
          term: {
            [`${i.fieldName}`]: input === null || input === void 0 ? void 0 : input[i.fieldName]
          }
        }];
      }, [])) !== null && _indexEntry$inputSche2 !== void 0 ? _indexEntry$inputSche2 : [];
      const params = {
        index: indexEntry.index,
        size: 10,
        retriever: {
          standard: {
            query: {
              nested: {
                path: `${indexEntry.field}.inference.chunks`,
                query: {
                  sparse_vector: {
                    inference_id: elserId,
                    field: `${indexEntry.field}.inference.chunks.embeddings`,
                    query: input.query
                  }
                },
                inner_hits: {
                  size: 2,
                  name: `${indexEntry.name}.${indexEntry.field}`,
                  _source: [`${indexEntry.field}.inference.chunks.text`]
                }
              }
            },
            filter
          }
        }
      };
      try {
        const result = await esClient.search(params);
        const kbDocs = result.hits.hits.map(hit => {
          var _hit$inner_hits;
          if (indexEntry.outputFields && indexEntry.outputFields.length > 0) {
            return indexEntry.outputFields.reduce((prev, field) => {
              // @ts-expect-error
              return {
                ...prev,
                [field]: hit._source[field]
              };
            }, {});
          }

          // We want to send relevant inner hits (chunks) to the LLM as a context
          const innerHitPath = `${indexEntry.name}.${indexEntry.field}`;
          if ((_hit$inner_hits = hit.inner_hits) !== null && _hit$inner_hits !== void 0 && _hit$inner_hits[innerHitPath]) {
            return {
              text: hit.inner_hits[innerHitPath].hits.hits.map(innerHit => innerHit._source.text).join('\n --- \n')
            };
          }
          return {
            text: (0, _lodash.get)(hit._source, `${indexEntry.field}.inference.chunks[0].text`)
          };
        });
        logger.debug(() => `Similarity Search Params:\n ${JSON.stringify(params)}`);
        logger.debug(() => `Similarity Search Results:\n ${JSON.stringify(result)}`);
        logger.debug(() => `Similarity Text Extract Results:\n ${JSON.stringify(kbDocs)}`);
        return `###\nBelow are all relevant documents in JSON format:\n${JSON.stringify(kbDocs)}\n###`;
      } catch (e) {
        logger.error(`Error performing IndexEntry KB Similarity Search: ${e.message}`);
        return `I'm sorry, but I was unable to find any information in the knowledge base. Perhaps this error would be useful to deliver to the user. Be sure to print it below your response and in a codeblock so it is rendered nicely: ${e.message}`;
      }
    },
    tags: ['knowledge-base']
    // TODO: Remove after ZodAny is fixed https://github.com/langchain-ai/langchainjs/blob/main/langchain-core/src/tools.ts
  });
};
exports.getStructuredToolForIndexEntry = getStructuredToolForIndexEntry;