"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transformToUpdateSchema = exports.transformToLegacyCreateSchema = exports.transformToCreateSchema = exports.getUpdateScript = exports.createKnowledgeBaseEntry = void 0;
var _uuid = require("uuid");
var _event_based_telemetry = require("../../lib/telemetry/event_based_telemetry");
var _get_knowledge_base_entry = require("./get_knowledge_base_entry");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createKnowledgeBaseEntry = async ({
  esClient,
  knowledgeBaseIndex,
  spaceId,
  user,
  knowledgeBaseEntry,
  logger,
  global = false,
  isV2 = false,
  telemetry
}) => {
  var _body$required;
  const createdAt = new Date().toISOString();
  const body = isV2 ? transformToCreateSchema({
    createdAt,
    spaceId,
    user,
    entry: knowledgeBaseEntry,
    global
  }) : transformToLegacyCreateSchema({
    createdAt,
    spaceId,
    user,
    entry: knowledgeBaseEntry,
    global
  });
  const telemetryPayload = {
    entryType: body.type,
    required: (_body$required = body.required) !== null && _body$required !== void 0 ? _body$required : false,
    sharing: body.users.length ? 'private' : 'global',
    ...(body.type === 'document' ? {
      source: body.source
    } : {})
  };
  try {
    const response = await esClient.create({
      body,
      id: (0, _uuid.v4)(),
      index: knowledgeBaseIndex,
      refresh: 'wait_for'
    });
    const newKnowledgeBaseEntry = await (0, _get_knowledge_base_entry.getKnowledgeBaseEntry)({
      esClient,
      knowledgeBaseIndex,
      id: response._id,
      logger,
      user
    });
    telemetry.reportEvent(_event_based_telemetry.CREATE_KNOWLEDGE_BASE_ENTRY_SUCCESS_EVENT.eventType, telemetryPayload);
    return newKnowledgeBaseEntry;
  } catch (err) {
    var _err$message;
    logger.error(`Error creating Knowledge Base Entry: ${err} with kbResource: ${knowledgeBaseEntry.name}`);
    telemetry.reportEvent(_event_based_telemetry.CREATE_KNOWLEDGE_BASE_ENTRY_ERROR_EVENT.eventType, {
      ...telemetryPayload,
      errorMessage: (_err$message = err.message) !== null && _err$message !== void 0 ? _err$message : 'Unknown error'
    });
    throw err;
  }
};
exports.createKnowledgeBaseEntry = createKnowledgeBaseEntry;
const transformToUpdateSchema = ({
  user,
  updatedAt,
  entry,
  global = false
}) => {
  var _user$profile_uid, _entry$required;
  const base = {
    id: entry.id,
    updated_at: updatedAt,
    updated_by: (_user$profile_uid = user.profile_uid) !== null && _user$profile_uid !== void 0 ? _user$profile_uid : 'unknown',
    name: entry.name,
    type: entry.type,
    users: global ? [] : [{
      id: user.profile_uid,
      name: user.username
    }]
  };
  if (entry.type === 'index') {
    var _entry$inputSchema$ma, _entry$inputSchema;
    const {
      inputSchema,
      outputFields,
      queryDescription,
      ...restEntry
    } = entry;
    return {
      ...base,
      ...restEntry,
      query_description: queryDescription,
      input_schema: (_entry$inputSchema$ma = (_entry$inputSchema = entry.inputSchema) === null || _entry$inputSchema === void 0 ? void 0 : _entry$inputSchema.map(schema => ({
        field_name: schema.fieldName,
        field_type: schema.fieldType,
        description: schema.description
      }))) !== null && _entry$inputSchema$ma !== void 0 ? _entry$inputSchema$ma : undefined,
      output_fields: outputFields !== null && outputFields !== void 0 ? outputFields : undefined
    };
  }
  return {
    ...base,
    kb_resource: entry.kbResource,
    required: (_entry$required = entry.required) !== null && _entry$required !== void 0 ? _entry$required : false,
    source: entry.source,
    text: entry.text,
    vector: undefined
  };
};
exports.transformToUpdateSchema = transformToUpdateSchema;
const getUpdateScript = ({
  entry,
  isPatch
}) => {
  // Cannot use script for updating documents with semantic_text fields
  return {
    doc: {
      ...entry,
      semantic_text: entry.text
    }
  };
};
exports.getUpdateScript = getUpdateScript;
const transformToCreateSchema = ({
  createdAt,
  spaceId,
  user,
  entry,
  global = false
}) => {
  var _user$profile_uid2, _user$profile_uid3, _entry$required2;
  const base = {
    '@timestamp': createdAt,
    created_at: createdAt,
    created_by: (_user$profile_uid2 = user.profile_uid) !== null && _user$profile_uid2 !== void 0 ? _user$profile_uid2 : 'unknown',
    updated_at: createdAt,
    updated_by: (_user$profile_uid3 = user.profile_uid) !== null && _user$profile_uid3 !== void 0 ? _user$profile_uid3 : 'unknown',
    name: entry.name,
    namespace: spaceId,
    type: entry.type,
    users: global ? [] : [{
      id: user.profile_uid,
      name: user.username
    }]
  };
  if (entry.type === 'index') {
    var _entry$inputSchema$ma2, _entry$inputSchema2;
    const {
      inputSchema,
      outputFields,
      queryDescription,
      ...restEntry
    } = entry;
    return {
      ...base,
      ...restEntry,
      query_description: queryDescription,
      input_schema: (_entry$inputSchema$ma2 = (_entry$inputSchema2 = entry.inputSchema) === null || _entry$inputSchema2 === void 0 ? void 0 : _entry$inputSchema2.map(schema => ({
        field_name: schema.fieldName,
        field_type: schema.fieldType,
        description: schema.description
      }))) !== null && _entry$inputSchema$ma2 !== void 0 ? _entry$inputSchema$ma2 : undefined,
      output_fields: outputFields !== null && outputFields !== void 0 ? outputFields : undefined
    };
  }
  return {
    ...base,
    kb_resource: entry.kbResource,
    required: (_entry$required2 = entry.required) !== null && _entry$required2 !== void 0 ? _entry$required2 : false,
    source: entry.source,
    text: entry.text,
    semantic_text: entry.text
  };
};
exports.transformToCreateSchema = transformToCreateSchema;
const transformToLegacyCreateSchema = ({
  createdAt,
  spaceId,
  user,
  entry,
  global = false
}) => {
  var _user$profile_uid4, _user$profile_uid5;
  return {
    '@timestamp': createdAt,
    created_at: createdAt,
    created_by: (_user$profile_uid4 = user.profile_uid) !== null && _user$profile_uid4 !== void 0 ? _user$profile_uid4 : 'unknown',
    updated_at: createdAt,
    updated_by: (_user$profile_uid5 = user.profile_uid) !== null && _user$profile_uid5 !== void 0 ? _user$profile_uid5 : 'unknown',
    namespace: spaceId,
    users: global ? [] : [{
      id: user.profile_uid,
      name: user.username
    }],
    ...entry,
    vector: undefined
  };
};
exports.transformToLegacyCreateSchema = transformToLegacyCreateSchema;