"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateConversation = exports.transformToUpdateScheme = void 0;
var _get_conversation = require("./get_conversation");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const updateConversation = async ({
  esClient,
  logger,
  conversationIndex,
  conversationUpdateProps,
  isPatch,
  user
}) => {
  const updatedAt = new Date().toISOString();
  const params = transformToUpdateScheme(updatedAt, conversationUpdateProps);
  try {
    const response = await esClient.updateByQuery({
      conflicts: 'proceed',
      index: conversationIndex,
      query: {
        ids: {
          values: [params.id]
        }
      },
      refresh: true,
      script: (0, _helpers.getUpdateScript)({
        conversation: params,
        isPatch
      }).script
    });
    if (response.failures && response.failures.length > 0) {
      logger.warn(`Error updating conversation: ${response.failures.map(f => f.id)} by ID: ${params.id}`);
      return null;
    }
    const updatedConversation = await (0, _get_conversation.getConversation)({
      esClient,
      conversationIndex,
      id: params.id,
      logger,
      user
    });
    return updatedConversation;
  } catch (err) {
    logger.warn(`Error updating conversation: ${err} by ID: ${params.id}`);
    throw err;
  }
};
exports.updateConversation = updateConversation;
const transformToUpdateScheme = (updatedAt, {
  title,
  apiConfig,
  excludeFromLastConversationStorage,
  messages,
  replacements,
  id
}) => {
  return {
    id,
    updated_at: updatedAt,
    title,
    ...(apiConfig ? {
      api_config: {
        action_type_id: apiConfig === null || apiConfig === void 0 ? void 0 : apiConfig.actionTypeId,
        connector_id: apiConfig === null || apiConfig === void 0 ? void 0 : apiConfig.connectorId,
        default_system_prompt_id: apiConfig === null || apiConfig === void 0 ? void 0 : apiConfig.defaultSystemPromptId,
        model: apiConfig === null || apiConfig === void 0 ? void 0 : apiConfig.model,
        provider: apiConfig === null || apiConfig === void 0 ? void 0 : apiConfig.provider
      }
    } : {}),
    exclude_from_last_conversation_storage: excludeFromLastConversationStorage,
    replacements: replacements ? Object.keys(replacements).map(key => ({
      uuid: key,
      value: replacements[key]
    })) : undefined,
    messages: messages === null || messages === void 0 ? void 0 : messages.map(message => ({
      '@timestamp': message.timestamp,
      content: message.content,
      is_error: message.isError,
      reader: message.reader,
      role: message.role,
      ...(message.traceData ? {
        trace_data: {
          trace_id: message.traceData.traceId,
          transaction_id: message.traceData.transactionId
        }
      } : {})
    }))
  };
};
exports.transformToUpdateScheme = transformToUpdateScheme;