"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getQuery = exports.getIndexResultsRoute = void 0;
var _constants = require("../../../common/constants");
var _build_response = require("../../lib/build_response");
var _common = require("../../schemas/common");
var _result = require("../../schemas/result");
var _translations = require("./translations");
var _translations2 = require("../../translations");
var _get_authorized_index_names = require("../../helpers/get_authorized_index_names");
var _get_hits_total = require("../../helpers/get_hits_total");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getQuery = ({
  indexNames,
  size,
  from,
  outcome,
  startDate,
  endDate
}) => {
  const filters = [];
  if (outcome !== undefined) {
    const incompatibleFieldCountValueFilter = outcome === 'pass' ? {
      lt: 1
    } : {
      gt: 0
    };
    filters.push({
      range: {
        incompatibleFieldCount: incompatibleFieldCountValueFilter
      }
    });
  }
  if (startDate || endDate) {
    const startDateValueFilter = startDate && {
      gte: startDate
    };
    const endDateValueFilter = endDate && {
      lte: endDate
    };
    filters.push({
      range: {
        '@timestamp': {
          ...startDateValueFilter,
          ...endDateValueFilter
        }
      }
    });
  }
  return {
    query: {
      bool: {
        filter: [{
          terms: {
            indexName: indexNames
          }
        }, ...filters]
      }
    },
    sort: [{
      '@timestamp': 'desc'
    }],
    ...(size != null && {
      size
    }),
    ...(from != null && {
      from
    })
  };
};
exports.getQuery = getQuery;
const getIndexResultsRoute = (router, logger) => {
  router.versioned.get({
    path: _constants.GET_INDEX_RESULTS,
    access: 'internal',
    options: {
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: _constants.INTERNAL_API_VERSION,
    validate: {
      request: {
        params: (0, _common.buildRouteValidation)(_result.GetIndexResultsParams),
        query: (0, _common.buildRouteValidation)(_result.GetIndexResultsQuery)
      }
    }
  }, async (context, request, response) => {
    const services = await context.resolve(['core', 'dataQualityDashboard']);
    const resp = (0, _build_response.buildResponse)(response);
    let index;
    try {
      index = await services.dataQualityDashboard.getResultsIndexName();
    } catch (err) {
      logger.error(`[GET results] Error retrieving results index name: ${err.message}`);
      return resp.error({
        body: `${_translations.API_RESULTS_INDEX_NOT_AVAILABLE}: ${err.message}`,
        statusCode: 503
      });
    }
    try {
      var _hits$hits$map;
      const {
        client
      } = services.core.elasticsearch;
      const {
        pattern
      } = request.params;
      const authorizedIndexNames = await (0, _get_authorized_index_names.getAuthorizedIndexNames)(client, pattern);
      if (authorizedIndexNames.length === 0) {
        return response.ok({
          body: {
            data: [],
            total: 0
          }
        });
      }
      const {
        from,
        size,
        startDate,
        endDate,
        outcome
      } = request.query;
      // Get all results for all index names
      const query = {
        index,
        ...getQuery({
          indexNames: authorizedIndexNames,
          from,
          size,
          startDate,
          endDate,
          outcome
        })
      };
      const {
        hits
      } = await client.asInternalUser.search(query);
      const resultsWithUndefined = (_hits$hits$map = hits.hits.map(doc => doc._source)) !== null && _hits$hits$map !== void 0 ? _hits$hits$map : [];
      const resultsWithoutUndefined = resultsWithUndefined.filter(r => r);
      return response.ok({
        body: {
          data: resultsWithoutUndefined,
          total: (0, _get_hits_total.getHitsTotal)(hits)
        }
      });
    } catch (err) {
      var _err$message, _err$statusCode;
      logger.error(err.message);
      return resp.error({
        body: (_err$message = err.message) !== null && _err$message !== void 0 ? _err$message : _translations2.API_DEFAULT_ERROR_MESSAGE,
        statusCode: (_err$statusCode = err.statusCode) !== null && _err$statusCode !== void 0 ? _err$statusCode : 500
      });
    }
  });
};
exports.getIndexResultsRoute = getIndexResultsRoute;