"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getIndexStatsRoute = void 0;
var _i18n = require("@kbn/i18n");
var _lib = require("../lib");
var _build_response = require("../lib/build_response");
var _constants = require("../../common/constants");
var _common = require("../schemas/common");
var _get_index_stats = require("../schemas/get_index_stats");
var _translations = require("../translations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getIndexStatsRoute = (router, logger) => {
  router.versioned.get({
    path: _constants.GET_INDEX_STATS,
    access: 'internal',
    options: {
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: _constants.INTERNAL_API_VERSION,
    validate: {
      request: {
        params: (0, _common.buildRouteValidation)(_get_index_stats.GetIndexStatsParams),
        query: (0, _common.buildRouteValidation)(_get_index_stats.GetIndexStatsQuery)
      }
    }
  }, async (context, request, response) => {
    const resp = (0, _build_response.buildResponse)(response);
    try {
      const {
        client
      } = (await context.core).elasticsearch;
      const esClient = client.asCurrentUser;
      const decodedIndexName = decodeURIComponent(request.params.pattern);
      const {
        isILMAvailable,
        startDate,
        endDate
      } = request.query;
      if (isILMAvailable === true) {
        const stats = await (0, _lib.fetchStats)(client, decodedIndexName);
        const parsedIndices = (0, _lib.parseIndicesStats)(stats.indices);
        return response.ok({
          body: parsedIndices
        });
      }

      /**
       * If ILM is not available, we need to fetch the available indices with the given date range.
       * `fetchAvailableIndices` returns indices that have data in the given date range.
       */
      if (startDate && endDate) {
        const decodedStartDate = decodeURIComponent(startDate);
        const decodedEndDate = decodeURIComponent(endDate);
        const meteringStats = await (0, _lib.fetchMeteringStats)(client, decodedIndexName, request.headers.authorization);
        if (!meteringStats.indices) {
          logger.warn(`No metering stats indices found under pattern: ${decodedIndexName}`);
          return response.ok({
            body: {}
          });
        }
        const meteringStatsIndices = (0, _lib.parseMeteringStats)(meteringStats.indices);
        const availableIndices = await (0, _lib.fetchAvailableIndices)(esClient, {
          indexNameOrPattern: decodedIndexName,
          startDate: decodedStartDate,
          endDate: decodedEndDate
        });
        if (availableIndices.length === 0) {
          logger.warn(`No available indices found under pattern: ${decodedIndexName}, in the given date range: ${decodedStartDate} - ${decodedEndDate}`);
          return response.ok({
            body: {}
          });
        }
        const indices = (0, _lib.pickAvailableMeteringStats)(availableIndices, meteringStatsIndices);
        return response.ok({
          body: indices
        });
      } else {
        return resp.error({
          body: _i18n.i18n.translate('xpack.ecsDataQualityDashboard.getIndexStats.dateRangeRequiredErrorMessage', {
            defaultMessage: 'startDate and endDate are required'
          }),
          statusCode: 400
        });
      }
    } catch (err) {
      var _err$message, _err$statusCode;
      logger.error(JSON.stringify(err));
      return resp.error({
        body: (_err$message = err.message) !== null && _err$message !== void 0 ? _err$message : _translations.API_DEFAULT_ERROR_MESSAGE,
        statusCode: (_err$statusCode = err.statusCode) !== null && _err$statusCode !== void 0 ? _err$statusCode : 500
      });
    }
  });
};
exports.getIndexStatsRoute = getIndexStatsRoute;