"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FieldUsageEventName = exports.DiscoverEBTManager = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const FIELD_USAGE_EVENT_TYPE = 'discover_field_usage';
const FIELD_USAGE_EVENT_NAME = 'eventName';
const FIELD_USAGE_FIELD_NAME = 'fieldName';
const FIELD_USAGE_FILTER_OPERATION = 'filterOperation';
let FieldUsageEventName = exports.FieldUsageEventName = /*#__PURE__*/function (FieldUsageEventName) {
  FieldUsageEventName["dataTableSelection"] = "dataTableSelection";
  FieldUsageEventName["dataTableRemoval"] = "dataTableRemoval";
  FieldUsageEventName["filterAddition"] = "filterAddition";
  return FieldUsageEventName;
}({});
class DiscoverEBTManager {
  constructor() {
    (0, _defineProperty2.default)(this, "isCustomContextEnabled", false);
    (0, _defineProperty2.default)(this, "customContext$", void 0);
    (0, _defineProperty2.default)(this, "reportEvent", void 0);
  }

  // https://docs.elastic.dev/telemetry/collection/event-based-telemetry
  initialize({
    core,
    shouldInitializeCustomContext,
    shouldInitializeCustomEvents
  }) {
    if (shouldInitializeCustomContext) {
      // Register Discover specific context to be used in EBT
      const context$ = new _rxjs.BehaviorSubject({
        discoverProfiles: []
      });
      core.analytics.registerContextProvider({
        name: 'discover_context',
        context$,
        schema: {
          discoverProfiles: {
            type: 'array',
            items: {
              type: 'keyword',
              _meta: {
                description: 'List of active Discover context awareness profiles'
              }
            }
          }
          // If we decide to extend EBT context with more properties, we can do it here
        }
      });
      this.customContext$ = context$;
    }
    if (shouldInitializeCustomEvents) {
      // Register Discover events to be used with EBT
      core.analytics.registerEventType({
        eventType: FIELD_USAGE_EVENT_TYPE,
        schema: {
          [FIELD_USAGE_EVENT_NAME]: {
            type: 'keyword',
            _meta: {
              description: 'The name of the event that is tracked in the metrics i.e. dataTableSelection, dataTableRemoval'
            }
          },
          [FIELD_USAGE_FIELD_NAME]: {
            type: 'keyword',
            _meta: {
              description: "Field name if it's a part of ECS schema",
              optional: true
            }
          },
          [FIELD_USAGE_FILTER_OPERATION]: {
            type: 'keyword',
            _meta: {
              description: "Operation type when a filter is added i.e. '+', '-', '_exists_'",
              optional: true
            }
          }
        }
      });
      this.reportEvent = core.analytics.reportEvent;
    }
  }
  enableContext() {
    this.isCustomContextEnabled = true;
  }
  disableAndResetContext() {
    this.updateProfilesContextWith([]);
    this.isCustomContextEnabled = false;
  }
  updateProfilesContextWith(discoverProfiles) {
    if (this.isCustomContextEnabled && this.customContext$ && !(0, _lodash.isEqual)(this.customContext$.getValue().discoverProfiles, discoverProfiles)) {
      this.customContext$.next({
        discoverProfiles
      });
    }
  }
  getProfilesContext() {
    var _this$customContext$, _this$customContext$$;
    return (_this$customContext$ = this.customContext$) === null || _this$customContext$ === void 0 ? void 0 : (_this$customContext$$ = _this$customContext$.getValue()) === null || _this$customContext$$ === void 0 ? void 0 : _this$customContext$$.discoverProfiles;
  }
  async trackFieldUsageEvent({
    eventName,
    fieldName,
    filterOperation,
    fieldsMetadata
  }) {
    if (!this.reportEvent) {
      return;
    }
    const eventData = {
      [FIELD_USAGE_EVENT_NAME]: eventName
    };
    if (fieldsMetadata) {
      var _fields$fieldName;
      const client = await fieldsMetadata.getClient();
      const {
        fields
      } = await client.find({
        attributes: ['short'],
        fieldNames: [fieldName]
      });

      // excludes non ECS fields
      if ((_fields$fieldName = fields[fieldName]) !== null && _fields$fieldName !== void 0 && _fields$fieldName.short) {
        eventData[FIELD_USAGE_FIELD_NAME] = fieldName;
      }
    }
    if (filterOperation) {
      eventData[FIELD_USAGE_FILTER_OPERATION] = filterOperation;
    }
    this.reportEvent(FIELD_USAGE_EVENT_TYPE, eventData);
  }
  async trackDataTableSelection({
    fieldName,
    fieldsMetadata
  }) {
    await this.trackFieldUsageEvent({
      eventName: FieldUsageEventName.dataTableSelection,
      fieldName,
      fieldsMetadata
    });
  }
  async trackDataTableRemoval({
    fieldName,
    fieldsMetadata
  }) {
    await this.trackFieldUsageEvent({
      eventName: FieldUsageEventName.dataTableRemoval,
      fieldName,
      fieldsMetadata
    });
  }
  async trackFilterAddition({
    fieldName,
    fieldsMetadata,
    filterOperation
  }) {
    await this.trackFieldUsageEvent({
      eventName: FieldUsageEventName.filterAddition,
      fieldName,
      fieldsMetadata,
      filterOperation
    });
  }
}
exports.DiscoverEBTManager = DiscoverEBTManager;