"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.loadSavedSearch = void 0;
var _lodash = require("lodash");
var _esQuery = require("@kbn/es-query");
var _get_esql_data_view = require("./get_esql_data_view");
var _resolve_data_view = require("./resolve_data_view");
var _cleanup_url_state = require("./cleanup_url_state");
var _get_valid_filters = require("../../../../utils/get_valid_filters");
var _add_log = require("../../../../utils/add_log");
var _discover_app_state_container = require("../discover_app_state_container");
var _data_sources = require("../../../../../common/data_sources");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Loading persisted saved searches or existing ones and updating services accordingly
 * @param params
 * @param deps
 */
const loadSavedSearch = async (params, deps) => {
  var _globalStateContainer, _services$filterManag;
  (0, _add_log.addLog)('[discoverState] loadSavedSearch');
  const {
    savedSearchId,
    initialAppState
  } = params !== null && params !== void 0 ? params : {};
  const {
    appStateContainer,
    internalStateContainer,
    savedSearchContainer,
    globalStateContainer,
    services
  } = deps;
  const appStateExists = !appStateContainer.isEmptyURL();
  const appState = appStateExists ? appStateContainer.getState() : initialAppState;

  // Loading the saved search or creating a new one
  let nextSavedSearch;
  if (savedSearchId) {
    nextSavedSearch = await savedSearchContainer.load(savedSearchId);
  } else {
    const dataViewId = (0, _data_sources.isDataSourceType)(appState === null || appState === void 0 ? void 0 : appState.dataSource, _data_sources.DataSourceType.DataView) ? appState === null || appState === void 0 ? void 0 : appState.dataSource.dataViewId : undefined;
    nextSavedSearch = await savedSearchContainer.new(await getStateDataView(params, {
      dataViewId,
      query: appState === null || appState === void 0 ? void 0 : appState.query,
      services,
      internalStateContainer
    }));
  }

  // Cleaning up the previous state
  services.filterManager.setAppFilters([]);
  services.data.query.queryString.clearQuery();

  // Sync global filters (coming from URL) to filter manager.
  // It needs to be done manually here as `syncGlobalQueryStateWithUrl` is being called after this `loadSavedSearch` function.
  const globalFilters = globalStateContainer === null || globalStateContainer === void 0 ? void 0 : (_globalStateContainer = globalStateContainer.get()) === null || _globalStateContainer === void 0 ? void 0 : _globalStateContainer.filters;
  const shouldUpdateWithGlobalFilters = (globalFilters === null || globalFilters === void 0 ? void 0 : globalFilters.length) && !((_services$filterManag = services.filterManager.getGlobalFilters()) !== null && _services$filterManag !== void 0 && _services$filterManag.length);
  if (shouldUpdateWithGlobalFilters) {
    services.filterManager.setGlobalFilters(globalFilters);
  }

  // reset appState in case a saved search with id is loaded and
  // the url is empty so the saved search is loaded in a clean
  // state else it might be updated by the previous app state
  if (!appStateExists) {
    appStateContainer.set({});
  }

  // Update saved search by a given app state (in URL)
  if (appState) {
    if (savedSearchId && (0, _data_sources.isDataSourceType)(appState.dataSource, _data_sources.DataSourceType.DataView)) {
      var _nextSavedSearch$sear;
      // This is for the case appState is overwriting the loaded saved search data view
      const savedSearchDataViewId = (_nextSavedSearch$sear = nextSavedSearch.searchSource.getField('index')) === null || _nextSavedSearch$sear === void 0 ? void 0 : _nextSavedSearch$sear.id;
      const stateDataView = await getStateDataView(params, {
        dataViewId: appState.dataSource.dataViewId,
        query: appState.query,
        savedSearch: nextSavedSearch,
        services,
        internalStateContainer
      });
      const dataViewDifferentToAppState = stateDataView.id !== savedSearchDataViewId;
      if (!nextSavedSearch.isTextBasedQuery && stateDataView && (dataViewDifferentToAppState || !savedSearchDataViewId)) {
        nextSavedSearch.searchSource.setField('index', stateDataView);
      }
    }
    nextSavedSearch = savedSearchContainer.update({
      nextDataView: nextSavedSearch.searchSource.getField('index'),
      nextState: appState
    });
  }

  // Update app state container with the next state derived from the next saved search
  const nextAppState = (0, _discover_app_state_container.getInitialState)(undefined, nextSavedSearch, services);
  const mergedAppState = appState ? {
    ...nextAppState,
    ...(0, _cleanup_url_state.cleanupUrlState)({
      ...appState
    }, services.uiSettings)
  } : nextAppState;
  appStateContainer.resetToState(mergedAppState);

  // Update all other services and state containers by the next saved search
  updateBySavedSearch(nextSavedSearch, deps);
  if (!appState && shouldUpdateWithGlobalFilters) {
    nextSavedSearch = savedSearchContainer.updateWithFilterManagerFilters();
  }
  internalStateContainer.transitions.resetOnSavedSearchChange();
  return nextSavedSearch;
};

/**
 * Update services and state containers based on the given saved search
 * @param savedSearch
 * @param deps
 */
exports.loadSavedSearch = loadSavedSearch;
function updateBySavedSearch(savedSearch, deps) {
  const {
    dataStateContainer,
    internalStateContainer,
    services,
    setDataView
  } = deps;
  const savedSearchDataView = savedSearch.searchSource.getField('index');
  setDataView(savedSearchDataView);
  if (!savedSearchDataView.isPersisted()) {
    internalStateContainer.transitions.appendAdHocDataViews(savedSearchDataView);
  }

  // Finally notify dataStateContainer, data.query and filterManager about new derived state
  dataStateContainer.reset();
  // set data service filters
  const filters = savedSearch.searchSource.getField('filter');
  if (Array.isArray(filters) && filters.length) {
    // Saved search SO persists all filters as app filters
    services.data.query.filterManager.setAppFilters((0, _lodash.cloneDeep)(filters));
  }
  // some filters may not be valid for this context, so update
  // the filter manager with a modified list of valid filters
  const currentFilters = services.filterManager.getFilters();
  const validFilters = (0, _get_valid_filters.getValidFilters)(savedSearchDataView, currentFilters);
  if (!(0, _lodash.isEqual)(currentFilters, validFilters)) {
    services.filterManager.setFilters(validFilters);
  }

  // set data service query
  const query = savedSearch.searchSource.getField('query');
  if (query) {
    services.data.query.queryString.setQuery(query);
  }
}

// Get the data view to actually use. There are several conditions to consider which data view is used
// 1. If a data view is passed in, use that
// 2. If an appState with index set is passed in, use the data view from that
// 3. If a saved search is passed in, use the data view from that
// And provide a fallback data view if 2. or 3. don't exist, were deleted or invalid
const getStateDataView = async (params, {
  dataViewId,
  query,
  savedSearch,
  services,
  internalStateContainer
}) => {
  const {
    dataView,
    dataViewSpec
  } = params;
  const isEsqlQuery = (0, _esQuery.isOfAggregateQueryType)(query);
  if (dataView) {
    return dataView;
  }
  if (isEsqlQuery) {
    return await (0, _get_esql_data_view.getEsqlDataView)(query, dataView, services);
  }
  const result = await (0, _resolve_data_view.loadAndResolveDataView)({
    id: dataViewId,
    dataViewSpec,
    savedSearch,
    isEsqlMode: isEsqlQuery
  }, {
    services,
    internalStateContainer
  });
  return result.dataView;
};