"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.checkHitCount = void 0;
exports.sendCompleteMsg = sendCompleteMsg;
exports.sendErrorMsg = sendErrorMsg;
exports.sendErrorTo = void 0;
exports.sendLoadingMoreFinishedMsg = sendLoadingMoreFinishedMsg;
exports.sendLoadingMoreMsg = sendLoadingMoreMsg;
exports.sendLoadingMsg = sendLoadingMsg;
exports.sendNoResultsFoundMsg = sendNoResultsFoundMsg;
exports.sendPartialMsg = sendPartialMsg;
exports.sendResetMsg = sendResetMsg;
var _types = require("../../types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Sends COMPLETE message to the main$ observable with the information
 * that no documents have been found, allowing Discover to show a no
 * results message.
 */
function sendNoResultsFoundMsg(main$) {
  sendCompleteMsg(main$, false);
}

/**
 * Send COMPLETE message via main observable used when
 * 1.) first fetch resolved, and there are no documents
 * 2.) all fetches resolved, and there are documents
 */
function sendCompleteMsg(main$, foundDocuments = true) {
  if (main$.getValue().fetchStatus === _types.FetchStatus.COMPLETE) {
    return;
  }
  main$.next({
    fetchStatus: _types.FetchStatus.COMPLETE,
    foundDocuments,
    error: undefined
  });
}

/**
 * Send PARTIAL message via main observable when first result is returned
 */
function sendPartialMsg(main$) {
  if (main$.getValue().fetchStatus === _types.FetchStatus.LOADING) {
    main$.next({
      fetchStatus: _types.FetchStatus.PARTIAL
    });
  }
}

/**
 * Send LOADING message via main observable
 */
function sendLoadingMsg(data$, props) {
  if (data$.getValue().fetchStatus !== _types.FetchStatus.LOADING) {
    data$.next({
      ...props,
      fetchStatus: _types.FetchStatus.LOADING
    });
  }
}

/**
 * Send LOADING_MORE message via main observable
 */
function sendLoadingMoreMsg(documents$) {
  if (documents$.getValue().fetchStatus !== _types.FetchStatus.LOADING_MORE) {
    documents$.next({
      ...documents$.getValue(),
      fetchStatus: _types.FetchStatus.LOADING_MORE
    });
  }
}

/**
 * Finishing LOADING_MORE message
 */
function sendLoadingMoreFinishedMsg(documents$, {
  moreRecords,
  interceptedWarnings
}) {
  const currentValue = documents$.getValue();
  if (currentValue.fetchStatus === _types.FetchStatus.LOADING_MORE) {
    documents$.next({
      ...currentValue,
      fetchStatus: _types.FetchStatus.COMPLETE,
      result: moreRecords !== null && moreRecords !== void 0 && moreRecords.length ? [...(currentValue.result || []), ...moreRecords] : currentValue.result,
      interceptedWarnings
    });
  }
}

/**
 * Send ERROR message
 */
function sendErrorMsg(data$, error) {
  data$.next({
    fetchStatus: _types.FetchStatus.ERROR,
    error
  });
}

/**
 * Sends a RESET message to all data subjects
 * Needed when data view is switched or a new runtime field is added
 */
function sendResetMsg(data, initialFetchStatus) {
  data.main$.next({
    fetchStatus: initialFetchStatus,
    foundDocuments: undefined
  });
  data.documents$.next({
    fetchStatus: initialFetchStatus,
    result: []
  });
  data.totalHits$.next({
    fetchStatus: initialFetchStatus,
    result: undefined
  });
}

/**
 * Method to create an error handler that will forward the received error
 * to the specified subjects. It will ignore AbortErrors.
 */
const sendErrorTo = (...errorSubjects) => {
  return error => {
    if (error instanceof Error && error.name === 'AbortError') {
      return;
    }
    errorSubjects.forEach(subject => sendErrorMsg(subject, error));
  };
};

/**
 * This method checks the passed in hit count and will send a PARTIAL message to main$
 * if there are results, indicating that we have finished some of the requests that have been
 * sent. If there are no results we already COMPLETE main$ with no results found, so Discover
 * can show the "no results" screen. We know at that point, that the other query returning
 * will neither carry any data, since there are no documents.
 */
exports.sendErrorTo = sendErrorTo;
const checkHitCount = (main$, hitsCount) => {
  if (hitsCount > 0) {
    sendPartialMsg(main$);
  } else {
    sendNoResultsFoundMsg(main$);
  }
};
exports.checkHitCount = checkHitCount;