"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useEsqlMode = useEsqlMode;
var _lodash = require("lodash");
var _esQuery = require("@kbn/es-query");
var _esqlUtils = require("@kbn/esql-utils");
var _react = require("react");
var _rxjs = require("rxjs");
var _discover_state_provider = require("../state_management/discover_state_provider");
var _get_valid_view_mode = require("../utils/get_valid_view_mode");
var _types = require("../../types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const MAX_NUM_OF_COLUMNS = 50;

/**
 * Hook to take care of ES|QL state transformations when a new result is returned
 * If necessary this is setting displayed columns and selected data view
 */
function useEsqlMode({
  dataViews,
  stateContainer
}) {
  const savedSearch = (0, _discover_state_provider.useSavedSearchInitial)();
  const prev = (0, _react.useRef)({
    initialFetch: true,
    query: '',
    allColumns: [],
    defaultColumns: []
  });
  const cleanup = (0, _react.useCallback)(() => {
    if (!prev.current.query) {
      return;
    }

    // cleanup when it's not an ES|QL query
    prev.current = {
      initialFetch: true,
      query: '',
      allColumns: [],
      defaultColumns: []
    };
  }, []);
  (0, _react.useEffect)(() => {
    const subscription = stateContainer.dataState.data$.documents$.pipe((0, _rxjs.switchMap)(async next => {
      var _next$result;
      const {
        query: nextQuery
      } = next;
      if (!nextQuery || next.fetchStatus === _types.FetchStatus.ERROR) {
        return;
      }
      if (!(0, _esQuery.isOfAggregateQueryType)(nextQuery)) {
        // cleanup for a "regular" query
        cleanup();
        return;
      }
      if (next.fetchStatus !== _types.FetchStatus.PARTIAL) {
        return;
      }
      let nextAllColumns = prev.current.allColumns;
      let nextDefaultColumns = prev.current.defaultColumns;
      if ((_next$result = next.result) !== null && _next$result !== void 0 && _next$result.length) {
        nextAllColumns = Object.keys(next.result[0].raw);
        if ((0, _esqlUtils.hasTransformationalCommand)(nextQuery.esql)) {
          nextDefaultColumns = nextAllColumns.slice(0, MAX_NUM_OF_COLUMNS);
        } else {
          nextDefaultColumns = [];
        }
      }
      if (prev.current.initialFetch) {
        prev.current.initialFetch = false;
        prev.current.query = nextQuery.esql;
        prev.current.allColumns = nextAllColumns;
        prev.current.defaultColumns = nextDefaultColumns;
      }
      const indexPatternChanged = (0, _esqlUtils.getIndexPatternFromESQLQuery)(nextQuery.esql) !== (0, _esqlUtils.getIndexPatternFromESQLQuery)(prev.current.query);
      const allColumnsChanged = !(0, _lodash.isEqual)(nextAllColumns, prev.current.allColumns);
      const changeDefaultColumns = indexPatternChanged || !(0, _lodash.isEqual)(nextDefaultColumns, prev.current.defaultColumns);
      const {
        viewMode
      } = stateContainer.appState.getState();
      const changeViewMode = viewMode !== (0, _get_valid_view_mode.getValidViewMode)({
        viewMode,
        isEsqlMode: true
      });
      if (indexPatternChanged) {
        stateContainer.internalState.transitions.setResetDefaultProfileState({
          columns: true,
          rowHeight: true
        });
      } else if (allColumnsChanged) {
        stateContainer.internalState.transitions.setResetDefaultProfileState({
          columns: true,
          rowHeight: false
        });
      }
      prev.current.allColumns = nextAllColumns;
      if (indexPatternChanged || changeDefaultColumns || changeViewMode) {
        prev.current.query = nextQuery.esql;
        prev.current.defaultColumns = nextDefaultColumns;

        // just change URL state if necessary
        if (changeDefaultColumns || changeViewMode) {
          const nextState = {
            ...(changeDefaultColumns && {
              columns: nextDefaultColumns
            }),
            ...(changeViewMode && {
              viewMode: undefined
            })
          };
          await stateContainer.appState.replaceUrlState(nextState);
        }
      }
      stateContainer.dataState.data$.documents$.next({
        ...next,
        fetchStatus: _types.FetchStatus.COMPLETE
      });
    })).subscribe();
    return () => {
      // cleanup for e.g. when savedSearch is switched
      cleanup();
      subscription.unsubscribe();
    };
  }, [dataViews, stateContainer, savedSearch, cleanup]);
}