"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDataStreamDetails = getDataStreamDetails;
exports.getDataStreamSettings = getDataStreamSettings;
var _boom = require("@hapi/boom");
var _common = require("@kbn/metrics-data-access-plugin/common");
var _server = require("@kbn/observability-plugin/server");
var _constants = require("../../../../common/constants");
var _es_fields = require("../../../../common/es_fields");
var _utils = require("../../../utils");
var _services = require("../../../services");
var _get_data_streams = require("../get_data_streams");
var _get_data_streams_metering_stats = require("../get_data_streams_metering_stats");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getDataStreamSettings({
  esClient,
  dataStream
}) {
  var _dataStreamInfo$_meta, _dataStreamInfo$_meta2, _dataStreamInfo$indic;
  throwIfInvalidDataStreamParams(dataStream);
  const [createdOn, [dataStreamInfo], datasetUserPrivileges] = await Promise.all([getDataStreamCreatedOn(esClient, dataStream), _services.dataStreamService.getMatchingDataStreams(esClient, dataStream), _services.datasetQualityPrivileges.getDatasetPrivileges(esClient, dataStream)]);
  const integration = dataStreamInfo === null || dataStreamInfo === void 0 ? void 0 : (_dataStreamInfo$_meta = dataStreamInfo._meta) === null || _dataStreamInfo$_meta === void 0 ? void 0 : (_dataStreamInfo$_meta2 = _dataStreamInfo$_meta.package) === null || _dataStreamInfo$_meta2 === void 0 ? void 0 : _dataStreamInfo$_meta2.name;
  const lastBackingIndex = dataStreamInfo === null || dataStreamInfo === void 0 ? void 0 : (_dataStreamInfo$indic = dataStreamInfo.indices) === null || _dataStreamInfo$indic === void 0 ? void 0 : _dataStreamInfo$indic.slice(-1)[0];
  return {
    createdOn,
    integration,
    datasetUserPrivileges,
    lastBackingIndexName: lastBackingIndex === null || lastBackingIndex === void 0 ? void 0 : lastBackingIndex.index_name
  };
}
async function getDataStreamDetails({
  esClient,
  dataStream,
  start,
  end,
  isServerless
}) {
  throwIfInvalidDataStreamParams(dataStream);

  // Query datastreams as the current user as the Kibana internal user may not have all the required permissions
  const esClientAsCurrentUser = esClient.asCurrentUser;
  const esClientAsSecondaryAuthUser = esClient.asSecondaryAuthUser;
  const hasAccessToDataStream = (await _services.datasetQualityPrivileges.getHasIndexPrivileges(esClientAsCurrentUser, [dataStream], ['monitor']))[dataStream];
  const esDataStream = hasAccessToDataStream ? (await (0, _get_data_streams.getDataStreams)({
    esClient: esClientAsCurrentUser,
    datasetQuery: dataStream
  })).dataStreams[0] : undefined;
  try {
    const dataStreamSummaryStats = await getDataStreamSummaryStats(esClientAsCurrentUser, dataStream, start, end);
    const avgDocSizeInBytes = hasAccessToDataStream && dataStreamSummaryStats.docsCount > 0 ? isServerless ? await getMeteringAvgDocSizeInBytes(esClientAsSecondaryAuthUser, dataStream) : await getAvgDocSizeInBytes(esClientAsCurrentUser, dataStream) : 0;
    const sizeBytes = Math.ceil(avgDocSizeInBytes * dataStreamSummaryStats.docsCount);
    return {
      ...dataStreamSummaryStats,
      sizeBytes,
      lastActivity: esDataStream === null || esDataStream === void 0 ? void 0 : esDataStream.lastActivity,
      userPrivileges: {
        canMonitor: hasAccessToDataStream
      }
    };
  } catch (e) {
    // Respond with empty object if data stream does not exist
    if (e.statusCode === 404) {
      return {};
    }
    throw e;
  }
}
async function getDataStreamCreatedOn(esClient, dataStream) {
  const indexSettings = await _services.dataStreamService.getDataStreamIndexSettings(esClient, dataStream);
  const indexesList = Object.values(indexSettings);
  return indexesList.map(index => {
    var _index$settings, _index$settings$index;
    return Number((_index$settings = index.settings) === null || _index$settings === void 0 ? void 0 : (_index$settings$index = _index$settings.index) === null || _index$settings$index === void 0 ? void 0 : _index$settings$index.creation_date);
  }).sort((a, b) => a - b)[0];
}
const MAX_HOSTS = _constants.MAX_HOSTS_METRIC_VALUE + 1; // Adding 1 so that we can show e.g. '50+'

// Gather service.name terms
const serviceNamesAgg = {
  ['service.name']: {
    terms: {
      field: 'service.name',
      size: MAX_HOSTS
    }
  }
};

// Gather host terms like 'host', 'pod', 'container'
const hostsAgg = _common.inventoryModels.map(model => (0, _common.findInventoryFields)(model.id)).reduce((acc, fields) => ({
  ...acc,
  [fields.id]: {
    terms: {
      field: fields.id,
      size: MAX_HOSTS
    }
  }
}), {});
async function getDataStreamSummaryStats(esClient, dataStream, start, end) {
  var _response$aggregation, _response$aggregation2, _response$aggregation3, _response$aggregation4;
  const datasetQualityESClient = (0, _utils.createDatasetQualityESClient)(esClient);
  const response = await datasetQualityESClient.search({
    index: dataStream,
    query: (0, _server.rangeQuery)(start, end)[0],
    size: 0,
    aggs: {
      total_count: {
        value_count: {
          field: '_index'
        }
      },
      degraded_count: {
        filter: {
          exists: {
            field: _es_fields._IGNORED
          }
        }
      },
      ...serviceNamesAgg,
      ...hostsAgg
    }
  });
  const docsCount = Number((_response$aggregation = (_response$aggregation2 = response.aggregations) === null || _response$aggregation2 === void 0 ? void 0 : _response$aggregation2.total_count.value) !== null && _response$aggregation !== void 0 ? _response$aggregation : 0);
  const degradedDocsCount = Number((_response$aggregation3 = (_response$aggregation4 = response.aggregations) === null || _response$aggregation4 === void 0 ? void 0 : _response$aggregation4.degraded_count.doc_count) !== null && _response$aggregation3 !== void 0 ? _response$aggregation3 : 0);
  return {
    docsCount,
    degradedDocsCount,
    services: getTermsFromAgg(serviceNamesAgg, response.aggregations),
    hosts: getTermsFromAgg(hostsAgg, response.aggregations)
  };
}
async function getMeteringAvgDocSizeInBytes(esClient, index) {
  var _meteringStats$index$, _meteringStats$index$2;
  const meteringStats = await (0, _get_data_streams_metering_stats.getDataStreamsMeteringStats)({
    esClient,
    dataStreams: [index]
  });
  const docCount = (_meteringStats$index$ = meteringStats[index].totalDocs) !== null && _meteringStats$index$ !== void 0 ? _meteringStats$index$ : 0;
  const sizeInBytes = (_meteringStats$index$2 = meteringStats[index].sizeBytes) !== null && _meteringStats$index$2 !== void 0 ? _meteringStats$index$2 : 0;
  return docCount ? sizeInBytes / docCount : 0;
}
async function getAvgDocSizeInBytes(esClient, index) {
  var _indexStats$_all$tota, _indexStats$_all$tota2, _indexStats$_all$tota3, _indexStats$_all$tota4, _indexStats$_all$tota5, _indexStats$_all$tota6;
  const indexStats = await esClient.indices.stats({
    index
  });
  const docCount = (_indexStats$_all$tota = (_indexStats$_all$tota2 = indexStats._all.total) === null || _indexStats$_all$tota2 === void 0 ? void 0 : (_indexStats$_all$tota3 = _indexStats$_all$tota2.docs) === null || _indexStats$_all$tota3 === void 0 ? void 0 : _indexStats$_all$tota3.count) !== null && _indexStats$_all$tota !== void 0 ? _indexStats$_all$tota : 0;
  const sizeInBytes = (_indexStats$_all$tota4 = (_indexStats$_all$tota5 = indexStats._all.total) === null || _indexStats$_all$tota5 === void 0 ? void 0 : (_indexStats$_all$tota6 = _indexStats$_all$tota5.store) === null || _indexStats$_all$tota6 === void 0 ? void 0 : _indexStats$_all$tota6.size_in_bytes) !== null && _indexStats$_all$tota4 !== void 0 ? _indexStats$_all$tota4 : 0;
  return docCount ? sizeInBytes / docCount : 0;
}
function getTermsFromAgg(termAgg, aggregations) {
  return Object.entries(termAgg).reduce((acc, [key, _value]) => {
    var _aggregations$key;
    const values = (_aggregations$key = aggregations[key]) === null || _aggregations$key === void 0 ? void 0 : _aggregations$key.buckets.map(bucket => bucket.key);
    return {
      ...acc,
      [key]: values
    };
  }, {});
}
function throwIfInvalidDataStreamParams(dataStream) {
  if (!(dataStream !== null && dataStream !== void 0 && dataStream.trim())) {
    throw (0, _boom.badRequest)(`Data Stream name cannot be empty. Received value "${dataStream}"`);
  }
}